## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  eval = TRUE,
  echo = TRUE,
  comment = "#>"
)

## ----'installation', eval=FALSE-----------------------------------------------
# ## Install < remotes > package (if not already installed) ----
# if (!requireNamespace("remotes", quietly = TRUE)) {
#   install.packages("remotes")
# }
# 
# ## Install dev version of < forcis > from GitHub ----
# remotes::install_github("FRBCesab/forcis")

## ----setup--------------------------------------------------------------------
library(forcis)

## ----'download-db', eval=FALSE------------------------------------------------
# # Create a data/ folder in the current directory ----
# dir.create("data")
# 
# # Download latest version of the database ----
# download_forcis_db(
#   path = "data",
#   version = NULL
# )

## ----'load-data', echo=FALSE--------------------------------------------------
file_name <- system.file(
  file.path("extdata", "FORCIS_net_sample.csv"),
  package = "forcis"
)

net_data <- read.csv(file_name) |>
  tibble::as_tibble()

## ----'load-data-user', eval=FALSE---------------------------------------------
# # Import plankton nets data ----
# net_data <- read_plankton_nets_data(path = "data")

## ----'print-data'-------------------------------------------------------------
# Print data ----
net_data

## ----'select-taxo'------------------------------------------------------------
# Select taxonomy ----
net_data_vt <- net_data |>
  select_taxonomy(taxonomy = "VT")

net_data_vt

## ----'use-case-1-metrics'-----------------------------------------------------
# How many subsamples do we have? ----
nrow(net_data_vt)

# How many species have been sampled? ----
net_data_vt |>
  get_species_names() |>
  length()

## ----'use-case-1-time'--------------------------------------------------------
# What is the temporal extent? ----
plot_record_by_year(net_data_vt)

## ----'use-case-1-space'-------------------------------------------------------
# What is the spatial extent? ----
ggmap_data(net_data_vt)

## ----'use-case-2-species'-----------------------------------------------------
# Get all species names ----
species_list <- net_data_vt |>
  get_species_names()

# Search for species containing the word 'pachyderma' ----
species_list[grep("pachyderma", species_list)]

# Store the species name ----
sp_name <- "n_pachyderma_VT"

## ----'use-case-2-species-filter'----------------------------------------------
# Filter data by species ----
net_data_vt_pachyderma <- net_data_vt |>
  filter_by_species(species = sp_name)

net_data_vt_pachyderma

# Remove empty samples for N. pachyderma ----
net_data_vt_pachyderma <- net_data_vt_pachyderma |>
  dplyr::filter(n_pachyderma_VT > 0)

net_data_vt_pachyderma

## ----'use-case-2-temporal-filter'---------------------------------------------
# Filter data by years ----
net_data_vt_pachyderma_7000 <- net_data_vt_pachyderma |>
  filter_by_year(years = 1970:2000)

# Number of records ----
nrow(net_data_vt_pachyderma_7000)

## ----'use-case-2-spatial-filter'----------------------------------------------
# Get the list of ocean names ----
get_ocean_names()

# Filter data by ocean ----
net_data_vt_pachyderma_7000_med <- net_data_vt_pachyderma_7000 |>
  filter_by_ocean(ocean = "Mediterranean Sea")

# Number of records ----
nrow(net_data_vt_pachyderma_7000_med)

## ----'use-case-2-map'---------------------------------------------------------
# Plot N. pachyderma records on a World map ----
ggmap_data(net_data_vt_pachyderma_7000_med)

## ----'use-case-2-all', eval=FALSE---------------------------------------------
# # Final use case 2 code ----
# net_data_vt |>
#   filter_by_species(species = "n_pachyderma_VT") |>
#   dplyr::filter(n_pachyderma_VT > 0) |>
#   filter_by_year(years = 1970:2000) |>
#   filter_by_ocean(ocean = "Mediterranean Sea") |>
#   ggmap_data()

