## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(KSIC)

## -----------------------------------------------------------------------------
library(KSIC)
# Set the default KSIC revision to the 10th
# 기본 KSIC 차수를 10차로 설정
options(ksic.C = 10)

# Now, ksic() and other functions will use C = 10 by default
# 이제 ksic() 및 다른 함수들은 기본적으로 C = 10을 사용합니다.
head(ksic(digit = 1))

# Reset to the default (11th revision)
# 기본값(11차)으로 재설정
options(ksic.C = 11)

## -----------------------------------------------------------------------------
# Get 1-digit codes from the default 11th revision
head(ksic(digit = 1))

# Get 1-digit codes including English names
head(ksic(digit = 1, eng_nm = TRUE))

## -----------------------------------------------------------------------------
is_ksic(c("A", "01", "99999", "invalid_code"))

## -----------------------------------------------------------------------------
ksic_group(c("31311", "4631", "25", "A"), digit = 2, name = TRUE)

# Example with an invalid code
# 잘못된 코드가 포함된 예시
ksic_group(c("26222", "99999", "58221"), digit = 2, name = TRUE)

## -----------------------------------------------------------------------------
result_list <- ksic_sub(c("26","96","52636"), digit = 4)
print(result_list)

# Example with an invalid code
# 잘못된 코드가 포함된 예시
ksic_sub(c("26", "99999", "58"), digit = 4)

## -----------------------------------------------------------------------------
# Convert 10th revision codes to 11th revision
ksic_convert(c("27192", "27195"), from_C = 10, to_C = 11)

# Convert 11th revision codes to 10th revision
ksic_convert(c("27192", "27195"), from_C = 11, to_C = 10)

## -----------------------------------------------------------------------------
# Search for classifications containing "소프트웨어" in the 11th revision
ksic_search("소프트웨어")

# Search for 5-digit classifications containing "software" in the 10th revision (case-sensitive)
ksic_search("software", C = 10, ignore.case = FALSE, digit = 5)

## -----------------------------------------------------------------------------
# Find information for a mix of codes
ksic_find(c("A", "01", "58221", "99999"))

# The result is ordered by the input vector
ksic_find(c("58221", "01", "A"))

## -----------------------------------------------------------------------------
my_data <- data.frame(
  company = c("A", "B", "C", "D"),
  ksic5_cd = c("26222", "58221", "26299", "61220")
)

my_data$ksic2_nm <- ksic_group(my_data$ksic5_cd, digit = 2, name = TRUE)
print(my_data)

## -----------------------------------------------------------------------------
# 분석할 중분류 코드 정의
# Define mid-level divisions for analysis
target_divisions <- c("58", "61") # 출판업, 우편 및 통신업

# ksic_sub를 사용하여 세세분류(5-digit) 코드와 코드명 찾기
# Use ksic_sub to find all 5-digit sub-category codes and names
sub_codes_list <- ksic_sub(target_divisions, digit = 5, name = FALSE)
sub_names_list <- ksic_sub(target_divisions, digit = 5, name = TRUE)

# --- Base R Approach ---
# Base R을 사용하여 리스트를 데이터프레임으로 변환
# Convert the list to a data.frame using Base R
sub_categories_df_base <- data.frame(
  ksic2_cd = rep(names(sub_codes_list), lengths(sub_codes_list)),
  ksic5_cd = unlist(sub_codes_list, use.names = FALSE),
  ksic5_nm = unlist(sub_names_list, use.names = FALSE)
)
print(head(sub_categories_df_base))

# --- Tidyverse Approach ---
# A more concise approach using the tidyverse (tidyr, tibble)
# tidyverse(tidyr, tibble)를 사용한 방법 (더 간결함)
# if (!require(tidyr)) install.packages("tidyr") 
# if (!require(tibble)) install.packages("tibble")

# 1. Create a nested tibble where some columns are lists
# 1. 리스트를 열로 포함하는 중첩된 tibble 생성
nested_tibble <- tibble::tibble(
    ksic2_cd = names(sub_codes_list),
    ksic5_cd = sub_codes_list,
    ksic5_nm = sub_names_list
)
# Step 1: Nested tibble (before unnesting)"
print(nested_tibble)

# 2. Use tidyr::unnest() to expand the list-columns into regular rows
# 2. tidyr::unnest()를 사용하여 리스트 열을 일반적인 행으로 펼침
unnested_df <- tidyr::unnest(nested_tibble, cols = c(ksic5_cd, ksic5_nm))
# Step 2: Unnested tibble (final result)"
print(head(unnested_df))

