params <-
list(css = "css/rmdformats.css")

## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = 'center'
)

## ----echo = FALSE, out.width = "100%", fig.cap = "Fig 1: Cl~int~ experimental set up", fig.topcaption = TRUE, fig.align = "center"----
knitr::include_graphics("img/clint_diagram.png")

## ----setup, message = FALSE, warning = FALSE----------------------------------
# Primary package 
library(invitroTKstats)
# Data manipulation package 
library(dplyr)
# Table formatting package 
library(flextable)

## ----Load example data--------------------------------------------------------
# Load example clint data 
data("Clint-example")

## ----echo = FALSE, warning = FALSE--------------------------------------------
head(clint_L0, n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 1: Level 0 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----required cols, echo = FALSE----------------------------------------------
# Create table of required parameters for Level 1 

req_cols = data.frame(matrix(nrow = 31, ncol = 5))
vars <- c("Argument", "Default", "Required in L0?", "Corresp. single-entry Argument", "Descr.")
colnames(req_cols) <- vars

# Argument names 
arguments <- c("FILENAME", "data.in", "sample.col", "date.col", "compound.col",
                "dtxsid.col", "lab.compound.col", "type.col", "density.col", 
                "cal.col", "dilution.col", "time.col", "istd.col", "istd.name.col", 
                "istd.conc.col", "test.conc.col", "test.nominal.conc.col",
                "area.col", "biological.replicates.col", "technical.replicates.col",
                "analysis.method.col", "analysis.instrument.col",
                "analysis.parameters.col", "note.col", "level0.file.col", 
                "level0.sheet.col", "output.res", "save.bad.types", "sig.figs", "INPUT.DIR",
                "OUTPUT.DIR")
req_cols[,"Argument"] <- arguments

# Default arguments 
defaults <- c("MYDATA", NA, "Lab.Sample.Name", "Date", "Compound.Name",
              "DTXSID", "Lab.Compound.Name", "Sample.Type", "Hep.Density", 
              "Cal", "Dilution.Factor", "Time", "ISTD.Area", "ISTD.Name", "ISTD.Conc", 
              "Test.Compound.Conc", "Test.Target.Conc", "Area", 
              "Biological.Replicates", "Technical.Replicates", "Analysis.Method", 
              "Analysis.Instrument", "Analysis.Parameters", "Note", 
              "Level0.File", "Level0.Sheet", "FALSE", "FALSE", 5, NA, NA)
req_cols[,"Default"] <- defaults

# Argument required in L0?
req_cols <- req_cols %>% 
  mutate("Required in L0?" = case_when(
    Argument %in% c("sample.col","compound.col", "dtxsid.col", "lab.compound.col", "type.col", "istd.col", "area.col") ~ "Y",
    Argument %in% c("FILENAME", "data.in", "output.res", "save.bad.types", "sig.figs", "INPUT.DIR", "OUTPUT.DIR") ~ "N/A", 
    .default = "N"
  ))

# Corresponding single-entry Argument 
req_cols <- req_cols %>% 
  mutate("Corresp. single-entry Argument" = ifelse(.data[[vars[[3]]]] == "N" & .data[[vars[[1]]]] != "note.col", gsub(".col","",Argument),
                NA))

# Brief description 
description <- c("Output and input filename", 
                 "Level 0 data frame", 
                 "Lab sample name",
                 "Lab measurement date",
                 "Formal test compound name",
                 "EPA's DSSTox Structure ID",
                 "Lab test compound name (abbr.)",
                 "Sample type (Blank/Cvst/Inactive/CC)",
                 "Hepatocyte density in the in vitro incubation",
                 "MS calibration",
                 "Number of times sample was diluted",
                 "Time since chemical was introduced",
                 "Internal standard peak area",
                 "Internal standard name", 
                 "Internal standard concentration",
                 "Standard test chemical concentration",
                 "Initial chemical concentration",
                 "Target analyte peak area",
                 "Replicates with the same analyte", 
                 "Repeated measurements from one sample", 
                 "Analytical chemistry analysis method",
                 "Analytical chemistry analysis instrument",
                 "Analytical chemistry analysis parameters", 
                 "Additional notes", 
                 "Raw data filename", 
                 "Raw data sheet name", 
                 "Export results (TSV)?",
                 "Export bad data (TSV)?",
                 "Number of significant figures",
                 "Input directory of Level 0 file",
                 "Export directory to save Level 1 files"
                 )


req_cols[,"Descr."] <- description

## ----echo = FALSE, warning = FALSE--------------------------------------------
req_cols %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        height = 200
      )
      )
  ) %>% 
  set_caption(caption = "Table 2: Level 1 'format_clint' function arguments",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L1 processing------------------------------------------------------------
clint_L1_curated <- format_clint(FILENAME = "Clint_vignette", 
                         data.in = clint_L0,
                         # columns present in L0 data 
                         sample.col = "Sample",
                         compound.col = "Compound",
                         lab.compound.col = "Lab.Compound.ID",
                         type.col = "Type",
                         istd.col = "ISTD.Peak.Area",
                         area.col = "Peak.Area",
                         note.col = "Sample Text",
                         test.conc.col = "Compound.Conc",
                         analysis.parameters.col = "Analysis.Params",
                         # columns not present in L0 data
                         density = 0.5,
                         test.nominal.conc = 1,
                         analysis.instrument = "Unknown",
                         analysis.method = "LCMS",
                         istd.conc = 0.01,
                         cal = 1,
                         biological.replicates = 1,
                         # don't export output TSV file
                         output.res = FALSE
                         )

## ----echo = FALSE-------------------------------------------------------------
clint_L0 %>% 
  filter(Type == "QC") %>% 
  head(n = 5) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        height = 200
      )
      )
  ) %>% 
  set_caption(caption = "Table 3: Subset of removed samples",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----eval = FALSE-------------------------------------------------------------
# # All the samples of an inappropriate sample type
# excluded <- clint_L0 %>%
#   filter(!Type %in% c("Blank", "Cvst", "Inactive", "CC"))
# 
# # Exclude based on Sample and DTXSID
# X <-  c(excluded$Sample)
# names(X) <- paste(excluded$Sample, excluded$DTXSID, sep = "+")
# Y <- c(excluded$DTXSID)
# 
# # Find samples in L1 data frame with matching sample name (X) and DTXSID (Y)
# matches <- as.data.frame(t(mapply(function(X,Y)
#   {subset(clint_L1_curated, Lab.Sample.Name == X & DTXSID == Y)},
#        X, Y, USE.NAMES = T)))
# 
# matches
# 
# # Check that no matches were returned for each sample in `excluded`
# check <-  rep(1,nrow(matches))
# names(check) <- rownames(matches)
# for (name in rownames(matches)) {
#   # If no matches were found, each element of check should evaluate to 0
#   check[name] <- sum(sapply(c(1:ncol(matches)),
#                            function(X) {length(matches[1,X][[1]])}))
# }
# 
# # Verify that each sample in `excluded` had no matches
# check
# 
# # If no matches, should evaluate to 0
# sum(check)

## ----echo = FALSE, warning = FALSE--------------------------------------------
clint_L1_curated %>% 
  head(n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 4: Level 1 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L2 processing exclusion--------------------------------------------------
# Use verification data from loaded in `clint_L2` data frame
exclusion <- clint_L2 %>% 
  filter(Verified %in% c("Wrong Mix", "Unknown Conc.")) %>% 
  mutate("Variables" = "Lab.Sample.Name|DTXSID") %>% 
  mutate("Values" = paste(Lab.Sample.Name, DTXSID, sep = "|")) %>% 
  mutate("Message" = Verified) %>% 
  select(Variables, Values, Message)

## ----echo = FALSE, warning = FALSE--------------------------------------------
exclusion %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        height = 200
      )
      )
  ) %>% 
  set_caption(caption = "Table 5: Exclusion data frame",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L2 processing------------------------------------------------------------
clint_L2_curated <- sample_verification(FILENAME = "Clint_vignette",
                                     data.in = clint_L1_curated,
                                     assay = "Clint",
                                     exclusion.info = exclusion,
                                     # don't export output TSV file 
                                     output.res = FALSE)

## ----echo = FALSE, warning = FALSE--------------------------------------------
df1 <- head(clint_L2_curated, n = 1)
df2 <- head(filter(clint_L2_curated, Verified == "Wrong Mix"), n = 1)
df3 <- head(filter(clint_L2_curated, Verified == "Unknown Conc."), n = 1)

temp <- rbind(df1, df2)
temp <- rbind(temp, df3)

## ----echo = FALSE, warning = FALSE--------------------------------------------
temp %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 6: Level 2 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L3 processing------------------------------------------------------------
clint_L3_curated <- calc_clint_point(FILENAME = "Clint_vignette",
                             data.in = clint_L2_curated,
                             # don't export output TSV file
                             output.res = FALSE)

## ----echo = FALSE-------------------------------------------------------------
clint_L3_curated %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 7: Level 3 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----L4 Processing, message= FALSE, eval = FALSE------------------------------
# clint_L4_curated <- calc_clint(FILENAME = "Clint_vignette",
#                        data.in = clint_L2_curated,
#                        JAGS.PATH = runjags::findjags()
#                        )

## ----echo = FALSE-------------------------------------------------------------
clint_L4_curated <- clint_L4

## ----echo = FALSE-------------------------------------------------------------
clint_L4_curated %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 8: Level 4 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

