/*
 * Portable Exception Handling for ANSI C.
 * Copyright (C) 1999 Kaz Kylheku <kaz@ashi.footprints.net>
 *
 * Free Software License:
 *
 * All rights are reserved by the author, with the following exceptions:
 * Permission is granted to freely reproduce and distribute this software,
 * possibly in exchange for a fee, provided that this copyright notice appears
 * intact. Permission is also granted to adapt this software to produce
 * derivative works, as long as the modified versions carry this copyright
 * notice and additional notices stating that the work has been modified.
 * This source code may be translated into executable form and incorporated
 * into proprietary software; there is no requirement for such software to
 * contain a copyright notice related to this source.
 *
 * $Id: except.h 1.4 2000/02/27 12:27:24 federico Exp $
 * $Name:  $
 * $Log: except.h $
 * Revision 1.4  2000/02/27 12:27:24  federico
 * Added additional void * parameter to memory management functions.
 *
 * Revision 1.3  2000/02/08 23:38:42  federico
 * updated macros definitions.
 *
 */

#ifndef XCEPT_H
#define XCEPT_H

#include <setjmp.h>
#include <stdlib.h>
#include <assert.h>

#ifdef __cplusplus
extern "C"
{
#endif

  enum
    {
      XCEPT_GROUP_ANY = 0,
      XCEPT_CODE_ANY = 0,
      XCEPT_BAD_ALLOC = 1,
      except_no_call,
      except_call
    };

  typedef struct
    {
      unsigned long except_group;
      unsigned long except_code;
    }
  except_id_t;

  typedef struct
    {
      except_id_t except_id;
      const char *except_message;
    }
  except_t;

  struct except_cleanup
    {
      void (*except_func) (void *);
      void *except_context;
    };

  struct except_catch
    {
      const except_id_t *except_id;
      size_t except_size;
      except_t except_obj;
      jmp_buf except_jmp;
    };

  enum except_stacktype
    {
      XCEPT_CLEANUP, XCEPT_CATCHER
    };

  struct except_stacknode
    {
      struct except_stacknode *except_down;
      enum except_stacktype except_type;
      union
        {
          struct except_catch *except_catcher;
          struct except_cleanup *except_cleanup;
        }
      except_info;
    };


  struct except_params
    {
      void (*uh_catcher_ptr) (except_t *);
      struct except_stacknode *top;
    };

/* private functions made external so they can be used in macros */
  void except_setup_clean (struct except_params *ex,
                   struct except_stacknode *esn, struct except_cleanup *ecl,
                           void (*cleanf) (void *), void *context);
  void except_setup_try (struct except_params *ex,
                     struct except_stacknode *esn, struct except_catch *ech,
                         const except_id_t id[], size_t size);
  struct except_stacknode *except_pop (struct except_params *ex);

/* public interface functions */
  int except_init (struct except_params *);
  void except_deinit (struct except_params *);
  void except_rethrow (struct except_params *, except_t *);
  void except_throw (struct except_params *, long group, long code, const char *msg);
  void except_throwd (struct except_params *, long, long, const char *, void *);
  void (*except_unhandled_catcher (struct except_params *ex, void (*new_catcher) (except_t *))) (except_t *);
  unsigned long except_code (except_t *);
  unsigned long except_group (except_t *);
  const char *except_message (except_t *);

#define except_code(E) ((E)->except_id.except_code)
#define except_group(E) ((E)->except_id.except_group)
#define except_message(E) ((E)->except_message)

#ifdef __cplusplus
}
#endif

/*
 * void except_cleanup_push(void (*)(void *), void *);
 * void except_cleanup_pop(int);
 * void except_checked_cleanup_pop(void (*)(void *), int);
 * void except_try_push(const except_id_t [], size_t, except_t **);
 * void except_try_pop(void);
 */

#define except_cleanup_push(F, C) 				\
    {								\
			struct except_stacknode except_sn;			\
			struct except_cleanup except_cl;			\
			except_setup_clean(ex_pars, &except_sn, &except_cl, F, C)

#define except_cleanup_pop(E)					\
	except_pop(ex_pars);						\
	if (E)							\
	    except_cl.except_func(except_cl.except_context);	\
  }

#define except_checked_cleanup_pop(F, E)			\
 	except_pop(ex_pars);						\
	assert (except_cl.except_func == (F));			\
	if (E != NULL)							\
	    except_cl.except_func(except_cl.except_context);	\
  }

#define except_try_push(ID, NUM, PPE)				\
     {								\
			struct except_stacknode except_sn;			\
			struct except_catch except_ch;				\
			except_setup_try(ex_pars, &except_sn, &except_ch, ID, NUM);	\
			if (setjmp(except_ch.except_jmp))			\
		    *(PPE) = &except_ch.except_obj;			\
			else							\
		    *(PPE) = 0

#define except_try_pop()					\
			except_pop(ex_pars);						\
    }

#endif
