#include "iuser.h"

static char_t *rcsid = "$Id: iuser.c 1.2 2000-08-25 18:32:24+02 fede Exp fede $";

#define SECS_PER_MIN IUSER_SECS_PER_MIN
#define SECS_PER_HOUR IUSER_SECS_PER_HOUR

/*
 * Converts a time returned by difftime (t)
 * into hours (*h), minutes (*m) and seconds (*s)
 */
static void
time2hms (double t, int *h, int *m, int *s)
{

  assert (SECS_PER_MIN != 0.0);
  assert (SECS_PER_HOUR != 0.0);

  *h = (int) floor (t / SECS_PER_HOUR);
  *m = (int) floor ((t - *h * SECS_PER_HOUR) / SECS_PER_MIN);
  *s = (int) floor (t - *h * SECS_PER_HOUR - *m * SECS_PER_MIN);
}

/*
 * manage application-defined callbacks to
 * communicate progress to the user
 */
int
iuser_progress (iuser_progress_t * params, const char_t *msg)
{
  assert (params != NULL);

  params->msg = msg;
  params->current = time (NULL);

  switch (params->where)
    {
    case IUSER_PROGRESS_START:
      if (iuser_progress_is_started (params))
        {
          return IUSER_PROGRESS_CAN_CONTINUE;
        }
      else
        {
          if (params->pprogress != NULL)
            {
              if (params->pprogress (params) != IUSER_OK)
                {
                  /*
                   * TODO:
                   * explain the error
                   */
									DEBUGGING (
										fprintf (stderr, "iuser: callback returned error");
										);
                  return IUSER_ERROR;
                }
            }

          params->start = params->prev = params->current;
          params->totlen = params->rlen;

          iuser_progress_set_started (params);
          params->to_go = 0.0;
          params->elapsed = 0.0;
        }
      break;
    case IUSER_PROGRESS_STEP:
      {
        int percentage = 0;
        double elapsed = 0.0;
        double to_go = 0.0;
        double fraction = 0.0;
        int e_hrs, e_min, e_sec;
        int tg_hrs, tg_min, tg_sec;
				int err;

        /*
         * only the top level 'user' updates the informations ...
         */
        if (params->users == 1)
          {
            /* the job done so far ... */
						if (params->totlen != 0.0)
							{
	            	fraction = ((params->totlen - params->rlen));
	              fraction /= (double) params->totlen;
							}
						else
							fraction = 0.0;

            /*
						 * elapsed time, in seconds
						 */
            elapsed = difftime (params->current, params->start);
            if (elapsed != params->elapsed)
              {
								if (fraction != 0.0)
                	to_go = (elapsed / fraction) - elapsed;
								else
									to_go = 0.0;

                if (to_go < 0.0)
                  to_go = 0.0;

                params->elapsed = elapsed;
                params->to_go = to_go;
              }
            else
              {
                to_go = params->to_go;
              }

            /* % done */

						if (elapsed != 0.0 && to_go != 0.0)
              percentage = (int) 100 * ((elapsed) / (to_go + elapsed));
            else
              percentage = 0;

            time2hms (elapsed, &e_hrs, &e_min, &e_sec);
            time2hms (to_go, &tg_hrs, &tg_min, &tg_sec);

            /*
             * A STEP
             * TODO:
             * - call the user function under  different conditions if
             *   the style of the progress is different
             */

            if (percentage != params->percentage)
              {
                params->percentage = percentage;
                iuser_progress_set_updated (params);
              }
            else
              {
                iuser_progress_reset_updated (params);
								assert (!iuser_progress_is_updated (params));
              }

          }

        if (params->pprogress != NULL)
          {
						/*
						 * stopped progress should be handled internally by
						 * the callback
						 */
            err = params->pprogress (params);
            iuser_progress_reset_updated (params);
						return err;
          }
      }
      break;
    case IUSER_PROGRESS_END:
      {
        double elapsed;

        /*
         * this user has finished: if there aren't any more
         * users change status
         */
        if (params->users == 0)
          {
            elapsed = difftime (params->current, params->start);
            if (params->pprogress != NULL)
              params->pprogress (params);

            params->totlen = 0;
            iuser_progress_reset_started (params);
          }
      }
      break;
		case IUSER_PROGRESS_INIT:
		case IUSER_PROGRESS_DEINIT:
			params->pprogress (params);
			break;
    default:
      {
				DEBUGGING (
        	fprintf (stderr, "iuser_progress: unrecognized input received\n");
					);
        return IUSER_INVALID_PARAMETER;
      }
      break;

    }
  return IUSER_OK;
}



/*
 * manage application-defined callbacks to
 * communicate logs to the user
 */
int
iuser_log (iuser_log_t *ilog, const char * file,
	size_t line, const char_t * fmt, ...)
{
	va_list ap;
	assert (ilog != NULL);

	va_start (ap, fmt);
	iuser_vmprintf (&ilog->msg, fmt, ap);
	va_end (ap);

	ilog->file = file;
	ilog->line = line;
	ilog->plog (ilog, IUSER_LOG_MESSAGE);

	iuser_log_free_message (&ilog->msg);

	return IUSER_OK;
}


/*
 * manage application-defined callbacks to
 * handle errors
 */
int
iuser_error (iuser_error_t *ierror, iuser_severity_t severity,
	int code, const char * file, size_t line, const char_t * fmt, ...)
{
	va_list ap;
	assert (ierror != NULL);

	ierror->_severity = severity;
	ierror->_code = code;
	ierror->_file = file;
	ierror->_line = line;

	va_start (ap, fmt);
	iuser_vmprintf (&ierror->_msg, fmt, ap);
	va_end (ap);

	switch (ierror->_context._dispatch_table[severity])
		{
			case IUSER_DISPATCH_NULL:
				break;
			case IUSER_DISPATCH_LOG:
				ierror->_cb (ierror, IUSER_ERROR_LOG);
				/* ? */
				break;
			case IUSER_DISPATCH_DUMP:
				/* ? */
				break;
			case IUSER_DISPATCH_ABORT:
				ierror->_cb (ierror, IUSER_ERROR_ABORTING);
				abort ();
				break;
			case IUSER_DISPATCH_EXIT:
				ierror->_cb (ierror, IUSER_ERROR_EXITING);
				exit (EXIT_FAILURE);
				break;
			case IUSER_DISPATCH_CALLBACK:
				return ierror->_cb (ierror, IUSER_ERROR_HANDLE);
				break;
		}

	iuser_log_free_message (&ierror->_msg);

	return IUSER_OK;
}

