#include "iuser.h"

static char_t * rcsid = "$Id: progress.c 1.17 2000-08-23 18:15:12+02 fede Exp fede $";

/*
 * TODO:
 * - define a standard behaviuor to manage user interruption in
 *   client code
 * - manage iuser_progress_set_msg
 * - assure publish new functions in iuser.h
 * - document: assumes (at least) 32-bit ints (WHY?)
 * - optimize with macros
 * - delay to return a valid remaining time until it hasn't stabilized
 * - manage params->users counts in unusual situations (?)
 * - manage situations where we have to wait for a while before
 *   terminating the action (pheraps via _STOPPING)
 * - if I stop the progress the iuser_progress function shouldn't return
 *   and should wait for confirmation, if requested
 * - CLOSE_PROGRESS in case of error and stop
 * - Unicode
 * - Different win32 dialogs: with Cancel button, simple, etc
 *   the dialogs with no cancellation possibility should be repainted,
 *   i.e. the window function should call DefDlgProc or so
 * - test for progress initialization on function entry
 * - an alternative to the current iuser_progress_init function is
 *   a function to be called after the parameter have been set
 * - use IUSER_PROGRESS_CAN_CANCEL
 *
 * $Log: progress.c $
 * Revision 1.17  2000-08-23 18:15:12+02  fede
 * Removed iuser_progress.
 * char -> char_t to eventually support UNICODE.
 * Removed DEBUGGING macro definition.
 * Renamed SECS_PER_MIN/SECS_PER_HOUR.
 * iuser_progress_init left here.
 *
 * Revision 1.16  2000-08-16 22:41:09+02  fede
 * Removed a strange code from iuser_progress_set_flags
 *
 * Revision 1.15  2000-08-15 16:51:09+02  fede
 * Fixed iuser_progress_stdio
 * Fixed iuser_porgress_deinit
 *
 * Revision 1.14  2000-08-15 12:04:55+02  fede
 * Added interface to stop fields.
 *
 * Revision 1.13  2000-08-14 17:01:11+02  fede
 * Modified the*_reset_* functions as they didn't work (I was using '^=' to remove a bit instead of '&= ~').
 * Remove the second call to the callback when it returns l a negative error: progress stopping is handled as follows: the callback handles its details and returns the IUSER_PROGRESS_STOPPED which is negative; then the iuser_progress returns that value to be used by the client application.
 *
 * Revision 1.12  2000-08-13 20:25:56+02  federico
 * Removed redundant enums, and functions mainly dedicated to progress
 * stop handling.
 *
 *
 * Revision 1.11  2000/06/25 12:48:21  federico
 * Added IUSER_PROGRESS_UPDATED flag and relative functions.
 * N. of users managed by specialized functions.
 * iuser_progress_why renamed to iuser_progress_where.
 *
 * Revision 1.10  2000/06/18 19:53:28  federico
 * Reverted to previous situation with better error reporting ...
 *
 * Revision 1.9  2000/06/14 22:18:15  federico
 * added several interfaces to manage info (active, was stopped)
 * and where (start, step, end, stopping).
 *
 * Revision 1.8  2000/06/12 23:24:18  federico
 * added iuser_progress_n_of_users.
 * some problems estimating remainig time.
 *
 * Revision 1.7  2000/06/10 15:05:40  federico
 * added iuser_progress_set_where.
 * modified iuser_progress.
 * Now the information about the step to be performed are taken directly from
 * the where field in the iuser_progress_t structure.
 *
 * Revision 1.6  2000/06/10 14:49:39  federico
 * added iuser_get_msg.
 * added IUSER_PROGRESS_STOPPED to iuser_progress_why.
 * added iuser_progress_stopping, iuser_progress_stopped, iuser_progress_is_stopping.
 * added rude management of progress stopping in iuser_progress.
 *
 * Revision 1.5  2000/05/04 23:54:32  federico
 * Added activation flag, initialization flag and relative interface.
 * main is compiled #ifdef TEST.
 *
 * Revision 1.3  2000/04/30 15:50:45  federico
 * Added something working with win32.
 *
 * Revision 1.2  2000/04/25 17:30:28  federico
 * The progress interface has evolved.
 * With fprintf/stderr works quite well.
 *
 * Revision 1.1  2000/04/19 22:33:16  federico
 * Initial revision
 *
 *
 */

/*
   ^.^ \./ _._
 */


#ifndef max
#define max(a,b) (((a) < (b)) ? (b) : (a))
#endif

const double IUSER_SECS_PER_MIN = 60.0;
const double IUSER_SECS_PER_HOUR = 3600.0;

#define SECS_PER_MIN IUSER_SECS_PER_MIN
#define SECS_PER_HOUR IUSER_SECS_PER_HOUR

void
iuser_progress_remove_user (iuser_progress_t * params)
{
  assert (params != NULL);
  --params->users;
}

void
iuser_progress_add_user (iuser_progress_t * params)
{
  assert (params != NULL);
  ++params->users;
}

void
iuser_progress_reset_stop (iuser_progress_t * params)
{
	assert (params != NULL);
	params->stop = 0;
}

int
iuser_progress_is_stop (const iuser_progress_t * params)
{
	return params->stop;
}

void
iuser_progress_set_stop (iuser_progress_t * params)
{
	assert (params != NULL);
	params->stop = 1;
}

void
iuser_progress_reset_wait (iuser_progress_t * params)
{
	assert (params != NULL);
	params->wait = 0;
}

int
iuser_progress_is_wait (const iuser_progress_t * params)
{
	return params->wait;
}

void
iuser_progress_set_wait (iuser_progress_t * params)
{
	assert (params != NULL);
	params->wait = 1;
}

void
iuser_progress_set_updated (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info |= IUSER_PROGRESS_IS_UPDATED;
}

void
iuser_progress_reset_updated (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info &= ~IUSER_PROGRESS_IS_UPDATED;
	assert (!(params->info & IUSER_PROGRESS_IS_UPDATED));
}

int
iuser_progress_is_updated (const iuser_progress_t * params)
{
  assert (params != NULL);
  return params->info & IUSER_PROGRESS_IS_UPDATED;
}

void
iuser_progress_set_where (iuser_progress_t * status, const int where)
{
  assert (status != NULL);
  status->where = where;
}

void
iuser_progress_set_length (iuser_progress_t * status, size_t i)
{
  assert (status != NULL);

  if (status->users == 1)
    status->rlen = i;
}

void
iuser_progress_set_type (iuser_progress_t * status, int flags)
{
  assert (status != NULL);
  status->flags = flags;
}

void
iuser_progress_set_opaque (iuser_progress_t * status, void *opaque)
{
  assert (status != NULL);
  status->opaque = opaque;
}

void
iuser_progress_set_callback (iuser_progress_t * status,
                             int (*callback) (iuser_progress_t *))
{
  assert (status != NULL);
  status->pprogress = callback;
}

void
iuser_progress_set_msg (iuser_progress_t * status, const char_t *msg)
{
  assert (status != NULL);
  status->msg = msg;
}

void
iuser_progress_set_flags (iuser_progress_t * status, int flags)
{
  assert (status != NULL);
  status->flags = flags;
}

char_t *
iuser_progress_get_msg (const iuser_progress_t * status)
{
  assert (status != NULL);
  return status->msg;
}

void *
iuser_progress_get_opaque (const iuser_progress_t * status)
{
  assert (status != NULL);
  return status->opaque;
}

int
iuser_progress_get_elapsed_hrs (const iuser_progress_t * status)
{
  assert (status != NULL);
  return (int) max (floor (status->elapsed / SECS_PER_HOUR), 0.0);
}

int
iuser_progress_get_elapsed_mins (const iuser_progress_t * status)
{
  int h;
  assert (status != NULL);
  h = (int) floor (status->elapsed / SECS_PER_HOUR);
  return (int) max (floor ((status->elapsed
                            - h * SECS_PER_HOUR) / SECS_PER_MIN), 0.0);
}

int
iuser_progress_get_elapsed_secs (const iuser_progress_t * status)
{
  int h, m;
  assert (status != NULL);
  h = (int) floor (status->elapsed / SECS_PER_HOUR);
  m = (int) floor ((status->elapsed - h * SECS_PER_HOUR) / SECS_PER_MIN);
  return (int) max (floor (status->elapsed
                           - h * SECS_PER_HOUR - m * SECS_PER_MIN), 0.0);
}

int
iuser_progress_get_remaining_hrs (const iuser_progress_t * status)
{
  assert (status != NULL);
  return (int) max (floor (status->elapsed / SECS_PER_HOUR), 0.0);
}

int
iuser_progress_get_remaining_mins (const iuser_progress_t * status)
{
  int h;
  assert (status != NULL);
  h = (int) floor (status->to_go / SECS_PER_HOUR);
  return (int) max (floor ((status->to_go
                            - h * SECS_PER_HOUR) / SECS_PER_MIN), 0.0);
}

int
iuser_progress_get_remaining_secs (const iuser_progress_t * status)
{
  int h, m;
  assert (status != NULL);
  h = (int) floor (status->to_go / SECS_PER_HOUR);
  m = (int) floor ((status->to_go - h * SECS_PER_HOUR) / SECS_PER_MIN);
  return (int) max (floor (status->to_go
                           - h * SECS_PER_HOUR - m * SECS_PER_MIN), 0.0);
}

int
iuser_progress_get_percentage (const iuser_progress_t * status)
{
  assert (status != NULL);
  return status->percentage;
}

int
iuser_progress_get_flags (const iuser_progress_t * status)
{
  assert (status != NULL);
  return status->flags;
}

int
iuser_progress_where (const iuser_progress_t * status)
{
  return status->where;
}

void
iuser_progress_activate (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info |= IUSER_PROGRESS_IS_ACTIVE;
}

void
iuser_progress_deactivate (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info &= ~IUSER_PROGRESS_IS_ACTIVE;
}

int
iuser_progress_is_active (const iuser_progress_t * params)
{
  assert (params != NULL);
  return (params->info & IUSER_PROGRESS_IS_ACTIVE);
}

int
iuser_progress_nusers (iuser_progress_t * params)
{
  assert (params != NULL);

  return params->users;
}

int
iuser_progress_is_started (iuser_progress_t * params)
{
  assert (params != NULL);
  return (params->info & IUSER_PROGRESS_IS_STARTED);
}

void
iuser_progress_set_started (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info |= IUSER_PROGRESS_IS_STARTED;
}

void
iuser_progress_reset_started (iuser_progress_t * params)
{
  assert (params != NULL);
  params->info &= ~IUSER_PROGRESS_IS_STARTED;
}


#define START_PROGRESS(i, msg) \
	if (iuser_progress_is_active(&status)) \
  { \
		iuser_progress_add_user (&status); \
		iuser_progress_set_where (&status, IUSER_PROGRESS_START);\
		iuser_progress_set_length (&status, i); \
		iuser_progress (&status, msg); \
	}

#define DO_PROGRESS(i) \
	if (iuser_progress_is_active(&status)) \
	{ \
		iuser_progress_set_where (&status, IUSER_PROGRESS_STEP);\
		iuser_progress_set_length (&status, i); \
		iuser_progress (&status, AS_TEXT("")); \
	}

#define END_PROGRESS() \
	if (iuser_progress_is_active(&status)) \
  { \
		iuser_progress_set_where (&status, IUSER_PROGRESS_END);\
		iuser_progress_set_length (&status, 0); \
		iuser_progress_remove_user (&status); \
		iuser_progress (&status, AS_TEXT("")); \
	}

typedef struct _my_progress_params_s
{
  FILE *stream;
  int (*f) (FILE * fp, const char_t *fmt,...);
}
my_progress_params_t;

int
iuser_progress_stdio (iuser_progress_t * status)
{
  static const char_t *fmt_both =
  AS_TEXT("\r > Completed: %3d%%, elapsed: %02d:%02d:%02d, to_go: %02d:%02d:%02d");

  static const char_t *fmt_time =
  AS_TEXT("\r > Elapsed: %02d:%02d:%02d, to_go: %02d:%02d:%02d");

  static const char_t *fmt_perc = AS_TEXT("\r > Completed: %3d%%");

  char_t iuser_progress_simple[] =
  {AS_TEXT('-'), AS_TEXT('\\'), AS_TEXT('|'), AS_TEXT('/')};

  my_progress_params_t *pp = (my_progress_params_t *)
		iuser_progress_get_opaque (status);
  FILE *fp = pp->stream;
  const char_t *fmt;
  int e_hrs, e_min, e_sec;
  int tg_hrs, tg_min, tg_sec;
  char_t tmp[IUSER_TEXT_SCREEN_WIDTH], tmp_out[IUSER_TEXT_SCREEN_WIDTH];

  switch (iuser_progress_where (status))
    {
    case IUSER_PROGRESS_START:
      pp->f (fp, AS_TEXT("PROGRESS - %s:\n"), status->msg);
      pp->f (fp, AS_TEXT(" > wait ... "));
      fflush (fp);
			return IUSER_OK;
      break;
    case IUSER_PROGRESS_STEP:
			if (!iuser_progress_is_updated (status))
				return IUSER_PROGRESS_CAN_CONTINUE;

      e_hrs = iuser_progress_get_elapsed_hrs (status);
      e_min = iuser_progress_get_elapsed_mins (status);
      e_sec = iuser_progress_get_elapsed_secs (status);
      tg_hrs = iuser_progress_get_remaining_hrs (status);
      tg_min = iuser_progress_get_remaining_mins (status);
      tg_sec = iuser_progress_get_remaining_secs (status);

      if (status->flags == IUSER_PROGRESS_STYLE_TIME)
        {
          fmt = fmt_time;
          pp->f (fp, fmt, e_hrs, e_min, e_sec,
                 tg_hrs, tg_min, tg_sec);
        }
      else if (status->flags == IUSER_PROGRESS_STYLE_PERCENTAGE)
        {
          fmt = fmt_perc;
          pp->f (fp, fmt, status->percentage);
        }
      else if ((status->flags & IUSER_PROGRESS_STYLE_TIME)
               && (status->flags & IUSER_PROGRESS_STYLE_PERCENTAGE))
        {
          fmt = fmt_both;
          pp->f (fp, fmt, status->percentage, e_hrs, e_min, e_sec,
                 tg_hrs, tg_min, tg_sec);
        }
      else
        {
          int i = status->percentage % sizeof (iuser_progress_simple);
          pp->f (stderr, AS_TEXT("\r > %-10s \r > %c"),
					AS_TEXT(""), iuser_progress_simple[i]);
        }

      fflush (fp);

			return IUSER_PROGRESS_CAN_CONTINUE;
      break;
    case IUSER_PROGRESS_END:
      e_hrs = iuser_progress_get_elapsed_hrs (status);
      e_min = iuser_progress_get_elapsed_mins (status);
      e_sec = iuser_progress_get_elapsed_secs (status);

      if (e_hrs == 0)
        {
          sprintf (tmp, AS_TEXT("DONE ! (elapsed: %02dm:%02ds)"), e_min, e_sec);
        }
      else
        {
          sprintf (tmp, AS_TEXT("DONE ! (elapsed: %dh:%02dm:%02ds)"),
                   e_hrs, e_min, e_sec);
        }
      sprintf (tmp_out, AS_TEXT("\r > %-*s\n"),
				IUSER_TEXT_SCREEN_WIDTH - 4, tmp);
      pp->f (fp, tmp_out);
      fflush (fp);
			return IUSER_OK;
      break;
		case IUSER_PROGRESS_INIT:
		case IUSER_PROGRESS_DEINIT:
			DEBUGGING (
				fprintf (stderr,
					iuser_progress_where (status) == IUSER_PROGRESS_INIT ?
					"Callback called upon initialization\n" :
					"Callback called upon deinitialization\n");
			);
			return IUSER_OK;
			break;
		default:
			break;
    }
  return IUSER_PROGRESS_CAN_CONTINUE;
}

int
iuser_progress_init (iuser_progress_t * status, int flags,
                     iuser_progress_callback_t cb, void *opaque)
{
	int rv = IUSER_OK;

  assert (status != NULL);
  memset (status, 0, sizeof (*status));

  status->pprogress = cb;
  status->opaque = opaque;
  status->flags = flags;
  status->info = 0;
	/*
	 * TODO: ???
	 */
  iuser_progress_activate (status);
  status->where = IUSER_PROGRESS_INIT;

  if (status->pprogress != NULL)
    rv = status->pprogress (status);

  if (rv == IUSER_OK) status->info |= IUSER_PROGRESS_IS_INITIALIZED;
	else status->info |= IUSER_PROGRESS_IS_DEINITIALIZED;

  return rv;
}

/*
 * set the fields to NULL, zero as appropriate.
 * calls the callback function provided by the caller if not NULL
 * passing IUSER_PROGRESS_DEINIT as where
 */

void
iuser_progress_deinit (iuser_progress_t * status)
{
  assert (status != NULL);

  iuser_progress_deactivate (status);
	iuser_progress_reset_started (status);

  iuser_progress_set_where (status, IUSER_PROGRESS_DEINIT);

  if (status->pprogress != NULL)
    status->pprogress (status);

  status->pprogress = NULL;
  memset (status, 0, sizeof (*status));
  status->info |= IUSER_PROGRESS_IS_DEINITIALIZED;
}

#ifdef TEST

static void
job (iuser_progress_t * _status, long MAX)
{
	long i;
	iuser_progress_t status;

	memcpy (&status, _status, sizeof (iuser_progress_t));

  START_PROGRESS (MAX, AS_TEXT("ciclo for (1)"));
  for (i = 0; i < MAX; ++i)
    {
      DO_PROGRESS (MAX - i);
    }
  END_PROGRESS ();

}

int
main (int argc, char_t **argv)
{
  my_progress_params_t _my_progress_params;
  my_progress_params_t *my_progress_params = &_my_progress_params;
  long MAX = atoi (argc > 1 ? argv[1] : "20000");
  iuser_progress_t status;
  long i;

  printf ("MAX: %ld\n", MAX);

  memset (&status, 0, sizeof (status));

  my_progress_params->stream = stderr;
  my_progress_params->f = fprintf;

  iuser_progress_init (&status, IUSER_PROGRESS_STYLE_SIMPLE,
                       iuser_progress_stdio, my_progress_params);
/*
   iuser_progress_set_type (&status, IUSER_PROGRESS_STYLE_SIMPLE);
   iuser_progress_set_opaque (&status, my_progress_params);
   iuser_progress_set_callback (&status, my_progress);
 */
  START_PROGRESS (MAX, AS_TEXT("ciclo for (1)"));
  for (i = 0; i < MAX; ++i)
    {
			job (&status, MAX);
      DO_PROGRESS (MAX - i);
    }
  END_PROGRESS ();

  iuser_progress_set_type (&status, IUSER_PROGRESS_STYLE_TIME);
  /*
     iuser_progress_deactivate (&status);
   */

  START_PROGRESS (MAX, AS_TEXT("ciclo for (2)"));
  for (i = 0; i < MAX; ++i)
    {
      DO_PROGRESS (MAX - i);
    }
  END_PROGRESS ();

  /*
     iuser_progress_activate (&status);
   */
  iuser_progress_set_type (&status, IUSER_PROGRESS_STYLE_PERCENTAGE);

  START_PROGRESS (MAX, AS_TEXT("ciclo for (3)"));
  for (i = 0; i < MAX; ++i)
    {
      DO_PROGRESS (MAX - i);
    }
  END_PROGRESS ();

  iuser_progress_set_type (&status, IUSER_PROGRESS_STYLE_PERCENTAGE |
                           IUSER_PROGRESS_STYLE_TIME);

  START_PROGRESS (MAX, AS_TEXT("ciclo for (4)"));
  for (i = 0; i < MAX; ++i)
    {
      DO_PROGRESS (MAX - i);
    }
  END_PROGRESS ();

	iuser_progress_deinit (&status);

  return EXIT_SUCCESS;
}

#endif /* TEST */
