% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RIDGEsigma.R
\name{RIDGEsigma}
\alias{RIDGEsigma}
\title{Ridge penalized precision matrix estimation}
\usage{
RIDGEsigma(X = NULL, S = NULL, lam = 10^seq(-2, 2, 0.1), path = FALSE,
  K = 5, cores = 1, trace = c("none", "progress", "print"))
}
\arguments{
\item{X}{option to provide a nxp data matrix. Each row corresponds to a single observation and each column contains n observations of a single feature/variable.}

\item{S}{option to provide a pxp sample covariance matrix (denominator n). If argument is \code{NULL} and \code{X} is provided instead then \code{S} will be computed automatically.}

\item{lam}{positive tuning parameters for ridge penalty. If a vector of parameters is provided, they should be in increasing order. Defaults to grid of values \code{10^seq(-2, 2, 0.1)}.}

\item{path}{option to return the regularization path. This option should be used with extreme care if the dimension is large. If set to TRUE, cores will be set to 1 and errors and optimal tuning parameters will based on the full sample. Defaults to FALSE.}

\item{K}{specify the number of folds for cross validation.}

\item{cores}{option to run CV in parallel. Defaults to \code{cores = 1}.}

\item{trace}{option to display progress of CV. Choose one of \code{progress} to print a progress bar, \code{print} to print completed tuning parameters, or \code{none}.}
}
\value{
returns class object \code{RIDGEsigma} which includes:
\item{Lambda}{optimal tuning parameter.}
\item{Lambdas}{grid of lambda values for CV.}
\item{Omega}{estimated penalized precision matrix.}
\item{Sigma}{estimated covariance matrix from the penalized precision matrix (inverse of Omega).}
\item{Path}{array containing the solution path. Solutions are ordered dense to sparse.}
\item{Gradient}{gradient of optimization function (penalized gaussian likelihood).}
\item{MIN.error}{minimum average cross validation error (cv.crit) for optimal parameters.}
\item{AVG.error}{average cross validation error (cv.crit) across all folds.}
\item{CV.error}{cross validation errors (cv.crit).}
}
\description{
Ridge penalized matrix estimation via closed-form solution. If one is only interested in the ridge penalty, this function will be faster and provide a more precise estimate than using \code{ADMMsigma}. \cr
Consider the case where
\eqn{X_{1}, ..., X_{n}} are iid \eqn{N_{p}(\mu, \Sigma)}
and we are tasked with estimating the precision matrix,
denoted \eqn{\Omega \equiv \Sigma^{-1}}. This function solves the
following optimization problem:
\describe{
\item{Objective:}{
\eqn{\hat{\Omega}_{\lambda} = \arg\min_{\Omega \in S_{+}^{p}}
\left\{ Tr\left(S\Omega\right) - \log \det\left(\Omega \right) +
\frac{\lambda}{2}\left\| \Omega \right\|_{F}^{2} \right\}}}
}
where \eqn{\lambda > 0} and \eqn{\left\|\cdot \right\|_{F}^{2}} is the Frobenius
norm.
}
\examples{
# generate data from a sparse matrix
# first compute covariance matrix
S = matrix(0.7, nrow = 5, ncol = 5)
for (i in 1:5){
 for (j in 1:5){
   S[i, j] = S[i, j]^abs(i - j)
 }
 }

# generate 100 x 5 matrix with rows drawn from iid N_p(0, S)
Z = matrix(rnorm(100*5), nrow = 100, ncol = 5)
out = eigen(S, symmetric = TRUE)
S.sqrt = out$vectors \%*\% diag(out$values^0.5)
S.sqrt = S.sqrt \%*\% t(out$vectors)
X = Z \%*\% S.sqrt

# ridge penalty no ADMM
RIDGEsigma(X, lam = 10^seq(-5, 5, 0.5))
}
\references{
\itemize{
\item Rothman, Adam. 2017. 'STAT 8931 notes on an algorithm to compute the Lasso-penalized Gaussian likelihood precision matrix estimator.'
}
}
\seealso{
\code{\link{plot.RIDGE}}, \code{\link{ADMMsigma}}
}
\author{
Matt Galloway \email{gall0441@umn.edu}
}
