% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AIPW.R
\name{AIPW}
\alias{AIPW}
\title{Augmented Inverse Probability Weighting (AIPW)}
\value{
\code{AIPW} object
}
\description{
An R6Class of AIPW for estimating the average causal effects with users' inputs of exposure, outcome, covariates and related
libraries for estimating the efficient influence function.
}
\details{
An AIPW object is constructed by \code{new()} with users' inputs of data and causal structures, then it \code{fit()} the data using the
libraries in \code{Q.SL.library} and \code{g.SL.library} with \code{k_split} cross-fitting, and provides results via the \code{summary()} method.
After using \code{fit()} and/or \code{summary()} methods, propensity scores  and inverse probability weights by exposure status can be
examined with \code{plot.p_score()} and \code{plot.ip_weights()}, respectively.

If outcome is missing, analysis assumes missing at random (MAR) by estimating propensity scores of I(A=a, observed=1) with all covariates \code{W}.
(\code{W.Q} and \code{W.g} are disabled.) Missing exposure is not supported.

See examples for illustration.
}
\section{Constructor}{

\code{AIPW$new(Y = NULL, A = NULL, W = NULL, W.Q = NULL, W.g = NULL, Q.SL.library = NULL, g.SL.library = NULL, k_split = 10, verbose = TRUE, save.sl.fit = FALSE)}
\subsection{Constructor Arguments}{

\tabular{lll}{
\strong{Argument}      \tab   \strong{Type}     \tab     \strong{Details} \cr
\code{Y}               \tab   Integer           \tab     A vector of outcome (binary (0, 1) or continuous) \cr
\code{A}               \tab   Integer           \tab     A vector of binary exposure (0 or 1) \cr
\code{W}               \tab   Data              \tab     Covariates for \strong{both} exposure and outcome models. \cr
\code{W.Q}             \tab   Data              \tab     Covariates for the \strong{outcome} model (Q).\cr
\code{W.g}             \tab   Data              \tab     Covariates for the \strong{exposure} model (g). \cr
\code{Q.SL.library}    \tab   SL.library        \tab     Algorithms used for the \strong{outcome} model (Q). \cr
\code{g.SL.library}    \tab   SL.library         \tab    Algorithms used for the \strong{exposure} model (g). \cr
\code{k_split}         \tab   Integer           \tab    Number of folds for splitting (Default = 10).\cr
\code{verbose}         \tab   Logical           \tab    Whether to print the result (Default = TRUE) \cr
\code{save.sl.fit}     \tab   Logical          \tab     Whether to save Q.fit and g.fit (Default = FALSE) \cr
}
}

\subsection{Constructor Argument Details}{

\describe{
\item{\code{W}, \code{W.Q} & \code{W.g}}{It can be a vector, matrix or data.frame. If and only if \code{W == NULL}, \code{W} would be replaced by \code{W.Q} and \code{W.g}. }
\item{\code{Q.SL.library} & \code{g.SL.library}}{Machine learning algorithms from \link{SuperLearner} libraries}
\item{\code{k_split}}{It ranges from 1 to number of observation-1.
If k_split=1, no cross-fitting; if k_split>=2, cross-fitting is used
(e.g., \code{k_split=10}, use 9/10 of the data to estimate and the remaining 1/10 leftover to predict).
\strong{NOTE: it's recommended to use cross-fitting.} }
\item{\code{save.sl.fit}}{This option allows users to save the fitted sl object (libs$Q.fit & libs$g.fit) for debug use.
\strong{Warning: Saving the SuperLearner fitted object may cause a substantive storage/memory use.}}
}
}
}

\section{Public Methods}{

\tabular{lll}{
\strong{Methods}      \tab   \strong{Details}                                      \tab \strong{Link}     \cr
\code{fit()}          \tab   Fit the data to the \link{AIPW} object                     \tab   \link{fit.AIPW}  \cr
\code{stratified_fit()}\tab   Fit the data to the \link{AIPW} object stratified by \code{A}  \tab   \link{stratified_fit.AIPW}  \cr
\code{summary()}      \tab   Summary of the average treatment effects from AIPW    \tab   \link{summary.AIPW_base}\cr
\code{plot.p_score()} \tab   Plot the propensity scores by exposure status         \tab   \link{plot.p_score}\cr
\code{plot.ip_weights()} \tab   Plot the inverse probability weights using truncated propensity scores  \tab   \link{plot.ip_weights}\cr
}
}

\section{Public Variables}{

\tabular{lll}{
\strong{Variable}     \tab   \strong{Generated by}      \tab     \strong{Return} \cr
\code{n}              \tab   Constructor                \tab     Number of observations \cr
\code{stratified_fitted} \tab   \code{stratified_fit()}      \tab    Fit the outcome model stratified by exposure status \cr
\code{obs_est}        \tab   \code{fit()} & \code{summary()}      \tab     Components calculating average causal effects \cr
\code{estimates}      \tab   \code{summary()}                \tab     A list of Risk difference, risk ratio, odds ratio \cr
\code{result}         \tab   \code{summary()}                \tab     A matrix contains RD, ATT, ATC, RR and OR with their SE and 95\%CI \cr
\code{g.plot}         \tab   \code{plot.p_score()}           \tab     A density plot of propensity scores by exposure status\cr
\code{ip_weights.plot}         \tab   \code{plot.ip_weights()}           \tab     A box plot of inverse probability weights \cr
\code{libs}           \tab   \code{fit()}                    \tab     \link{SuperLearner} libraries and their fitted objects \cr
\code{sl.fit}         \tab   Constructor                \tab     A wrapper function for fitting \link{SuperLearner} \cr
\code{sl.predict}     \tab   Constructor                \tab     A wrapper function using \code{sl.fit} to predict \cr
}
\subsection{Public Variable Details}{

\describe{
\item{\code{stratified_fit}}{An indicator for whether the outcome model is fitted stratified by exposure status in the\code{fit()} method.
Only when using \code{stratified_fit()} to turn on \code{stratified_fit = TRUE}, \code{summary} outputs average treatment effects among the treated and the controls.}
\item{\code{obs_est}}{After using \code{fit()} and \code{summary()} methods, this list contains the propensity scores (\code{p_score}),
counterfactual predictions (\code{mu}, \code{mu1} & \code{mu0}) and
efficient influence functions (\code{aipw_eif1} & \code{aipw_eif0}) for later average treatment effect calculations.}
\item{\code{g.plot}}{This plot is generated by \code{ggplot2::geom_density}}
\item{\code{ip_weights.plot}}{This plot uses truncated propensity scores stratified by exposure status (\code{ggplot2::geom_boxplot})}
}
}
}

\examples{
library(SuperLearner)
library(ggplot2)

#create an object
aipw_sl <- AIPW$new(Y=rbinom(100,1,0.5), A=rbinom(100,1,0.5),
                    W.Q=rbinom(100,1,0.5), W.g=rbinom(100,1,0.5),
                    Q.SL.library="SL.mean",g.SL.library="SL.mean",
                    k_split=1,verbose=FALSE)

#fit the object
aipw_sl$fit()
# or use `aipw_sl$stratified_fit()` to estimate ATE and ATT/ATC

#calculate the results
aipw_sl$summary(g.bound = 0.025)

#check the propensity scores by exposure status after truncation
aipw_sl$plot.p_score()

}
\references{
Zhong Y, Kennedy EH, Bodnar LM, Naimi AI (2021, In Press). AIPW: An R Package for Augmented Inverse Probability Weighted Estimation of Average Causal Effects. \emph{American Journal of Epidemiology}.

Robins JM, Rotnitzky A (1995). Semiparametric efficiency in multivariate regression models with missing data. \emph{Journal of the American Statistical Association}.

Chernozhukov V, Chetverikov V, Demirer M, et al (2018). Double/debiased machine learning for treatment and structural parameters. \emph{The Econometrics Journal}.

Kennedy EH, Sjolander A, Small DS (2015). Semiparametric causal inference in matched cohort studies. \emph{Biometrika}.
}
