% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/certificates.R
\docType{class}
\name{certificates}
\alias{certificates}
\alias{certs}
\title{Certificates in Key Vault}
\description{
This class represents the collection of certificates stored in a vault. It provides methods for managing certificates, including creating, importing and deleting certificates, and doing backups and restores. For operations with a specific certificate, see \link{certificate}.
}
\section{Methods}{

This class provides the following methods:\preformatted{create(name, subject, x509=cert_x509_properties(), issuer=cert_issuer_properties(),
       key=cert_key_properties(), format=c("pem", "pkcs12"),
       expiry_action=cert_expiry_action(),
       attributes=vault_object_attrs(),
       ..., wait=TRUE)
import(name, value, pwd=NULL,
       attributes=vault_object_attrs(),
       ..., wait=TRUE)
get(name)
delete(name, confirm=TRUE)
list()
backup(name)
restore(backup)
get_contacts()
set_contacts(email)
add_issuer(issuer, provider, credentials=NULL, details=NULL)
remove_issuer(issuer)
get_issuer(issuer)
list_issuers()
}
}

\section{Arguments}{

\itemize{
\item \code{name}: The name of the certificate.
\item \code{subject}: For \code{create}, The subject or X.500 distinguished name for the certificate.
\item \code{x509}: Other X.509 properties for the certificate, such as the domain name(s) and validity period. A convenient way to provide this is via the \link{cert_x509_properties} helper function.
\item \code{issuer}: Issuer properties for the certificate. A convenient way to provide this is via the \link{cert_issuer_properties} helper function. The default is to specify a self-signed certificate.
\item \code{key}: Key properties for the certificate. A convenient way to provide this is via the \link{cert_key_properties} helper function.
\item \code{format}: The format to store the certificate in. Can be either PEM or PFX, aka PKCS#12. This also determines the format in which the certificate will be exported (see \link{certificate}).
\item \code{expiry_action}: What Key Vault should do when the certificate is about to expire. A convenient way to provide this is via the \link{cert_expiry_action} helper function.
\item \code{attributes}: Optional attributes for the secret. A convenient way to provide this is via the \link{vault_object_attrs} helper function.
\item \code{value}: For \code{import}, the certificate to import. This can be the name of a PFX file, or a raw vector with the contents of the file.
\item \code{pwd}: For \code{import}, the password if the imported certificate is password-protected.
\item \code{...}: For \code{create} and \code{import}, other named arguments which will be treated as tags.
\item \code{wait}: For \code{create} and \code{import}, whether to wait until the certificate has been created before returning. If FALSE, you can check on the status of the certificate via the returned object's \code{sync} method.
\item \code{backup}: For \code{restore}, a string representing the backup blob for a key.
\item \code{email}: For \code{set_contacts}, the email addresses of the contacts.
\item \code{issuer}: For the issuer methods, the name by which to refer to an issuer.
\item \code{provider}: For \code{add_issuer}, the provider name as a string.
\item \code{credentials}: For \code{add_issuer}, the credentials for the issuer, if required. Should be a list containing the components \code{account_id} and \code{password}.
\item \code{details}: For \code{add_issuer}, the organisation details, if required. See the \href{https://docs.microsoft.com/en-us/rest/api/keyvault/setcertificateissuer/setcertificateissuer#administratordetails}{Azure docs} for more information.
}
}

\section{Value}{

For \code{get}, \code{create} and \code{import}, an object of class \code{stored_certificate}, representing the certificate itself.

For \code{list}, a vector of key names.

For \code{add_issuer} and \code{get_issuer}, an object representing an issuer. For \code{list_issuers}, a vector of issuer names.

For \code{backup}, a string representing the backup blob for a certificate. If the certificate has multiple versions, the blob will contain all versions.
}

\examples{
\dontrun{

vault <- key_vault("mykeyvault")

vault$certificates$create("mynewcert", "CN=mydomain.com")
vault$certificates$list()
vault$certificates$get("mynewcert")

# specifying some domain names
vault$certificates$create("mynewcert", "CN=mydomain.com",
    x509=cert_x509_properties(dns_names=c("mydomain.com", "otherdomain.com")))

# specifying a validity period of 2 years (24 months)
vault$certificates$create("mynewcert", "CN=mydomain.com",
    x509=cert_x509_properties(validity_months=24))

# setting management tags
vault$certificates$create("mynewcert", "CN=mydomain.com", tag1="a value", othertag="another value")

# importing a cert from a PFX file
vault$certificates$import("importedcert", "mycert.pfx")

# backup and restore a cert
bak <- vault$certificates$backup("mynewcert")
vault$certificates$delete("mynewcert", confirm=FALSE)
vault$certificates$restore(bak)

# set a contact
vault$certificates$set_contacts("username@mydomain.com")
vault$certificates$get_contacts()

# add an issuer and then obtain a cert
# this can take a long time, so set wait=FALSE to return immediately
vault$certificates$add_issuer("newissuer", provider="OneCert")
vault$certificates$create("issuedcert", "CN=mydomain.com",
    issuer=cert_issuer_properties("newissuer"),
    wait=FALSE)

}
}
\seealso{
\link{certificate}, \link{cert_key_properties}, \link{cert_x509_properties}, \link{cert_issuer_properties}, \link{vault_object_attrs}

\href{https://docs.microsoft.com/en-us/azure/key-vault/}{Azure Key Vault documentation},
\href{https://docs.microsoft.com/en-us/rest/api/keyvault}{Azure Key Vault API reference}
}
