% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_mallows.R
\name{compute_mallows}
\alias{compute_mallows}
\title{Preference Learning with the Mallows Rank Model}
\usage{
compute_mallows(
  rankings = NULL,
  preferences = NULL,
  obs_freq = NULL,
  metric = "footrule",
  error_model = NULL,
  n_clusters = 1L,
  clus_thin = 1L,
  nmc = 2000L,
  leap_size = max(1L, floor(n_items/5)),
  swap_leap = 1L,
  rho_init = NULL,
  rho_thinning = 1L,
  alpha_prop_sd = 0.1,
  alpha_init = 1,
  alpha_jump = 1L,
  lambda = 0.001,
  alpha_max = 1e+06,
  psi = 10L,
  include_wcd = (n_clusters > 1),
  save_aug = FALSE,
  aug_thinning = 1L,
  logz_estimate = NULL,
  verbose = FALSE,
  validate_rankings = TRUE,
  na_action = "augment",
  constraints = NULL,
  save_ind_clus = FALSE,
  seed = NULL,
  cl = NULL
)
}
\arguments{
\item{rankings}{A matrix of ranked items, of size \code{n_assessors x
n_items}. See \code{\link{create_ranking}} if you have an ordered set of
items that need to be converted to rankings. If \code{preferences} is
provided, \code{rankings} is an optional initial value of the rankings,
generated by \code{\link{generate_initial_ranking}}. If \code{rankings} has
column names, these are assumed to be the names of the items. \code{NA}
values in rankings are treated as missing data and automatically augmented;
to change this behavior, see the \code{na_action} argument.}

\item{preferences}{A dataframe with pairwise comparisons, with 3 columns,
named \code{assessor}, \code{bottom_item}, and \code{top_item}, and one row
for each stated preference. Given a set of pairwise preferences, generate a
transitive closure using \code{\link{generate_transitive_closure}}. This
will give \code{preferences} the class \code{"BayesMallowsTC"}. If
\code{preferences} is not of class \code{"BayesMallowsTC"},
\code{compute_mallows} will call \code{\link{generate_transitive_closure}}
on \code{preferences} before computations are done. In the current version,
the pairwise preferences are assumed to be mutually compatible.}

\item{obs_freq}{A vector of observation frequencies (weights) to apply do
each row in \code{rankings}. This can speed up computation if a large
number of assessors share the same rank pattern. Defaults to \code{NULL},
which means that each row of \code{rankings} is multiplied by 1. If
provided, \code{obs_freq} must have the same number of elements as there
are rows in \code{rankings}, and \code{rankings} cannot be \code{NULL}. See
\code{\link{obs_freq}} for more information and
\code{\link{rank_freq_distr}} for a convenience function for computing it.}

\item{metric}{A character string specifying the distance metric to use in the
Bayesian Mallows Model. Available options are \code{"footrule"},
\code{"spearman"}, \code{"cayley"}, \code{"hamming"}, \code{"kendall"}, and
\code{"ulam"}. The distance given by \code{metric} is also used to compute
within-cluster distances, when \code{include_wcd = TRUE}.}

\item{error_model}{Character string specifying which model to use for
inconsistent rankings. Defaults to \code{NULL}, which means that
inconsistent rankings are not allowed. At the moment, the only available
other option is \code{"bernoulli"}, which means that the Bernoulli error
model is used. See \insertCite{crispino2019;textual}{BayesMallows} for a
definition of the Bernoulli model.}

\item{n_clusters}{Integer specifying the number of clusters, i.e., the number
of mixture components to use. Defaults to \code{1L}, which means no
clustering is performed. See \code{\link{compute_mallows_mixtures}} for a
convenience function for computing several models with varying numbers of
mixtures.}

\item{clus_thin}{Integer specifying the thinning to be applied to cluster
assignments and cluster probabilities. Defaults to \code{1L}.}

\item{nmc}{Integer specifying the number of iteration of the
Metropolis-Hastings algorithm to run. Defaults to \code{2000L}. See
\code{\link{assess_convergence}} for tools to check convergence of the
Markov chain.}

\item{leap_size}{Integer specifying the step size of the leap-and-shift
proposal distribution. Defaults \code{floor(n_items / 5)}.}

\item{swap_leap}{Integer specifying the step size of the Swap proposal. Only
used when \code{error_model} is not \code{NULL}.}

\item{rho_init}{Numeric vector specifying the initial value of the latent
consensus ranking \eqn{\rho}. Defaults to NULL, which means that the
initial value is set randomly. If \code{rho_init} is provided when
\code{n_clusters > 1}, each mixture component \eqn{\rho_{c}} gets the same
initial value.}

\item{rho_thinning}{Integer specifying the thinning of \code{rho} to be
performed in the Metropolis- Hastings algorithm. Defaults to \code{1L}.
\code{compute_mallows} save every \code{rho_thinning}th value of
\eqn{\rho}.}

\item{alpha_prop_sd}{Numeric value specifying the standard deviation of the
lognormal proposal distribution used for \eqn{\alpha} in the
Metropolis-Hastings algorithm. Defaults to \code{0.1}.}

\item{alpha_init}{Numeric value specifying the initial value of the scale
parameter \eqn{\alpha}. Defaults to \code{1}. When \code{n_clusters > 1},
each mixture component \eqn{\alpha_{c}} gets the same initial value. When
chains are run in parallel, by providing an argument \code{cl = cl}, then
\code{alpha_init} can be a vector of of length \code{length(cl)}, each
element of which becomes an initial value for the given chain.}

\item{alpha_jump}{Integer specifying how many times to sample \eqn{\rho}
between each sampling of \eqn{\alpha}. In other words, how many times to
jump over \eqn{\alpha} while sampling \eqn{\rho}, and possibly other
parameters like augmented ranks \eqn{\tilde{R}} or cluster assignments
\eqn{z}. Setting \code{alpha_jump} to a high number can speed up
computation time, by reducing the number of times the partition function
for the Mallows model needs to be computed. Defaults to \code{1L}.}

\item{lambda}{Strictly positive numeric value specifying the rate parameter
of the truncated exponential prior distribution of \eqn{\alpha}. Defaults
to \code{0.1}. When \code{n_cluster > 1}, each mixture component
\eqn{\alpha_{c}} has the same prior distribution.}

\item{alpha_max}{Maximum value of \code{alpha} in the truncated exponential
prior distribution.}

\item{psi}{Integer specifying the concentration parameter \eqn{\psi} of the
Dirichlet prior distribution used for the cluster probabilities
\eqn{\tau_{1}, \tau_{2}, \dots, \tau_{C}}, where \eqn{C} is the value of
\code{n_clusters}. Defaults to \code{10L}. When \code{n_clusters = 1}, this
argument is not used.}

\item{include_wcd}{Logical indicating whether to store the within-cluster
distances computed during the Metropolis-Hastings algorithm. Defaults to
\code{TRUE} if \code{n_clusters > 1} and otherwise \code{FALSE}. Setting
\code{include_wcd = TRUE} is useful when deciding the number of mixture
components to include, and is required by \code{\link{plot_elbow}}.}

\item{save_aug}{Logical specifying whether or not to save the augmented
rankings every \code{aug_thinning}th iteration, for the case of missing
data or pairwise preferences. Defaults to \code{FALSE}. Saving augmented
data is useful for predicting the rankings each assessor would give to the
items not yet ranked, and is required by \code{\link{plot_top_k}}.}

\item{aug_thinning}{Integer specifying the thinning for saving augmented
data. Only used when \code{save_aug = TRUE}. Defaults to \code{1L}.}

\item{logz_estimate}{Estimate of the partition function, computed with
\code{\link{estimate_partition_function}}. Be aware that when using an
estimated partition function when \code{n_clusters > 1}, the partition
function should be estimated over the whole range of \eqn{\alpha} values
covered by the prior distribution for \eqn{\alpha} with high probability.
In the case that a cluster \eqn{\alpha_c} becomes empty during the
Metropolis-Hastings algorithm, the posterior of \eqn{\alpha_c} equals its
prior. For example, if the rate parameter of the exponential prior equals,
say \eqn{\lambda = 0.001}, there is about 37 \% (or exactly: \code{1 -
pexp(1000, 0.001)}) prior probability that \eqn{\alpha_c > 1000}. Hence
when \code{n_clusters > 1}, the estimated partition function should cover
this range, or \eqn{\lambda} should be increased.}

\item{verbose}{Logical specifying whether to print out the progress of the
Metropolis-Hastings algorithm. If \code{TRUE}, a notification is printed
every 1000th iteration. Defaults to \code{FALSE}.}

\item{validate_rankings}{Logical specifying whether the rankings provided (or
generated from \code{preferences}) should be validated. Defaults to
\code{TRUE}. Turning off this check will reduce computing time with a large
number of items or assessors.}

\item{na_action}{Character specifying how to deal with \code{NA} values in
the \code{rankings} matrix, if provided. Defaults to \code{"augment"},
which means that missing values are automatically filled in using the
Bayesian data augmentation scheme described in
\insertCite{vitelli2018;textual}{BayesMallows}. The other options for this
argument are \code{"fail"}, which means that an error message is printed
and the algorithm stops if there are \code{NA}s in \code{rankings}, and
\code{"omit"} which simply deletes rows with \code{NA}s in them.}

\item{constraints}{Optional constraint set returned from
\code{\link{generate_constraints}}. Defaults to \code{NULL}, which means
the the constraint set is computed internally. In repeated calls to
\code{compute_mallows}, with very large datasets, computing the constraint
set may be time consuming. In this case it can be beneficial to precompute
it and provide it as a separate argument.}

\item{save_ind_clus}{Whether or not to save the individual cluster
probabilities in each step. This results in csv files
\code{cluster_probs1.csv}, \code{cluster_probs2.csv}, ..., being saved in
the calling directory. This option may slow down the code considerably, but
is necessary for detecting label switching using Stephen's algorithm. See
\code{\link{label_switching}} for more information.}

\item{seed}{Optional integer to be used as random number seed.}

\item{cl}{Optional cluster.}
}
\value{
A list of class BayesMallows.
}
\description{
Compute the posterior distributions of the parameters of the
  Bayesian Mallows Rank Model, given rankings or preferences stated by a set
  of assessors.

  The \code{BayesMallows} package uses the following parametrization of the
  Mallows rank model \insertCite{mallows1957}{BayesMallows}:
  \deqn{p(r|\alpha,\rho) = (1/Z_{n}(\alpha)) \exp{-\alpha/n d(r,\rho)}} where
  \eqn{r} is a ranking, \eqn{\alpha} is a scale parameter, \eqn{\rho} is the
  latent consensus ranking, \eqn{Z_{n}(\alpha)} is the partition function
  (normalizing constant), and \eqn{d(r,\rho)} is a distance function
  measuring the distance between \eqn{r} and \eqn{\rho}. Note that some
  authors use a Mallows model without division by \eqn{n} in the exponent;
  this includes the \code{PerMallows} package, whose scale parameter
  \eqn{\theta} corresponds to \eqn{\alpha/n} in the \code{BayesMallows}
  package. We refer to \insertCite{vitelli2018}{BayesMallows} for further
  details of the Bayesian Mallows model.

  \code{compute_mallows} always returns posterior distributions of the latent
  consensus ranking \eqn{\rho} and the scale parameter \eqn{\alpha}. Several
  distance measures are supported, and the preferences can take the form of
  complete or incomplete rankings, as well as pairwise preferences.
  \code{compute_mallows} can also compute mixtures of Mallows models, for
  clustering of assessors with similar preferences.
}
\examples{
# ANALYSIS OF COMPLETE RANKINGS
# The example datasets potato_visual and potato_weighing contain complete
# rankings of 20 items, by 12 assessors. We first analyse these using the Mallows
# model:
model_fit <- compute_mallows(potato_visual)

# We study the trace plot of the parameters
assess_convergence(model_fit, parameter = "alpha")
\dontrun{assess_convergence(model_fit, parameter = "rho")}

# Based on these plots, we set burnin = 1000.
model_fit$burnin <- 1000
# Next, we use the generic plot function to study the posterior distributions
# of alpha and rho
plot(model_fit, parameter = "alpha")
\dontrun{plot(model_fit, parameter = "rho", items = 10:15)}

# We can also compute the CP consensus posterior ranking
compute_consensus(model_fit, type = "CP")

# And we can compute the posterior intervals:
# First we compute the interval for alpha
compute_posterior_intervals(model_fit, parameter = "alpha")
# Then we compute the interval for all the items
\dontrun{compute_posterior_intervals(model_fit, parameter = "rho")}

# ANALYSIS OF PAIRWISE PREFERENCES
\dontrun{
  # The example dataset beach_preferences contains pairwise
  # preferences between beaches stated by 60 assessors. There
  # is a total of 15 beaches in the dataset.
  # In order to use it, we first generate all the orderings
  # implied by the pairwise preferences.
  beach_tc <- generate_transitive_closure(beach_preferences)
  # We also generate an inital rankings
  beach_rankings <- generate_initial_ranking(beach_tc, n_items = 15)
  # We then run the Bayesian Mallows rank model
  # We save the augmented data for diagnostics purposes.
  model_fit <- compute_mallows(rankings = beach_rankings,
                               preferences = beach_tc,
                               save_aug = TRUE,
                               verbose = TRUE)
  # We can assess the convergence of the scale parameter
  assess_convergence(model_fit)
  # We can assess the convergence of latent rankings. Here we
  # show beaches 1-5.
  assess_convergence(model_fit, parameter = "rho", items = 1:5)
  # We can also look at the convergence of the augmented rankings for
  # each assessor.
  assess_convergence(model_fit, parameter = "Rtilde",
                     items = c(2, 4), assessors = c(1, 2))
  # Notice how, for assessor 1, the lines cross each other, while
  # beach 2 consistently has a higher rank value (lower preference) for
  # assessor 2. We can see why by looking at the implied orderings in
  # beach_tc
  subset(beach_tc, assessor \%in\% c(1, 2) &
           bottom_item \%in\% c(2, 4) & top_item \%in\% c(2, 4))
  # Assessor 1 has no implied ordering between beach 2 and beach 4,
  # while assessor 2 has the implied ordering that beach 4 is preferred
  # to beach 2. This is reflected in the trace plots.
}

# CLUSTERING OF ASSESSORS WITH SIMILAR PREFERENCES
\dontrun{
  # The example dataset sushi_rankings contains 5000 complete
  # rankings of 10 types of sushi
  # We start with computing a 3-cluster solution
  model_fit <- compute_mallows(sushi_rankings, n_clusters = 3,
                               nmc = 10000, verbose = TRUE)
  # We then assess convergence of the scale parameter alpha
  assess_convergence(model_fit)
  # Next, we assess convergence of the cluster probabilities
  assess_convergence(model_fit, parameter = "cluster_probs")
  # Based on this, we set burnin = 1000
  # We now plot the posterior density of the scale parameters alpha in
  # each mixture:
  model_fit$burnin <- 1000
  plot(model_fit, parameter = "alpha")
  # We can also compute the posterior density of the cluster probabilities
  plot(model_fit, parameter = "cluster_probs")
  # We can also plot the posterior cluster assignment. In this case,
  # the assessors are sorted according to their maximum a posteriori cluster estimate.
  plot(model_fit, parameter = "cluster_assignment")
  # We can also assign each assessor to a cluster
  cluster_assignments <- assign_cluster(model_fit, soft = FALSE)
  }

# DETERMINING THE NUMBER OF CLUSTERS
\dontrun{
  # Continuing with the sushi data, we can determine the number of cluster
  # Let us look at any number of clusters from 1 to 10
  # We use the convenience function compute_mallows_mixtures
  n_clusters <- seq(from = 1, to = 10)
  models <- compute_mallows_mixtures(n_clusters = n_clusters, rankings = sushi_rankings,
                                     nmc = 6000, alpha_jump = 10, include_wcd = TRUE)
  # models is a list in which each element is an object of class BayesMallows,
  # returned from compute_mallows
  # We can create an elbow plot
  plot_elbow(models, burnin = 1000)
  # We then select the number of cluster at a point where this plot has
  # an "elbow", e.g., at 6 clusters.
}

# SPEEDING UP COMPUTION WITH OBSERVATION FREQUENCIES
# With a large number of assessors taking on a relatively low number of unique rankings,
# the obs_freq argument allows providing a rankings matrix with the unique set of rankings,
# and the obs_freq vector giving the number of assessors with each ranking.
# This is illustrated here for the potato_visual dataset
#
# assume each row of potato_visual corresponds to between 1 and 5 assessors, as
# given by the obs_freq vector
set.seed(1234)
obs_freq <- sample.int(n = 5, size = nrow(potato_visual), replace = TRUE)
m <- compute_mallows(rankings = potato_visual, obs_freq = obs_freq)

# See the separate help page for more examples, with the following code
help("obs_freq")
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{compute_mallows_mixtures}} for a function that computes
  separate Mallows models for varying numbers of clusters.

Other modeling: 
\code{\link{compute_mallows_mixtures}()},
\code{\link{smc_mallows_new_item_rank}()},
\code{\link{smc_mallows_new_users}()}
}
\concept{modeling}
