#' @title IIS
#' @description
#' A function is used to calculates and identify the key nodes,the formula of IIS is:
#'               IIS= nDFS x DFS + nDS x DS + nES x ES + nAS x AS;
#' nDFS,nDS,nES,nAS indict the weight value of DFS,DS,ES and AS, and nDFS + nDS + nES + nAS = 1;
#' DFS means Difference Score;
#' DS means Degree Score;
#' ES means Edge Score;
#' AS means Abundance Score.
#'
#' @param microApath A data frame including microbes and predicted KEGG pathways.
#' @param metaApath A data frame including metabolites with compound ID and predicted KEGG pathways from Meta2pathway function.
#' @param conf  A data frame including confounders. Default value is NULL.
#' @param groupInfo A data frame including sample grouping information
#' @param nDFS A value indicated the weight of difference Score in the IIS formula. Default value is 0.4.
#' @param nDS A value indicated the weight of degree score in the IIS formula. Default value is 0.3.
#' @param nES A value indicated the weight of edge score in the IIS formula. Default value is 0.2.
#' @param nAS A value indicated the weight of abundance score in the IIS formula. Default value is 0.1.
#' @param NS A threshold value of IIS to identify the key nodes. Default value is 2.5.
#' @param r A threshold value of correlation coefficient to construct the network. Default value is 0.5.
#' @param p_adjust A threshold value of correlation adjust p value to construct the network. Default  value is 0.05.
#' @return IIS_res A list including IIS and other sub-score results.
#' @export IIS
#' @importFrom igraph graph_from_data_frame
#' @importFrom dplyr group_by
#' @importFrom dplyr summarise
#' @import ggrepel
#' @import ppcor
#' @importFrom utils write.csv
#' @importFrom stats cor.test
#' @importFrom stats p.adjust
#' @importFrom stats wilcox.test
#' @examples
#' IIScore <- IIS(microApath = micro.eg, metaApath = metabo.eg,
#'                conf = confounder.eg, groupInfo = groupInfo.eg)
IIS<-function(microApath, metaApath, conf = NULL, groupInfo = NULL, nDFS = 0.4,
              nDS = 0.3, nES = 0.2, nAS = 0.1, NS = 2.5, r = 0.5, p_adjust = 0.05){
  from = NULL;to = NULL
  A=microApath
  B=metaApath
  mmApath=get0("sysdata", envir = asNamespace("BiOFI"))$mm4path

  MMData1=data.frame(metaandpath=colnames(B),group=0)
  for(i in 1:length(colnames(B))){

    if(MMData1$metaandpath[i]%in% unique(mmApath$meta_path)){
      MMData1$group[i]='met_path'
    }else{
      MMData1$group[i]='metabolite'
    }
  }


  if(!'met_path'%in% unique(MMData1$group)){
    stop('please tranfer metabolites to pathway using Meta2pathway function')
  }


  MMData2=data.frame(microandpath=colnames(A),group=0)
  for(i in 1:length(colnames(A))){
    if(substr(MMData2$microandpath[i],1,3)=='k__'){
      MMData2$group[i]='microbe'
    }else{MMData2$group[i]='mic_path'}

  }

  if(length(rownames(groupInfo))==0){
    stop('Group information is lack, please check!')

  }

  colnames(B)=paste(colnames(B),'-M',sep=" ")

  mic_meta_cor = matrix(NA, nrow = ncol (A), ncol = ncol (B))
  mic_intra_cor=matrix(NA, nrow = ncol (A), ncol = ncol (A))
  met_intra_cor=matrix(NA, nrow = ncol (B), ncol = ncol (B))

  rownames (mic_meta_cor) = colnames (A)
  colnames (mic_meta_cor) = colnames (B)
  rownames (mic_intra_cor) = colnames (A)
  colnames (mic_intra_cor) = colnames (A)
  rownames (met_intra_cor) = colnames (B)
  colnames (met_intra_cor) = colnames (B)

  mic_meta_cor_p = mic_meta_cor
  mic_meta_cor_p.adjust = mic_meta_cor

  mic_intra_cor_p=mic_intra_cor
  mic_intra_cor_p.adjust=mic_intra_cor

  met_intra_cor_p=met_intra_cor
  met_intra_cor_p.adjust=met_intra_cor

  for (m in colnames (B)) {

    if(length(rownames(conf))==0){
      mic_meta_cor[,m] = apply (A, MARGIN = 2, FUN = function (x)
        cor.test (as.numeric(x),as.numeric(B[,m]), method = "spearman",use = "pairwise.complete.obs")$estimate)

      mic_meta_cor_p[,m] = apply (A, MARGIN = 2, FUN = function (x)
        cor.test (as.numeric(x),as.numeric(B[,m]), method ="spearman", use = "pairwise.complete.obs")$p.value)

    }else{
      tryCatch({
        mic_meta_cor[,m] = apply (A, MARGIN = 2, FUN = function (x)
          ppcor::pcor.test (as.numeric(x),as.numeric(B[,m]), conf, method = "spearman")$estimate)

        mic_meta_cor_p[,m] = apply (A, MARGIN = 2, FUN = function (x)
          ppcor::pcor.test (as.numeric(x),as.numeric(B[,m]), conf, method ="spearman")$p.value)
      }, error=function(e) {
      })

    }
  }

  mic_meta_cor_p.adjust = apply(mic_meta_cor_p, 2, FUN = function(x){
    p.adjust(x,method = "BH")})

  del_var <- c()
  for (m in 1:ncol (A)) {
    del_var <- append(del_var,m)
    if(length(rownames(conf))==0){
      temp_cor = apply(as.data.frame(A[,-del_var]), MARGIN = 2,FUN = function (x)
        cor.test (as.numeric(x), as.numeric(A[,m]), method = "spearman", use = "pairwise.complete.obs")$estimate)
      mic_intra_cor [ , m] = c(rep(NA,m),temp_cor)

      temp_p = apply(as.data.frame(A[,-del_var]), MARGIN = 2, FUN = function (x)
        cor.test (as.numeric(x), as.numeric(A[,m]), method ="spearman", use = "pairwise.complete.obs")$p.value)
      mic_intra_cor_p[,m] = c(rep(NA,m),temp_p)
    }else{
      tryCatch({
        if(m!=ncol (A)){
          temp_cor = apply (as.data.frame(A[,-del_var]), MARGIN = 2, FUN = function (x)
            ppcor::pcor.test (as.numeric(x),as.numeric(A[,m]), conf, method = "spearman")$estimate)
          mic_intra_cor[,m] = c(rep(NA,m),temp_cor)

          temp_p = apply (as.data.frame(A[,-del_var]), MARGIN = 2, FUN = function (x)
            ppcor::pcor.test (as.numeric(x),as.numeric(A[,m]), conf, method ="spearman")$p.value)
          mic_intra_cor_p[,m] =  c(rep(NA,m),temp_p)
        }else{
          mic_intra_cor[,m] = rep(NA,m)
          mic_intra_cor_p[,m] =  rep(NA,m)
        }


      }, error=function(e) {
      })
    }
  }


  mic_intra_cor_p.adjust = apply(mic_intra_cor_p, 2, FUN = function(x){
    p.adjust(x,method = "BH")})

  del_var <- c()
  for (m in 1:ncol (B)) {
    del_var <- append(del_var,m)
    if(length(rownames(conf))==0){
      temp_cor = apply(as.data.frame(B[,-del_var]), MARGIN = 2, FUN = function (x)
        cor.test (as.numeric(x),as.numeric(B[,m]), method = "spearman", use = "pairwise.complete.obs")$estimate)
      met_intra_cor [ , m] =  c(rep(NA,m),temp_cor)

      temp_p = apply(as.data.frame(B[,-del_var]), MARGIN = 2, FUN = function (x)
        cor.test (as.numeric(x), as.numeric(B[,m]), method ="spearman", use = "pairwise.complete.obs")$p.value)
      met_intra_cor_p[,m] =  c(rep(NA,m),temp_p)
    }else{
      tryCatch({
        if(m!=ncol (B)){
          temp_cor = apply (as.data.frame(B[,-del_var]), MARGIN = 2, FUN = function (x)
            ppcor::pcor.test (as.numeric(x),as.numeric(B[,m]), conf, method = "spearman")$estimate)
          met_intra_cor[,m] =  c(rep(NA,m),temp_cor)

          temp_p = apply (as.data.frame(B[,-del_var]), MARGIN = 2, FUN = function (x)
            ppcor::pcor.test (as.numeric(x),as.numeric(B[,m]), conf, method ="spearman")$p.value)
          met_intra_cor_p[,m] =  c(rep(NA,m),temp_p)
        }else{
          met_intra_cor[,m] = rep(NA,m)
          met_intra_cor_p[,m] =  rep(NA,m)
        }

      }, error=function(e) {
      })
    }
  }


  met_intra_cor_p.adjust = apply(met_intra_cor_p, 2, FUN = function(x){
    p.adjust(x,method = "BH")
  })

  microbes<-matrix(MMData2[MMData2$group=='microbe',]$microandpath)
  metabolites<-matrix(paste(MMData1[MMData1$group=='metabolite',]$metaandpath,'-M',sep=" "))
  mic_path<-matrix(MMData2[MMData2$group=='mic_path',]$microandpath)
  met_path<-matrix(paste(MMData1[MMData1$group=='met_path',]$metaandpath,'-M',sep=" "))

  CorrDF <- function(cormat,pmat) {
    ut = matrix (TRUE, nrow = nrow (cormat), ncol = ncol (pmat))
    data.frame(
      from = rownames(cormat)[row(cormat)[ut]],
      to = colnames(cormat)[col(cormat)[ut]],
      r = (cormat)[ut],
      p.adjust = pmat[ut]
    )
  }

  mic_meta_cor_df <- CorrDF(mic_meta_cor,mic_meta_cor_p.adjust)
  mic_meta_cor_filtered <- mic_meta_cor_df[which(abs(mic_meta_cor_df$r) > r & mic_meta_cor_df$p.adjust < p_adjust),]


  met_intra_cor_df <- CorrDF(met_intra_cor,met_intra_cor_p.adjust)
  met_intra_cor_filtered <- met_intra_cor_df[which(abs(met_intra_cor_df$r) > r & met_intra_cor_df$p.adjust < p_adjust),]
  mic_intra_cor_df <- CorrDF(mic_intra_cor,mic_intra_cor_p.adjust)
  mic_intra_cor_filtered <- mic_intra_cor_df[which(abs(mic_intra_cor_df$r) > r & mic_intra_cor_df$p.adjust < p_adjust),]

  mic_meta_cor_filtered <- rbind(mic_meta_cor_filtered,met_intra_cor_filtered)
  mic_meta_cor_filtered <- rbind(mic_meta_cor_filtered,mic_intra_cor_filtered)

  centrality = "degree"

  nodeattrib <- data.frame(nodes = union(mic_meta_cor_filtered$from,mic_meta_cor_filtered$to))

  mic_meta_cor_filtered <- mic_meta_cor_filtered


  nodeattrib$group <- 0
  for (i in as.character(nodeattrib$nodes)){
    if (i %in% microbes[,1] == TRUE){
      nodeattrib[nodeattrib$nodes == i,"group"] <- "Microbe"
    }else if(i %in% mic_path[,1] == TRUE){
      nodeattrib[nodeattrib$nodes == i,"group"] <- "Microbiome_pathway"
    }else if(i %in% metabolites[,1] == TRUE){
      nodeattrib[nodeattrib$nodes == i,"group"] <- "Metabolite"
    }else if(i %in% met_path[,1]==TRUE){
      nodeattrib[nodeattrib$nodes == i,"group"] <- "Metabolome_pathway"
    }
  }

  rownames(nodeattrib) <- nodeattrib$nodes
  co_net <- graph_from_data_frame(mic_meta_cor_filtered,directed  = F, vertices = nodeattrib)
  net_nodes <- rownames(nodeattrib[nodeattrib$group %in% co_net,])


  nodes_size = centr_degree(co_net)$res

  DS <- as.data.frame(cbind(names(V(co_net)),nodes_size))
  colnames(DS) <- c("nodes","size")
  DS$size <- as.numeric(DS$size)
  DS <- DS[order(DS$size,decreasing = T),]
  DS$score <-  seq(nrow(DS),1,-1)
  DS$group <- 0
  for (i in DS$nodes){
    if (i %in% microbes[,1]== TRUE){
      DS[DS$nodes == i,"group"] <- "Microbe"
    }else if(i %in% mic_path[,1] == TRUE){
      DS[DS$nodes == i,"group"] <- "Microbiome_pathway"
    }else if(i %in% metabolites[,1] == TRUE){
      DS[DS$nodes == i,"group"] <- "Metabolite"
    }else{
      DS[DS$nodes == i,"group"] <- "Metabolome_pathway"
    }
  }
  DS[which(DS$group=="Metabolome_pathway"),]$nodes=gsub(" -M",'',DS[which(DS$group=="Metabolome_pathway"),]$nodes)
  DS[which(DS$group=="Metabolite"),]$nodes=gsub(" -M",'',DS[which(DS$group=="Metabolite"),]$nodes)


  DS_metabolites <- DS[which(DS$group == "Metabolite"),]
  DS_met_pathway <- DS[which(DS$group == "Metabolome_pathway"),]
  DS_microbes <- DS[which(DS$group == "Microbe"),]
  DS_mic_pathway <- DS[which(DS$group == "Microbiome_pathway"),]

  DS_metabolites$score <-  seq(nrow(DS_metabolites),1,-1)
  DS_metabolites$score <- (DS_metabolites$score - 1)/(nrow(DS_metabolites) - 1) * 2 + 1
  DS_met_pathway$score <-  seq(nrow(DS_met_pathway),1,-1)
  DS_met_pathway$score <- (DS_met_pathway$score - 1)/(nrow(DS_met_pathway) - 1) * 2 + 1
  DS_microbes$score <-  seq(nrow(DS_microbes),1,-1)
  DS_microbes$score <- (DS_microbes$score - 1)/(nrow(DS_microbes) - 1) * 2 + 1
  DS_mic_pathway$score <-  seq(nrow(DS_mic_pathway),1,-1)
  DS_mic_pathway$score <- (DS_mic_pathway$score - 1)/(nrow(DS_mic_pathway) - 1) * 2 + 1

  DS<- rbind(DS_metabolites,DS_met_pathway)
  DS<-rbind(DS,DS_microbes)
  DS<-rbind(DS,DS_mic_pathway)

  # groupInfo value ------------------------------------------------------------

  group1=groupInfo[which(groupInfo$Group==unique(groupInfo$Group)[1]),]$SampleID
  group2=groupInfo[which(groupInfo$Group==unique(groupInfo$Group)[2]),]$SampleID

  wil_p.value <- c()
  for (i in colnames(B)) {
    wil_p.value <- append(wil_p.value, wilcox.test(B[group1,][,i],B[group2,][,i])$p.value)
  }

  met_log2fc <- c()

  for (i in colnames(B)) {
    met_log2fc <- append(met_log2fc, log2(mean(B[group2,][,i])/mean(B[group1,][,i])))
  }


  diff_met <- data.frame(nodes = colnames(B),log2FC=met_log2fc,p.value=wil_p.value)
  diff_met<-diff_met[which(diff_met$p.value<0.05),]

  wil_p.value <- c()
  for (i in colnames(A)) {
    wil_p.value <- append(wil_p.value, wilcox.test(as.numeric(A[group1,][,i],A[group2,][,i]))$p.value)
  }
  mic_log2fc <- c()
  for (i in colnames(A)) {
    mic_log2fc <- append(mic_log2fc, log2(mean(A[group2,][,i])/mean(A[group1,][,i])))
  }

  diff_mic<- data.frame(nodes = colnames(A),log2FC=mic_log2fc,p.value=wil_p.value)
  diff_mic<-diff_mic[which(diff_mic$p.value<0.05),]

  diff_mic_met<-rbind(diff_mic,diff_met)
  diff_df<-as.data.frame(diff_mic_met$nodes)
  colnames(diff_df)<-"Diff"

  mic_meta_cor_filtered$ES <- 1
  dfii <- unique(diff_df$Diff)
  # n1 <- n2 <- 0
  for (i in 1:nrow(mic_meta_cor_filtered)) {
    if((mic_meta_cor_filtered$from[i] %in% dfii) & (mic_meta_cor_filtered$to[i] %in% dfii)){
      mic_meta_cor_filtered$ES[i] <- 3
      # n1 <- n1 + 1
    }else{
      # n2 <- n2 + 1
      mic_meta_cor_filtered$ES[i] <- 1
    }
  }


  from_es <- mic_meta_cor_filtered %>%
    group_by(from) %>%
    summarise("ES" = sum(ES))
  colnames(from_es) <- c("nodes","ES")

  to_es <- mic_meta_cor_filtered %>%
    group_by(to) %>%
    summarise("ES" = sum(ES))
  colnames(to_es) <- c("nodes","ES")

  mic_met_es <- merge(from_es,to_es,by = "nodes",all = T)
  mic_met_es[is.na(mic_met_es)] <- 0
  mic_met_es$ES <- mic_met_es$ES.x + mic_met_es$ES.y
  mic_met_es <- mic_met_es[,-(2:3)]
  ES <- mic_met_es
  ES <- ES[order(ES$ES,decreasing = T),]
  ES$score <-  seq(nrow(ES),1,-1)
  ES$group <- 0
  ES <- as.data.frame(ES)
  for (i in ES$nodes){
    if (i %in% microbes[,1] == TRUE){
      ES[ES$nodes == i,"group"] <- "Microbe"
    }else if(i %in% mic_path[,1] == TRUE){
      ES[ES$nodes == i,"group"] <- "Microbiome_pathway"
    }else if(i %in% metabolites[,1] == TRUE){
      ES[ES$nodes == i,"group"] <- "Metabolite"
    }else{
      ES[ES$nodes == i,"group"] <- "Metabolome_pathway"
    }
  }
  ES[which(ES$group=="Metabolome_pathway"),]$nodes=gsub(" -M",'',ES[which(ES$group=="Metabolome_pathway"),]$nodes)
  ES[which(ES$group=="Metabolite"),]$nodes=gsub(" -M",'',ES[which(ES$group=="Metabolite"),]$nodes)


  ES_metabolites <- ES[which(ES$group == "Metabolite"),]
  ES_met_pathway <- ES[which(ES$group == "Metabolome_pathway"),]
  ES_microbes <- ES[which(ES$group == "Microbe"),]
  ES_mic_pathway <- ES[which(ES$group == "Microbiome_pathway"),]

  ES_metabolites$score <-  seq(nrow(ES_metabolites),1,-1)
  ES_metabolites$score <- (ES_metabolites$score - 1)/(nrow(ES_metabolites) - 1) * 2 + 1
  ES_met_pathway$score <-  seq(nrow(ES_met_pathway),1,-1)
  ES_met_pathway$score <- (ES_met_pathway$score - 1)/(nrow(ES_met_pathway) - 1) * 2 + 1
  ES_microbes$score <-  seq(nrow(ES_microbes),1,-1)
  ES_microbes$score <- (ES_microbes$score - 1)/(nrow(ES_microbes) - 1) * 2 + 1
  ES_mic_pathway$score <-  seq(nrow(ES_mic_pathway),1,-1)
  ES_mic_pathway$score <- (ES_mic_pathway$score - 1)/(nrow(ES_mic_pathway) - 1) * 2 + 1

  ES<-rbind(ES_metabolites,ES_met_pathway)
  ES<-rbind(ES,ES_microbes)
  ES<-rbind(ES,ES_mic_pathway)

  colnames(B)=gsub(" -M",'',colnames(B))
  metabolites_abun <- B[,which(colnames(B) %in% DS_metabolites$nodes)]
  metabolites_pathway_abun <- B[,which(colnames(B) %in% DS_met_pathway$nodes )]



  AS_metabolites <- apply(metabolites_abun,2,mean)
  AS_metabolites <- data.frame(nodes = colnames(metabolites_abun), abundance = AS_metabolites)
  AS_met_pathway <- apply(metabolites_pathway_abun,2,mean)
  AS_met_pathway <- data.frame(nodes = colnames(metabolites_pathway_abun), abundance = AS_met_pathway)



  AS_metabolites <- AS_metabolites[order(AS_metabolites$abundance,decreasing = T),]
  AS_metabolites$score <-  seq(nrow(AS_metabolites),1,-1)
  AS_metabolites$score <- (AS_metabolites$score - 1)/(nrow(AS_metabolites) - 1) * 2 + 1
  AS_metabolites$group <- "Metabolite"

  AS_met_pathway <- AS_met_pathway[order(AS_met_pathway$abundance,decreasing = T),]
  AS_met_pathway$score <-  seq(nrow(AS_met_pathway),1,-1)
  AS_met_pathway$score <- (AS_met_pathway$score - 1)/(nrow(AS_met_pathway) - 1) * 2 + 1
  AS_met_pathway$group <- "MetaBolome_pathway"

  microbes_abun <- A[,which(DS_microbes$nodes%in% colnames(A))]

  if(length(rownames(DS_microbes))>1){
    microbes_abun <- A[,which(colnames(A) %in% DS_microbes$nodes)]
  }else{
    microbes_abun=as.data.frame(A[,which(colnames(A) %in% DS_microbes$nodes)])
    rownames(microbes_abun)=rownames(A)
    colnames(microbes_abun)=DS_microbes$nodes
  }

  if(length(rownames(DS_mic_pathway$nodes))>1){
    microbes_pathway_abun <- A[,which(colnames(A) %in% DS_mic_pathway$nodes)]
  }else{
    microbes_pathway_abun=as.data.frame(A[,which(colnames(A) %in% DS_mic_pathway$nodes)])
    rownames(microbes_pathway_abun)=rownames(A)
    colnames(microbes_pathway_abun)=DS_mic_pathway$nodes
  }

  AS_microbes <- apply(microbes_abun,2,mean)
  AS_microbes <- data.frame(nodes = colnames(microbes_abun), abundance = AS_microbes)
  AS_mic_pathway <- apply(microbes_pathway_abun,2,mean)
  AS_mic_pathway <- data.frame(nodes = colnames(microbes_pathway_abun), abundance = AS_mic_pathway)


  AS_microbes <- AS_microbes[order(AS_microbes$abundance,decreasing = T),]
  AS_microbes$score <-  seq(nrow(AS_microbes),1,-1)
  AS_microbes$score <- (AS_microbes$score - 1)/(nrow(AS_microbes) - 1) * 2 + 1
  AS_microbes$group <- "Microbe"

  AS_mic_pathway <- AS_mic_pathway[order(AS_mic_pathway$abundance,decreasing = T),]
  AS_mic_pathway$score <-  seq(nrow(AS_mic_pathway),1,-1)

  AS_mic_pathway$score <- (AS_mic_pathway$score - 1)/(nrow(AS_mic_pathway) - 1) * 2 + 1
  AS_mic_pathway$group <- "Microbiome_pathway"

  AS<-rbind(AS_metabolites,AS_met_pathway)
  AS<-rbind(AS,AS_microbes)
  AS<-rbind(AS,AS_mic_pathway)

  #####DFS
  diff_met$DFS <- sqrt((log10(diff_met$p.value))^2 + (diff_met$log2FC)^2)
  diff_mic$DFS <- sqrt((log10(diff_mic$p.value))^2 + (diff_mic$log2FC)^2)
  met_diff <- data.frame(nodes = diff_met$nodes, DFS = diff_met$DFS)
  mic_diff <- data.frame(nodes = diff_mic$nodes, DFS = diff_mic$DFS)

  met_diff$nodes=gsub(' -M','',met_diff$nodes)
  DFS_metabolites <- unique(merge(AS_metabolites,met_diff,by="nodes"))
  DFS_metabolites$abundance <- DFS_metabolites$DFS
  DFS_metabolites <- DFS_metabolites[,-length(DFS_metabolites)]
  colnames(DFS_metabolites)[2] <- "DFS"
  DFS_metabolites <- DFS_metabolites[order(DFS_metabolites$DFS,decreasing = T),]
  DFS_metabolites$score <-  seq(nrow(DFS_metabolites),1,-1)
  DFS_metabolites$score <- (DFS_metabolites$score - 1)/(nrow(DFS_metabolites) - 1) * 2 + 1
  DFS_metabolites$group <- "Metabolite"

  DFS_met_pathway <- unique(merge(AS_met_pathway,met_diff,by="nodes"))
  DFS_met_pathway$abundance <- DFS_met_pathway$DFS
  DFS_met_pathway <- DFS_met_pathway[,-length(DFS_met_pathway)]
  colnames(DFS_met_pathway)[2] <- "DFS"
  DFS_met_pathway <- DFS_met_pathway[order(DFS_met_pathway$DFS,decreasing = T),]
  DFS_met_pathway$score <-  seq(nrow(DFS_met_pathway),1,-1)
  DFS_met_pathway$score <- (DFS_met_pathway$score - 1)/(nrow(DFS_met_pathway) - 1) * 2 + 1
  DFS_met_pathway$group <- "Metabolome_pathway"



  DFS_microbes <- unique(merge(AS_microbes,mic_diff,by="nodes"))
  DFS_microbes$abundance <- DFS_microbes$DFS
  DFS_microbes <- DFS_microbes[,-length(DFS_microbes)]
  colnames(DFS_microbes)[2] <- "DFS"
  DFS_microbes <- DFS_microbes[order(DFS_microbes$DFS,decreasing = T),]


  flag_inf <- is.finite(DFS_microbes$DFS)
  flag_inf2 <- is.infinite(DFS_microbes$DFS)

  DFS_microbes[flag_inf,]$score <- seq(nrow(DFS_microbes[flag_inf,]),1,-1)



  DFS_microbes[flag_inf,]$score <- (DFS_microbes[flag_inf,]$score - 1)/(nrow(DFS_microbes[flag_inf,]) - 1) * 2 + 1

  DFS_microbes$group <- "Microbe"

  DFS_mic_pathway <- unique(merge(AS_mic_pathway,mic_diff,by="nodes"))
  DFS_mic_pathway$abundance <- DFS_mic_pathway$DFS
  DFS_mic_pathway <- DFS_mic_pathway[,-length(DFS_mic_pathway)]
  colnames(DFS_mic_pathway)[2] <- "DFS"
  DFS_mic_pathway <- DFS_mic_pathway[order(DFS_mic_pathway$DFS,decreasing = T),]

  flag_inf <- is.finite(DFS_mic_pathway$DFS)
  flag_inf2 <- is.infinite(DFS_mic_pathway$DFS)
  DFS_mic_pathway[flag_inf,]$score <- seq(nrow(DFS_mic_pathway[flag_inf,]),1,-1)
  DFS_mic_pathway[flag_inf,]$score <- (DFS_mic_pathway[flag_inf,]$score - 1)/(nrow(DFS_mic_pathway[flag_inf,]) - 1) * 2 + 1


  DFS_mic_pathway$group <- "Microbiome_pathway"
  DFS<-rbind(DFS_metabolites,DFS_met_pathway)
  DFS<-rbind(DFS,DFS_microbes)
  DFS<-rbind(DFS,DFS_mic_pathway)

  NS_metabolites <- data.frame(nodes = DS_metabolites$nodes,ns = (nDFS*DFS_metabolites$score + nDS*DS_metabolites$score + nES*ES_metabolites$score + nAS*AS_metabolites$score),group = DS_metabolites$group)
  NS_met_pathway <- data.frame(nodes = DS_met_pathway$nodes,ns = (nDFS*DFS_met_pathway$score + nDS*DS_met_pathway$score + nES*ES_met_pathway$score + nAS*AS_met_pathway$score),group = DS_met_pathway$group)


  NS_microbes <- data.frame(nodes = DS_microbes$nodes,ns = (nDFS*DFS_microbes$score + nDS*DS_microbes$score + nES*ES_microbes$score + nAS*AS_microbes$score),group = DS_microbes$group)

  NS_mic_pathway <- data.frame(nodes = DS_mic_pathway$nodes,ns = (nDFS*DFS_mic_pathway$score + nDS*DS_mic_pathway$score + nES*ES_mic_pathway$score + nAS*AS_mic_pathway$score),group = DS_mic_pathway$group)



  NS_metabolites <- NS_metabolites[order(NS_metabolites$ns,decreasing = T),]
  NS_met_pathway <- NS_met_pathway[order(NS_met_pathway$ns,decreasing = T),]
  NS_microbes <- NS_microbes[order(NS_microbes$ns,decreasing = T),]
  NS_mic_pathway <- NS_mic_pathway[order(NS_mic_pathway$ns,decreasing = T),]

  for (i in 1:nrow(NS_mic_pathway)) {
    NS_mic_pathway[i,1] <- strsplit(NS_mic_pathway[i,1], " \\[")[[1]][1]
  }

  Node_Score<-rbind(NS_metabolites,NS_met_pathway)
  Node_Score<-rbind(Node_Score,NS_microbes)
  Node_Score<-rbind(Node_Score,NS_mic_pathway)
  NScore=Node_Score[Node_Score$ns>NS,]


  mic_meta_cor_filtered$from=gsub(' -M','',mic_meta_cor_filtered$from)
  mic_meta_cor_filtered$to=gsub(' -M','',mic_meta_cor_filtered$to)

  colnames(mic_meta_cor)=gsub(' -M','',colnames(mic_meta_cor))
  colnames(mic_meta_cor_p.adjust)=gsub(' -M','',colnames(mic_meta_cor_p.adjust))

  mic_meta_cor=as.data.frame(mic_meta_cor)
  mic_meta_cor_p.adjust=as.data.frame(mic_meta_cor_p.adjust)

  IIS_res=list("Correlation analysis results"=mic_meta_cor_filtered,'mic_meta_cor'=mic_meta_cor,'mic_meta_cor_p.adjust'=mic_meta_cor_p.adjust,"Dgree_Score"=DS,"Abundance_Score"=AS,"Edge_Score"=ES,"Diff_Score"=DFS,"NScore"=NScore,'Node_Score'=Node_Score)

  return(IIS_res)
}


