\name{ensemble.envirem.masterstack}
\alias{ensemble.envirem.masterstack}
\alias{ensemble.envirem.solradstack}
\alias{ensemble.envirem.run}

\title{
Calculate bioclimatic variables via the \code{envirem} package for data.frames.
}

\description{
Function \code{\link[envirem]{layerCreation}} uses RasterStack (\code{\link[raster]{stack}}) objects as input and also generates outputs in the same format. The functions described here can be used to generate the bioclimatic variables for \code{data.frames} while using \code{envirem} functions internally. This feature can be useful in situations where models are calibrated with higher resolution data, but where maps will only be generated in lower resolutions, thus avoiding the need to generate the higher resolution envirem layers first.  
}

\usage{
ensemble.envirem.masterstack(
    x,
    precipstack, 
    tmaxstack, tminstack, 
    tmeanstack=NULL)

ensemble.envirem.solradstack(
    x, solrad)

ensemble.envirem.run(
    masterstack, solradstack,
    var="all", ...)
}

\arguments{
  \item{x}{ Point locations provided in 2-column (eg, LON-LAT) format.}
  \item{precipstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly precipitation data.  }
  \item{tmaxstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly maximum temperature data.  }
  \item{tminstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly minimum temperature data.  }
  \item{tmeanstack}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly average temperature data.  }
  \item{solrad}{ RasterStack object (\code{\link[raster]{stack}}) or SpatRaster object (\code{\link[terra]{rast}}) containing monthly extrasolar radiation data.  }  
  \item{masterstack}{ RasterStack object (\code{\link[raster]{stack}}) expected to have been created via \code{\link[BiodiversityR]{ensemble.envirem.masterstack}}.  }
  \item{solradstack}{ RasterStack object (\code{\link[raster]{stack}}) expected to have been created via \code{\link[BiodiversityR]{ensemble.envirem.solradstack}}.  }
  \item{var}{ Names of bioclimatic variables to be created; see: \code{\link[envirem]{layerCreation}}.  }
  \item{...}{ Other arguments for \code{\link[envirem]{layerCreation}}, dealing with the scale of temperature or precipitation data.  }
}

\details{

The objective of these functions is to expand a data.frame of explanatory variables that will be used for calibrating species distribution models with bioclimatic variables that are generated by the envirem package (See details in \code{\link[envirem]{layerCreation}}).

It is important that monthly values are sorted sequentially (January - December) as the \code{ensemble.envirem.masterstack} and \code{ensemble.envirem.solradstack} functions expect the inputs to be sorted in this order.

Function \code{ensemble.envirem.solradstack} requires monthly extraterrestrial solar radiation layers at the same resolution as the climatic layers. It is possible, however, to also calculate these values directly for point observation data as shown below in the examples.

}

\value{
Function \code{ensemble.envirem.run} returns a data.frame with bioclimatic variables for each point location. 
}

\author{ Roeland Kindt (World Agroforestry Centre)}

\seealso{
\code{\link[envirem]{layerCreation}}, \code{\link[BiodiversityR]{ensemble.calibrate.models}}, \code{\link[BiodiversityR]{ensemble.calibrate.weights}}
}

\references{
Title P.O., Bemmels J.B. 2018. ENVIREM: An expanded set of bioclimatic and topographic variables increases flexibility and improves performance of ecological niche modeling. Ecography 41: 291-307.
}

\examples{

\dontrun{
# Based on examples in envirem package for envirem::layerCreation

# Find example rasters
rasterFiles <- list.files(system.file('extdata', package='envirem'), 
                          full.names=TRUE)

precip.files <- rasterFiles[grepl(pattern="prec_", 
                                x=rasterFiles)]
precip.files <- precip.files[c(1, 5:12, 2:4)]
precip.stack <- raster::stack(precip.files)
precip.stack

tmin.files <- rasterFiles[grepl(pattern="tmin_", 
                                x=rasterFiles)]
tmin.files <- tmin.files[c(1, 5:12, 2:4)]
tmin.stack <- raster::stack(tmin.files)
tmin.stack

tmax.files <- rasterFiles[grepl(pattern="tmax_", 
                                x=rasterFiles)]
tmax.files <- tmax.files[c(1, 5:12, 2:4)]
tmax.stack <- raster::stack(tmax.files)
tmax.stack

tmean.files <- rasterFiles[grepl(pattern="tmean_", 
                                x=rasterFiles)]
tmean.files <- tmean.files[c(1, 5:12, 2:4)]
tmean.stack <- raster::stack(tmean.files)
tmean.stack

# Random locations
locs <- dismo::randomPoints(precip.stack[[1]], n=50)

# Climatic data
master.input <- ensemble.envirem.masterstack(x=locs,
                             precipstack=precip.stack,
                             tmaxstack=tmax.stack,
                             tminstack=tmin.stack,
                             tmeanstack=tmean.stack)

# Calculate solar radiation for 1985
# (Midpoint for the 1970-2000 WorldClim 2.1 baseline)
solrad.stack <- ETsolradRasters(precip.stack[[1]], 
                             year = 1985-1950, 
                             outputDir = NULL)

solrad.input <- ensemble.envirem.solradstack(x=locs,
                             solrad=solrad.stack)

# Obtain the envirem bioclimatic data

envirem.data1 <- ensemble.envirem.run(masterstack=master.input,
                             solradstack=solrad.input,
                             tempScale=10)

# Generate all the envirem layers, then extract
# See envirem package for envirem::layerCreation

solradFiles <- grep('solrad', rasterFiles, value=TRUE)
worldclim <- stack(setdiff(rasterFiles, solradFiles))
assignNames(precip = 'prec_##')
envirem.stack <- layerCreation(worldclim, solrad.stack, 
                               var='all', tempScale = 10)
assignNames(reset = TRUE)

envirem.data2 <- extract(envirem.stack, y=locs)

# compare
envirem.data1 - envirem.data2

# Calculate extraterrestrial solar radiation for point observations
solrad1 <- extract(solrad.stack, y=locs)
solrad2 <- array(dim=c(nrow(locs), 12))
for (i in 1:nrow(locs)) {
  lat.i <- locs[i, 2]
  for (m in 1:12) {
    solrad2[i, m] <- envirem:::calcSolRad(year=1985-1950,
                                          lat=lat.i,
                                          month=m)
  }
}

solrad1 - solrad2

}

}

