% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/homophily.fe.R
\name{homophily.fe}
\alias{homophily.fe}
\title{Estimating network formation models with degree heterogeneity: the fixed effect approach}
\usage{
homophily.fe(
  network,
  formula,
  data,
  symmetry = FALSE,
  fe.way = 1,
  init = NULL,
  opt.ctr = list(maxit = 10000, eps_f = 1e-09, eps_g = 1e-09),
  print = TRUE
)
}
\arguments{
\item{network}{matrix or list of sub-matrix of social interactions containing 0 and 1, where links are represented by 1}

\item{formula}{an object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be as for example \code{~ x1 + x2}
where \code{x1}, \code{x2} are explanatory variable of links formation. If missing, the model is estimated with fixed effects only.}

\item{data}{an optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{homophily} is called.}

\item{symmetry}{indicates whether the network model is symmetric (see details).}

\item{fe.way}{indicates whether it is a one-way or two-way fixed effect model. The expected value is 1 or 2 (see details).}

\item{init}{(optional) either a list of starting values containing \code{beta}, an K-dimensional vector of the explanatory variables parameter,
\code{mu} an n-dimensional vector, and \code{nu} an n-dimensional vector,
where K is the number of explanatory variables and n is the number of individuals; or a vector of starting value for \code{c(beta, mu, nu)}.}

\item{opt.ctr}{(optional) is a list of \code{maxit}, \code{eps_f}, and \code{eps_g}, which are control parameters used by the solver \code{optim_lbfgs}, of the package \pkg{RcppNumerical}.}

\item{print}{Boolean indicating if the estimation progression should be printed.}
}
\value{
A list consisting of:
\item{model.info}{list of model information, such as the type of fixed effects, whether the model is symmetric,
number of observations, etc.}
\item{estimate}{maximizer of the log-likelihood.}
\item{loglike}{maximized log-likelihood.}
\item{optim}{returned value of the optimization solver, which contains details of the optimization. The solver used is \code{optim_lbfgs} of the
package \pkg{RcppNumerical}.}
\item{init}{returned list of starting value.}
\item{loglike(init)}{log-likelihood at the starting value.}
}
\description{
\code{homophily.fe} implements a Logit estimator for network formation model with homophily. The model includes degree heterogeneity using fixed effects (see details).
}
\details{
Let \eqn{p_{ij}}{Pij} be a probability for a link to go from the individual \eqn{i} to the individual \eqn{j}.
This probability is specified for two-way effect models (\code{fe.way = 2}) as
\deqn{p_{ij} = F(\mathbf{x}_{ij}'\beta + \mu_j + \nu_j)}{Pij = F(Xij'*\beta + \mu_i + \nu_j),}
where \eqn{F} is the cumulative of the standard logistic distribution. Unobserved degree heterogeneity is captured by
\eqn{\mu_i} and \eqn{\nu_j}. The latter are treated as fixed effects (see \code{\link{homophily.re}} for random effect models).
As shown by Yan et al. (2019), the estimator of
the parameter \eqn{\beta} is biased. A bias correction is then necessary and is not implemented in this version. However
the estimator of \eqn{\mu_i} and \eqn{\nu_j} are consistent.\cr
For one-way fixed effect models (\code{fe.way = 1}), \eqn{\nu_j = \mu_j}. For symmetric models, the network is not directed and the
fixed effects need to be one way.
}
\examples{
\donttest{
set.seed(1234)
M            <- 2 # Number of sub-groups
nvec         <- round(runif(M, 20, 50))
beta         <- c(.1, -.1)
Glist        <- list()
dX           <- matrix(0, 0, 2)
mu           <- list()
nu           <- list()
Emunu        <- runif(M, -1.5, 0) #expectation of mu + nu
smu2         <- 0.2
snu2         <- 0.2
for (m in 1:M) {
  n          <- nvec[m]
  mum        <- rnorm(n, 0.7*Emunu[m], smu2)
  num        <- rnorm(n, 0.3*Emunu[m], snu2)
  X1         <- rnorm(n, 0, 1)
  X2         <- rbinom(n, 1, 0.2)
  Z1         <- matrix(0, n, n)  
  Z2         <- matrix(0, n, n)
  
  for (i in 1:n) {
    for (j in 1:n) {
      Z1[i, j] <- abs(X1[i] - X1[j])
      Z2[i, j] <- 1*(X2[i] == X2[j])
    }
  }
  
  Gm           <- 1*((Z1*beta[1] + Z2*beta[2] +
                       kronecker(mum, t(num), "+") + rlogis(n^2)) > 0)
  diag(Gm)     <- 0
  diag(Z1)     <- NA
  diag(Z2)     <- NA
  Z1           <- Z1[!is.na(Z1)]
  Z2           <- Z2[!is.na(Z2)]
  
  dX           <- rbind(dX, cbind(Z1, Z2))
  Glist[[m]]   <- Gm
  mu[[m]]      <- mum
  nu[[m]]      <- num
}

mu  <- unlist(mu)
nu  <- unlist(nu)

out   <- homophily.fe(network =  Glist, formula = ~ -1 + dX, fe.way = 2)
muhat <- out$estimate$mu
nuhat <- out$estimate$nu
plot(mu, muhat)
plot(nu, nuhat)
}
}
\references{
Yan, T., Jiang, B., Fienberg, S. E., & Leng, C. (2019). Statistical inference in a directed network model with covariates. \emph{Journal of the American Statistical Association}, 114(526), 857-868, \doi{https://doi.org/10.1080/01621459.2018.1448829}.
}
\seealso{
\code{\link{homophily.re}}.
}
