% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CST_Analogs.R
\name{Analogs}
\alias{Analogs}
\title{Analogs based on large scale fields.}
\usage{
Analogs(
  expL,
  obsL,
  time_obsL,
  time_expL = NULL,
  lonL = NULL,
  latL = NULL,
  expVar = NULL,
  obsVar = NULL,
  sdate_dim = "sdate",
  criteria = "Large_dist",
  excludeTime = NULL,
  lonVar = NULL,
  latVar = NULL,
  region = NULL,
  nAnalogs = NULL,
  AnalogsInfo = FALSE,
  ncores = NULL
)
}
\arguments{
\item{expL}{An array of N named dimensions containing the experimental field
on the large scale for which the analog is aimed. This field is used to in 
all the criterias. If parameter 'expVar' is not provided, the function will
return the expL analog. The element 'data' in the 's2dv_cube' object must 
have, at least, latitudinal and longitudinal dimensions. The object is 
expect to be already subset for the desired large scale region. Latitudinal 
dimension accepted names: 'lat', 'lats', 'latitude', 'y', 'j', 'nav_lat'. 
Longitudinal dimension accepted names: 'lon', 'lons','longitude', 'x', 'i', 
'nav_lon'.}

\item{obsL}{An array of N named dimensions containing the observational field 
on the large scale. The element 'data' in the 's2dv_cube' object must have 
the same latitudinal and longitudinal dimensions as parameter 'expL' and a 
single temporal dimension with the maximum number of available observations.}

\item{time_obsL}{A character string indicating the date of the observations 
in the format "dd-mm-yyyy". Reference time to search for analogs. It must 
have time dimensions.}

\item{time_expL}{An array of N named dimensions (coinciding with time 
dimensions in expL) of character string(s) indicating the date(s) of the 
experiment in the format "dd-mm-yyyy". Time(s) to find the analogs. If it 
is not an scalar it must have named dimensions.}

\item{lonL}{A vector containing the longitude of parameter 'expL'.}

\item{latL}{A vector containing the latitude of parameter 'expL'.}

\item{expVar}{An array of N named dimensions containing the experimental 
field on the local scale, usually a different variable to the parameter 
'expL'. If it is not NULL (by default, NULL), the returned field by this 
function will be the analog of parameter 'expVar'.}

\item{obsVar}{An array of N named dimensions containing the field of the 
same variable as the passed in parameter 'expVar' for the same region.}

\item{sdate_dim}{A character string indicating the name of the start date 
dimension. By default, it is set to 'sdate'.}

\item{criteria}{A character string indicating the criteria to be used for the
selection of analogs:
 \itemize{\item{Large_dist} minimum Euclidean distance in the large scale pattern; 
          \item{Local_dist} minimum Euclidean distance in the large scale pattern 
           and minimum Euclidean distance in the local scale pattern; and
          \item{Local_cor} minimum Euclidean distance in the large scale pattern, 
           minimum Euclidean distance in the local scale pattern and highest 
           correlation in the local variable to downscale.}}

\item{excludeTime}{An array of N named dimensions (coinciding with time 
dimensions in expL) of character string(s) indicating the date(s) of the 
observations in the format "dd/mm/yyyy" to be excluded during the search of 
analogs. It can be NULL but if expL is not a forecast (time_expL contained 
in time_obsL), by default time_expL will be removed during the search of 
analogs.}

\item{lonVar}{A vector containing the longitude of parameter 'expVar'.}

\item{latVar}{A vector containing the latitude of parameter 'expVar'.}

\item{region}{A vector of length four indicating the minimum longitude, 
the maximum longitude, the minimum latitude and the maximum latitude.}

\item{nAnalogs}{Number of Analogs to be selected to apply the criterias 
'Local_dist' or 'Local_cor'. This is not the necessary the number of analogs 
that the user can get, but the number of events with minimum distance in 
which perform the search of the best Analog. The default value for the 
'Large_dist' criteria is 1, for 'Local_dist' and 'Local_cor' criterias must
be greater than 1 in order to match with the first criteria, if nAnalogs is
NULL for 'Local_dist' and 'Local_cor' the default value will be set at the 
length of 'time_obsL'. If AnalogsInfo is FALSE the function returns just 
the best analog.}

\item{AnalogsInfo}{A logical value. If it is TRUE it returns a list
with two elements: 1) the downscaled field and
2) the AnalogsInfo which contains: a) the number of the best 
analogs, b) the corresponding value of the metric used in the selected 
criteria (distance values for Large_dist and Local_dist,correlation values 
for Local_cor), c)dates of the analogs). The analogs are listed in 
decreasing order, the first one is the best analog (i.e if the selected 
criteria is Local_cor the best analog will be the one with highest 
correlation, while for Large_dist criteria the best analog will be the day 
with minimum Euclidean distance). Set to FALSE to get a single analog, the 
best analog, for instance for downscaling.}

\item{ncores}{The number of cores to use in parallel computation.}
}
\value{
An array with the dowscaled values of the best analogs for the criteria 
selected. If 'AnalogsInfo' is set to TRUE it returns a list with an array 
of the dowsncaled field and the analogs information in elements 'analogs', 
'metric' and 'dates'.
}
\description{
This function perform a downscaling using Analogs. To compute 
the analogs, the function search for days with similar large scale conditions
to downscaled fields in the local scale. The large scale and the local scale 
regions are defined by the user. The large scale is usually given by 
atmospheric circulation as sea level pressure or geopotential height (Yiou 
et al, 2013) but the function gives the possibility to use another field. The
local scale will be usually given by precipitation or temperature fields, but
might be another variable. 
The analogs function will find the best analogs based in three criterias:
(1) Minimum Euclidean distance in the large scale pattern (i.e. SLP)
(2) Minimum Euclidean distance in the large scale pattern (i.e. SLP) 
and minimum Euclidean distance in the local scale pattern (i.e. SLP). 
(3) Minimum Euclidean distance in the large scale pattern (i.e. SLP), 
minimum distance in the local scale pattern (i.e. SLP) and highest 
correlation in the local variable to downscale (i.e Precipitation).
The search of analogs must be done in the longest dataset posible. This is 
important since it is necessary to have a good representation of the 
possible states of the field in the past, and therefore, to get better 
analogs. Once the search of the analogs is complete, and in order to used the 
three criterias the user can select a number of analogs , using parameter 
'nAnalogs' to restrict 
the selection of the best analogs in a short number of posibilities, the best
ones. This function has not constrains of specific regions, variables to 
downscale, or data to be used (seasonal forecast data, climate projections 
data, reanalyses data). The regrid into a finner scale is done interpolating 
with CST_Start. Then, this interpolation is corrected selecting the analogs in
the large and local scale in based of the observations. The function is an 
adapted version of the method of Yiou et al 2013.
}
\examples{
# Example 1: Downscaling using criteria 'Large_dist' and a single variable:
expSLP <- rnorm(1:20)
dim(expSLP) <- c(lat = 4, lon = 5)
obsSLP <- c(rnorm(1:180), expSLP * 1.2)
dim(obsSLP) <- c(time = 10, lat = 4, lon = 5)
time_obsSLP <- paste(rep("01", 10), rep("01", 10), 1994 : 2003, sep = "-")
dim(time_obsSLP) <- c(time = 10)
downscale_field <- Analogs(expL = expSLP, obsL = obsSLP, 
                          time_obsL = time_obsSLP,time_expL = "01-01-1994")

# Example 2: Downscaling using criteria 'Large_dist' and 2 variables:
obs.pr <- c(rnorm(1:200) * 0.001)
dim(obs.pr) <- dim(obsSLP)
downscale_field <- Analogs(expL = expSLP, obsL = obsSLP, obsVar = obs.pr,
                          time_obsL = time_obsSLP, time_expL = "01-01-1994")

# Example 3: Downscaling using criteria 'Local_dist' and 2 variables:
# analogs of local scale using criteria 2
region = c(lonmin = -1 ,lonmax = 2, latmin = 30, latmax = 33)
Local_scale <- Analogs(expL = expSLP, obsL = obsSLP, time_obsL = time_obsSLP,
                      obsVar = obs.pr, criteria = "Local_dist", 
                      lonL = seq(-1, 5, 1.5),latL = seq(30, 35, 1.5), 
                      region = region,time_expL = "01-10-2000", 
                      nAnalogs = 10, AnalogsInfo = TRUE)

# Example 4: Downscaling using criteria 'Local_cor' and 2 variables:
exp.pr <- c(rnorm(1:20) * 0.001)
dim(exp.pr) <- dim(expSLP)
Local_scalecor <- Analogs(expL = expSLP, obsL = obsSLP, time_obsL = time_obsSLP,
                         obsVar = obs.pr, expVar = exp.pr,
                         criteria = "Local_cor", lonL = seq(-1, 5, 1.5),
                         time_expL = "01-10-2000", latL = seq(30, 35, 1.5),
                         lonVar = seq(-1, 5, 1.5), latVar = seq(30, 35, 1.5), 
                         nAnalogs = 8, region = region, AnalogsInfo = FALSE)

# Example 5: List of best analogs in the three criterias Large_dist, 
Large_scale <- Analogs(expL = expSLP, obsL = obsSLP, time_obsL = time_obsSLP,
                      criteria = "Large_dist", time_expL = "01-10-2000",
                      nAnalogs = 7, AnalogsInfo = TRUE)
Local_scale <- Analogs(expL = expSLP, obsL = obsSLP, time_obsL = time_obsSLP,
                      time_expL = "01-10-2000", criteria = "Local_dist",
                      lonL = seq(-1, 5, 1.5), latL = seq(30, 35, 1.5), 
                      nAnalogs = 7, region = region, AnalogsInfo = TRUE)
Local_scalecor <- Analogs(expL = expSLP, obsL = obsSLP, time_obsL = time_obsSLP,
                         obsVar = obsSLP, expVar = expSLP, 
                         time_expL = "01-10-2000", criteria = "Local_cor", 
                         lonL = seq(-1, 5, 1.5), latL = seq(30, 35, 1.5),
                         lonVar = seq(-1, 5, 1.5), latVar = seq(30, 35, 1.5),
                         nAnalogs = 7, region = region, 
                         AnalogsInfo = TRUE)
}
\references{
Yiou, P., T. Salameh, P. Drobinski, L. Menut, R. Vautard,
and M. Vrac, 2013 : Ensemble reconstruction of the atmospheric column 
from surface pressure using analogues.  Clim. Dyn., 41, 1419-1437. 
\email{pascal.yiou@lsce.ipsl.fr}
}
\author{
M. Carmen Alvarez-Castro, \email{carmen.alvarez-castro@cmcc.it}

Maria M. Chaves-Montero, \email{mariadm.chaves@cmcc.it }

Veronica Torralba, \email{veronica.torralba@cmcc.it}

Nuria Perez-Zanon \email{nuria.perez@bsc.es}
}
