% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/set_restrictions.R
\name{set_restrictions}
\alias{set_restrictions}
\title{Restrict a model}
\usage{
set_restrictions(
  model,
  statement = NULL,
  join_by = "|",
  labels = NULL,
  param_names = NULL,
  given = NULL,
  keep = FALSE
)
}
\arguments{
\item{model}{A \code{causal_model}. A model object generated by \code{\link{make_model}}.}

\item{statement}{A quoted expressions defining the restriction. If values for some parents are not specified, statements should be surrounded by parentheses, for instance \code{(Y[A = 1] > Y[A=0])} will be interpreted for all combinations of other parents of Y set at possible levels they might take.}

\item{join_by}{A string. The logical operator joining expanded types when \code{statement} contains wildcard (\code{.}). Can take values \code{'&'} (logical AND) or \code{'|'} (logical OR). When restriction contains wildcard (\code{.}) and \code{join_by} is not specified, it defaults to \code{'|'}, otherwise it defaults to \code{NULL}. Note that join_by joins within statements, not across statements.}

\item{labels}{A list of character vectors specifying nodal types to be kept or removed from the model. Use \code{get_nodal_types} to see syntax. Note that \code{labels} gets overwritten by \code{statement} if \code{statement} is not NULL.}

\item{param_names}{A character vector of names of parameters to restrict on.}

\item{given}{A character vector or list of character vectors specifying nodes on which the parameter set to be restricted depends. When restricting by \code{statement}, \code{given} must either be \code{NULL} or of the same length as \code{statement}. When mixing statements that are further restricted by \code{given} and ones that are not, statements without \code{given} restrictions should have \code{given} specified as one of \code{NULL}, \code{NA}, \code{""} or \code{" "}.}

\item{keep}{Logical. If `FALSE`, removes and if `TRUE` keeps only causal types specified by \code{statement} or \code{labels}.}
}
\value{
An object of class \code{model}. The causal types and nodal types in the model are reduced according to the stated restriction.
}
\description{
Restrict a model's parameter space. This reduces the number of nodal types and in consequence the number of unit causal types.
}
\details{
Restrictions are made to nodal types, not to unit causal types. Thus for instance in a
model \code{X -> M -> Y}, one cannot apply a simple restriction so that \code{Y} is nondecreasing
in  \code{X}, however one can restrict so that \code{M} is nondecreasing in \code{X} and \code{Y} nondecreasing in \code{M}.
To have a restriction that \code{Y} be nondecreasing in \code{X} would otherwise require restrictions on causal types, not nodal types,
which implies a form of undeclared confounding (i.e. that in cases in which \code{M} is decreasing in \code{X}, \code{Y} is decreasing in \code{M}).

Since restrictions are to nodal types, all parents of a node are implicitly fixed.  Thus for model \code{make_model(`X -> Y <- W`)} the request
\code{set_restrictions(`(Y[X=1] == 0)`)}
is interpreted as \code{set_restrictions(`(Y[X=1, W=0] == 0 | Y[X=1, W=1] == 0)`)}.


Statements with implicitly controlled nodes should be surrounded by parentheses, as in these examples.


Note that prior probabilities are redistributed over remaining types.
}
\examples{

# 1. Restrict parameter space using statements
model <- make_model('X->Y') \%>\%
  set_restrictions(statement = c('X[] == 0'))

model <- make_model('X->Y') \%>\%
  set_restrictions(non_increasing('X', 'Y'))

model <- make_model('X -> Y <- W') \%>\%
  set_restrictions(c(decreasing('X', 'Y'), substitutes('X', 'W', 'Y')))

model$parameters_df

model <- make_model('X-> Y <- W') \%>\%
  set_restrictions(statement = decreasing('X', 'Y'))
model$parameters_df

model <- make_model('X->Y') \%>\%
  set_restrictions(decreasing('X', 'Y'))
model$parameters_df

model <- make_model('X->Y') \%>\%
  set_restrictions(c(increasing('X', 'Y'), decreasing('X', 'Y')))
model$parameters_df
\donttest{
# Restrict to define a model with monotonicity
model <- make_model('X->Y') \%>\%
set_restrictions(statement = c('Y[X=1] < Y[X=0]'))
get_parameter_matrix(model)

# Restrict to a single type in endogenous node
model <- make_model('X->Y') \%>\%
set_restrictions(statement =  '(Y[X = 1] == 1)', join_by = '&', keep = TRUE)
get_parameter_matrix(model)

#  Use of | and &
# Keep node if *for some value of B* Y[A = 1] == 1
model <- make_model('A->Y<-B') \%>\%
set_restrictions(statement =  '(Y[A = 1] == 1)', join_by = '|', keep = TRUE)
dim(get_parameter_matrix(model))


# Keep node if *for all values of B* Y[A = 1] == 1
model <- make_model('A->Y<-B') \%>\%
set_restrictions(statement =  '(Y[A = 1] == 1)', join_by = '&', keep = TRUE)
dim(get_parameter_matrix(model))

# Restrict multiple nodes
model <- make_model('X->Y<-M; X -> M' ) \%>\%
set_restrictions(statement =  c('(Y[X = 1] == 1)', '(M[X = 1] == 1)'), join_by = '&', keep = TRUE)
get_parameter_matrix(model)

# Restrict using statements and given:
model <- make_model("X -> Y -> Z; X <-> Z") \%>\%
 set_restrictions(list(decreasing('X','Y'), decreasing('Y','Z')), given = c(NA,'X.0'))
get_parameter_matrix(model)

# Restrictions on levels for endogenous nodes aren't allowed
\dontrun{
model <- make_model('X->Y') \%>\%
set_restrictions(statement =  '(Y == 1)')
}

# 2. Restrict parameter space Using labels:
model <- make_model('X->Y') \%>\%
set_restrictions(labels = list(X = '0', Y = '00'))

# Restrictions can be  with wildcards
model <- make_model('X->Y') \%>\%
set_restrictions(labels = list(Y = '?0'))
get_parameter_matrix(model)

# Deterministic model
model <- make_model('S -> C -> Y <- R <- X; X -> C -> R') \%>\%
set_restrictions(labels = list(C = '1000', R = '0001', Y = '0001'), keep = TRUE)
get_parameter_matrix(model)

# Restrict using labels and given:
model <- make_model("X -> Y -> Z; X <-> Z") \%>\%
 set_restrictions(labels = list(X = '0', Z = '00'), given = c(NA,'X.0'))
get_parameter_matrix(model)
}
}
\seealso{
Other restrictions: 
\code{\link{restrict_by_labels}()},
\code{\link{restrict_by_query}()}
}
\concept{restrictions}
