% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf2DistGP.R
\name{cf2DistGP}
\alias{cf2DistGP}
\title{Evaluating CDF/PDF/QF from CF of a continous distribution F by using the Gil-Pelaez inversion formulae.}
\usage{
cf2DistGP(cf, x, prob, option, isCompound, isCircular, N, SixSigmaRule, xMin,
  xMax, isPlot)
}
\arguments{
\item{cf}{function handle for the characteristic function CF}

\item{x}{vector of values from domain of the distribution F, if missing,
cf2DistFFT automatically selects vector x from the domain}

\item{prob}{vector of values from [0,1] for which the quantiles will be estimated}

\item{option}{list with the following (default) parameters:
\itemize{
    \item option$isCompound   = FALSE   treat the compound distributions
    \item option$isCircular   = FALSE   treat the circular distributions
    \item option$isInterp     = FALSE   create and use the interpolant functions for PDF/CDF/RND
    \item option$N            = 2^10    set N points used by FFT
    \item option$xMin         = -Inf    set the lower limit of X
    \item option$xMax         = Inf     set the upper limit of X
    \item option$xMean        = NULL    set the MEAN value of X
    \item option$xStd         = NULL    set the STD value of X
    \item option$dt           = NULL    set grid step dt = 2*pi/xRange
    \item option$T            = NULL    set upper limit of (0,T), T = N*dt
    \item option$SixSigmaRule = 6       set the rule for computing domain
    \item option$tolDiff      = 1e-4    tol for numerical differentiation
    \item option$isPlot       = TRUE    plot the graphs of PDF/CDF

    \item option$DIST                   list with information for future evaluations.
                                        option$DIST is created automatically after first call:
    \itemize{
        \item option$DIST$xMean
        \item option$DIST$xMin
        \item option$DIST$xMax
        \item option$DIST$xRange
        \item option$DIST$cft
        \item option$DIST$N
        \item option$DIST$k
        \item option$DIST$t
        \item option$DIST$dt
        }
    }}

\item{isCompound}{treat the compound distributions, default FALSE}

\item{isCircular}{treat the circular distributions, default FALSE}

\item{N}{N points used by GP, default 2^10 (2^14 for compound distribution)}

\item{SixSigmaRule}{set the rule for computing domain, default 6}

\item{xMin}{set the lower limit of X}

\item{xMax}{set the upper limit of X}

\item{isPlot}{plot the graphs of PDF/CDF, default TRUE}
}
\value{
result - list with the following results values:
   \item{result$x}{<- x}
   \item{result$cdf}{<- cdf (Cumulative Distribution Function)}
   \item{result$pdf}{<- pdf (Probability Density Function)}
   \item{result$prob}{<- prob}
   \item{result$qf}{<- qf (Quantile Function)}
   \item{result$SixSigmaRule}{<- option$SixSigmaRule}
   \item{result$N}{<- option$N}
   \item{result$dt}{<- dt}
   \item{result$T}{<- t[(length(t))]}
   \item{result$PrecisionCrit}{<- PrecisionCrit}
   \item{result$myPrecisionCrit}{<- options$crit}
   \item{result$isPrecisionOK}{<- isPrecisionOK}
   \item{result$xMean}{<- xMean}
   \item{result$xRange}{<- xRange}
   \item{result$xSTD}{<- xStd}
   \item{result$xMin}{<- xMin}
   \item{result$xMAx}{<- xMax}
   \item{result$cf}{<- cf}
   \item{result$const}{<- const}
   \item{result$isCompound}{<- option$isCompound}
   \item{result$details$count}{<- count}
   \item{result$details$correction}{<- correction}
   \item{result$option}{<- option}
   \item{result$tictoc}{<- toc}
}
\description{
\code{cf2DistGP(cf, x, prob, options)} evaluates the CDF/PDF/QF (Cumulative Distribution Function,
Probability Density Function, Quantile Function) from the Characteristic Function CF
by using the Gil-Pelaez inversion formulae: \cr
cdf(x) = 1/2 + (1/pi) * Integral_0^inf imag(exp(-1i*t*x)*cf(t)/t)*dt. \cr
pdf(x) = (1/pi) * Integral_0^inf real(exp(-1i*t*x)*cf(t))*dt.
}
\note{
If options.DIST is provided, then cf2DistGP evaluates CDF/PDF based on
this information only (it is useful, e.g., for subsequent evaluation of
the quantiles). options.DIST is created automatically after first call.
This is supposed to be much faster, bacause there is no need for further
evaluations of the characteristic function. In fact, in such case the
function handle of the CF is not required, i.e. in such case set cf = []

The required integrals are evaluated approximately by using the simple
trapezoidal rule on the interval(0,T), where T = N * dt is a sufficienly
large integration upper limit in the frequency domain.

If the optimum values of N and T are unknown, we suggest, as a simple
rule of thumb, to start with the application of the six-sigma-rule for
determining the value of dt = (2*pi)/(xMax-xMin), where xMax = xMean +
6*xStd, and xMin = xMean - 6*xStd, see \code{[1]}.

Please note that THIS (TRAPEZOIDAL) METHOD IS AN APPROXIMATE METHOD:
Frequently, with relatively low numerical precision of the results of the
calculated PDF/CDF/QF, but frequently more efficient and more precise
than comparable Monte Carlo methods.

However, the numerical error (truncation error and/or the integration
error) could be and should be properly controled!

CONTROLING THE PRECISION:
Simple criterion for controling numerical precision is as follows: Set N
and T = N*dt such that the value of the integrand function
Imag(e^(-1i*t*x) * cf(t)/t) is sufficiently small for all t > T, i.e.
PrecisionCrit = abs(cf(t)/t) <= tol,
for pre-selected small tolerance value, say tol = 10^-8. If this
criterion is not valid, the numerical precission of the result is
violated, and the method should be improved (e.g. by selecting larger N
or considering other more sofisticated algorithm - not considered here).
}
\examples{
## EXAMPLE1 (Calculate CDF/PDF of N(0,1) by inverting its CF)
cf <- function(t)
  exp(-t ^ 2 / 2)
result <- cf2DistGP(cf)

## EXAMPLE2 (PDF/CDF of the compound Binomial-Exponential distribution)
n <- 25
p <- 0.3
lambda <- 5
cfX <- function(t)
  cfX_Exponential(t, lambda)
cf <- function(t)
  cfN_Binomial(t, n, p, cfX)
x <- seq(0, 5, length.out = 101)
prob <- c(0.9, 0.95, 0.99)
result <- cf2DistGP(cf, x, prob, isCompound = TRUE)

## EXAMPLE3 (PDF/CDF of the compound Poisson-Exponential distribution)
lambda1 <- 10
lambda2 <- 5
cfX <- function(t)
  cfX_Exponential(t, lambda2)
cf <- function(t)
  cfN_Poisson(t, lambda1, cfX)
x <- seq(0, 8, length.out = 101)
prob <- c(0.9, 0.95, 0.99)
result <- cf2DistGP(cf, x, prob, isCompound = TRUE)
}
\seealso{
For more details see:
\url{https://arxiv.org/pdf/1701.08299.pdf}
}

