% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dmcData.R
\name{dmcObservedData}
\alias{dmcObservedData}
\title{dmcObservedData: Run standard analyses on observed raw data}
\usage{
dmcObservedData(
  dat,
  nCAF = 5,
  nDelta = 19,
  pDelta = vector(),
  tDelta = 1,
  outlier = c(200, 1200),
  columns = c("Subject", "Comp", "RT", "Error"),
  compCoding = c("comp", "incomp"),
  errorCoding = c(0, 1),
  quantileType = 5,
  keepRaw = FALSE,
  delim = "\\t",
  skip = 0
)
}
\arguments{
\item{dat}{A text file(s) containing the observed data or an R DataFrame (see createDF/addDataDF)}

\item{nCAF}{The number of CAF bins.}

\item{nDelta}{The number of delta bins.}

\item{pDelta}{An alternative option to nDelta by directly specifying required percentile values (vector of values 0-100)}

\item{tDelta}{The type of delta calculation (1=direct percentiles points, 2=percentile bounds (tile) averaging)}

\item{outlier}{Outlier limits in ms (e.g., c(200, 1200))}

\item{columns}{Name of required columns DEFAULT = c("Subject", "Comp", "RT", "Error")}

\item{compCoding}{Coding for compatibility DEFAULT = c("comp", "incomp")}

\item{errorCoding}{Coding for errors DEFAULT = c(0, 1))}

\item{quantileType}{Argument (1-9) from R function quantile specifying the algorithm (?quantile)}

\item{keepRaw}{TRUE/FALSE}

\item{delim}{Single character used to separate fields within a record if reading from external text file.}

\item{skip}{The number of lines to skip before reading data if reading from external text file.}
}
\value{
DataFrame
}
\description{
Basic analysis to create data object required for observed data.
Example raw *.txt files are flankerData.txt and simonData.txt. There are four critical columns:
A column containing subject number
A column coding for compatible or incompatible
A column with RT (in ms)
A column indicating of the response was correct
}
\examples{
# Example 1
plot(flankerData)  # flanker data from Ulrich et al. (2015)
plot(simonData)    # simon data from Ulrich et al. (2015)

# Example 2 (Basic behavioural analysis from Ulrich et al. 2015)
flankerDat <- cbind(Task = "flanker", flankerData$summarySubject)
simonDat   <- cbind(Task = "simon",   simonData$summarySubject)
datAgg     <- rbind(flankerDat, simonDat)

datAgg$Subject <- factor(datAgg$Subject)
datAgg$Task    <- factor(datAgg$Task)
datAgg$Comp    <- factor(datAgg$Comp)

aovErr <- aov(perErr ~ Comp*Task + Error(Subject/(Comp*Task)), datAgg)
summary(aovErr)
model.tables(aovErr, type = "mean")

aovRt <- aov(rtCor ~ Comp*Task + Error(Subject/(Comp*Task)), datAgg)
summary(aovRt)
model.tables(aovRt, type = "mean")

# Example 3
dat <- createDF(nSubjects = 50, nTrl = 500, design = list("Comp" = c("comp", "incomp")))
dat <- addDataDF(dat,
                 RT = list("Comp_comp"    = c(500, 75, 120),
                           "Comp_incomp"  = c(530, 75, 100)),
                 Error = list("Comp_comp" = c(3, 2, 2, 1, 1),
                            "Comp_incomp" = c(21, 3, 2, 1, 1)))
datOb <- dmcObservedData(dat)
plot(datOb)
plot(datOb, subject = 1)

# Example 4
dat <- createDF(nSubjects = 50, nTrl = 500, design = list("Congruency" = c("cong", "incong")))
dat <- addDataDF(dat,
                 RT = list("Congruency_cong"   = c(500, 75, 100),
                           "Congruency_incong" = c(530, 100, 110)),
                 Error = list("Congruency_cong"   = c(3, 2, 2, 1, 1),
                              "Congruency_incong" = c(21, 3, 2, 1, 1)))
datOb <- dmcObservedData(dat, nCAF = 5, nDelta = 9,
                         columns = c("Subject", "Congruency", "RT", "Error"),
                         compCoding = c("cong", "incong"))
plot(datOb, labels = c("Congruent", "Incongruent"))
plot(datOb, subject = 1)

}
