% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/StatModels.R
\name{WeightedERMDP.CMS}
\alias{WeightedERMDP.CMS}
\title{Privacy-preserving Weighted Empirical Risk Minimization}
\description{
This class implements differentially private empirical risk
minimization in the case where weighted observation-level losses are
desired (such as weighted SVM \insertCite{Yang2005}{DPpack}). Currently,
only the output perturbation method is implemented.
}
\details{
To use this class for weighted empirical risk minimization, first
use the \code{new} method to construct an object of this class with the
desired function values and hyperparameters. After constructing the object,
the \code{fit} method can be applied with a provided dataset, data bounds,
weights, and weight bounds to fit the model. In fitting, the model stores a
vector of coefficients \code{coeff} which satisfy differential privacy.
These can be released directly, or used in conjunction with the
\code{predict} method to privately predict the outcomes of new datapoints.

Note that in order to guarantee differential privacy for weighted empirical
risk minimization, certain constraints must be satisfied for the values
used to construct the object, as well as for the data used to fit. These
conditions depend on the chosen perturbation method, though currently only
output perturbation is implemented. Specifically, the provided loss
function must be convex and differentiable with respect to \code{y.hat},
and the absolute value of the first derivative of the loss function must be
at most 1. If objective perturbation is chosen (not currently implemented),
the loss function must also be doubly differentiable and the absolute value
of the second derivative of the loss function must be bounded above by a
constant c for all possible values of \code{y.hat} and \code{y}, where
\code{y.hat} is the predicted label and \code{y} is the true label. The
regularizer must be 1-strongly convex and differentiable. It also must be
doubly differentiable if objective perturbation is chosen. For the data x,
it is assumed that if x represents a single row of the dataset X, then the
l2-norm of x is at most 1 for all x. Note that because of this, a bias term
cannot be included without appropriate scaling/preprocessing of the
dataset. To ensure privacy, the add.bias argument in the \code{fit} and
\code{predict} methods should only be utilized in subclasses within this
package where appropriate preprocessing is implemented, not in this class.
Finally, if weights are provided, they should be nonnegative, of the same
length as y, and be upper bounded by a global or public bound which must
also be provided.
}
\examples{
# Build train dataset X and y, and test dataset Xtest and ytest
N <- 200
K <- 2
X <- data.frame()
y <- data.frame()
for (j in (1:K)){
  t <- seq(-.25, .25, length.out = N)
  if (j==1) m <- stats::rnorm(N,-.2, .1)
  if (j==2) m <- stats::rnorm(N, .2, .1)
  Xtemp <- data.frame(x1 = 3*t , x2 = m - t)
  ytemp <- data.frame(matrix(j-1, N, 1))
  X <- rbind(X, Xtemp)
  y <- rbind(y, ytemp)
}
Xtest <- X[seq(1,(N*K),10),]
ytest <- y[seq(1,(N*K),10),,drop=FALSE]
X <- X[-seq(1,(N*K),10),]
y <- y[-seq(1,(N*K),10),,drop=FALSE]

# Construct object for weighted linear SVM
mapXy <- function(X, coeff) X\%*\%coeff
# Huber loss from DPpack
huber.h <- 0.5
loss <- generate.loss.huber(huber.h)
regularizer <- 'l2' # Alternatively, function(coeff) coeff\%*\%coeff/2
eps <- 1
gamma <- 1
perturbation.method <- 'output'
c <- 1/(2*huber.h) # Required value for SVM
mapXy.gr <- function(X, coeff) t(X)
loss.gr <- generate.loss.gr.huber(huber.h)
regularizer.gr <- function(coeff) coeff
wermdp <- WeightedERMDP.CMS$new(mapXy, loss, regularizer, eps,
                                gamma, perturbation.method, c,
                                mapXy.gr, loss.gr,
                                regularizer.gr)

# Fit with data
# Bounds for X based on construction
upper.bounds <- c( 1, 1)
lower.bounds <- c(-1,-1)
weights <- rep(1, nrow(y)) # Uniform weighting
weights[nrow(y)] <- 0.5 # half weight for last observation
wub <- 1 # Public upper bound for weights
wermdp$fit(X, y, upper.bounds, lower.bounds, weights=weights,
           weights.upper.bound=wub)
wermdp$coeff # Gets private coefficients

# Predict new data points
predicted.y <- wermdp$predict(Xtest)
n.errors <- sum(round(predicted.y)!=ytest)

}
\references{
\insertRef{chaudhuri2011}{DPpack}

\insertRef{Yang2005}{DPpack}
}
\keyword{internal}
\section{Super class}{
\code{\link[DPpack:EmpiricalRiskMinimizationDP.CMS]{DPpack::EmpiricalRiskMinimizationDP.CMS}} -> \code{WeightedERMDP.CMS}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-WeightedERMDP.CMS-new}{\code{WeightedERMDP.CMS$new()}}
\item \href{#method-WeightedERMDP.CMS-fit}{\code{WeightedERMDP.CMS$fit()}}
\item \href{#method-WeightedERMDP.CMS-predict}{\code{WeightedERMDP.CMS$predict()}}
\item \href{#method-WeightedERMDP.CMS-clone}{\code{WeightedERMDP.CMS$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-WeightedERMDP.CMS-new"></a>}}
\if{latex}{\out{\hypertarget{method-WeightedERMDP.CMS-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{WeightedERMDP.CMS} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{WeightedERMDP.CMS$new(
  mapXy,
  loss,
  regularizer,
  eps,
  gamma,
  perturbation.method = "objective",
  c = NULL,
  mapXy.gr = NULL,
  loss.gr = NULL,
  regularizer.gr = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mapXy}}{Map function of the form \code{mapXy(X, coeff)} mapping input
data matrix \code{X} and coefficient vector or matrix \code{coeff} to
output labels \code{y}. Should return a column matrix of predicted labels
for each row of \code{X}. See \code{\link{mapXy.sigmoid}} for an example.}

\item{\code{loss}}{Loss function of the form \code{loss(y.hat, y, w)}, where
\code{y.hat} and \code{y} are matrices and \code{w} is a matrix or vector
of weights of the same length as \code{y}. Should be defined such that it
returns a matrix of weighted loss values for each element of \code{y.hat}
and \code{y}. If \code{w} is not given, the function should operate as if
uniform weights were given. See \code{\link{generate.loss.huber}} for an
example. It must be convex and differentiable, and the absolute value of
the first derivative of the loss function must be at most 1.
Additionally, if the objective perturbation method is chosen, it must be
doubly differentiable and the absolute value of the second derivative of
the loss function must be bounded above by a constant c for all possible
values of \code{y.hat} and \code{y}.}

\item{\code{regularizer}}{String or regularization function. If a string, must be
'l2', indicating to use l2 regularization. If a function, must have form
\code{regularizer(coeff)}, where \code{coeff} is a vector or matrix, and
return the value of the regularizer at \code{coeff}. See
\code{\link{regularizer.l2}} for an example. Additionally, in order to
ensure differential privacy, the function must be 1-strongly convex and
differentiable. If the objective perturbation method is chosen, it must
also be doubly differentiable.}

\item{\code{eps}}{Positive real number defining the epsilon privacy budget. If set
to Inf, runs algorithm without differential privacy.}

\item{\code{gamma}}{Nonnegative real number representing the regularization
constant.}

\item{\code{perturbation.method}}{String indicating whether to use the 'output' or
the 'objective' perturbation methods \insertCite{chaudhuri2011}{DPpack}.
Defaults to 'objective'. Currently, only the output perturbation method
is supported.}

\item{\code{c}}{Positive real number denoting the upper bound on the absolute
value of the second derivative of the loss function, as required to
ensure differential privacy for the objective perturbation method. This
input is unnecessary if perturbation.method is 'output', but is required
if perturbation.method is 'objective'. Defaults to NULL.}

\item{\code{mapXy.gr}}{Optional function representing the gradient of the map
function with respect to the values in \code{coeff}. If given, must be of
the form \code{mapXy.gr(X, coeff)}, where \code{X} is a matrix and
\code{coeff} is a matrix or numeric vector. Should be defined such that
the ith row of the output represents the gradient with respect to the ith
coefficient. See \code{\link{mapXy.gr.sigmoid}} for an example. If not
given, non-gradient based optimization methods are used to compute the
coefficient values in fitting the model.}

\item{\code{loss.gr}}{Optional function representing the gradient of the loss
function with respect to \code{y.hat} and of the form
\code{loss.gr(y.hat, y, w)}, where \code{y.hat} and \code{y} are matrices
and \code{w} is a matrix or vector of weights. Should be defined such
that the ith row of the output represents the gradient of the (possibly
weighted) loss function at the ith set of input values. See
\code{\link{generate.loss.gr.huber}} for an example. If not given,
non-gradient based optimization methods are used to compute the
coefficient values in fitting the model.}

\item{\code{regularizer.gr}}{Optional function representing the gradient of the
regularization function with respect to \code{coeff} and of the form
\code{regularizer.gr(coeff)}. Should return a vector. See
\code{\link{regularizer.gr.l2}} for an example. If \code{regularizer} is
given as a string, this value is ignored. If not given and
\code{regularizer} is a function, non-gradient based optimization methods
are used to compute the coefficient values in fitting the model.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{WeightedERMDP.CMS} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-WeightedERMDP.CMS-fit"></a>}}
\if{latex}{\out{\hypertarget{method-WeightedERMDP.CMS-fit}{}}}
\subsection{Method \code{fit()}}{
Fit the differentially private weighted empirical risk
minimization model. This method runs either the output perturbation or
the objective perturbation algorithm \insertCite{chaudhuri2011}{DPpack}
(only output is currently implemented), depending on the value of
perturbation.method used to construct the object, to generate an
objective function. A numerical optimization method is then run to find
optimal coefficients for fitting the model given the training data,
weights, and hyperparameters. The built-in \code{\link{optim}} function
using the "BFGS" optimization method is used. If \code{mapXy.gr},
\code{loss.gr}, and \code{regularizer.gr} are all given in the
construction of the object, the gradient of the objective function is
utilized by \code{optim} as well. Otherwise, non-gradient based
optimization methods are used. The resulting privacy-preserving
coefficients are stored in \code{coeff}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{WeightedERMDP.CMS$fit(
  X,
  y,
  upper.bounds,
  lower.bounds,
  add.bias = FALSE,
  weights = NULL,
  weights.upper.bound = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Dataframe of data to be fit.}

\item{\code{y}}{Vector or matrix of true labels for each row of \code{X}.}

\item{\code{upper.bounds}}{Numeric vector of length \code{ncol(X)} giving upper
bounds on the values in each column of X. The \code{ncol(X)} values are
assumed to be in the same order as the corresponding columns of \code{X}.
Any value in the columns of \code{X} larger than the corresponding upper
bound is clipped at the bound.}

\item{\code{lower.bounds}}{Numeric vector of length \code{ncol(X)} giving lower
bounds on the values in each column of \code{X}. The \code{ncol(X)}
values are assumed to be in the same order as the corresponding columns
of \code{X}. Any value in the columns of \code{X} larger than the
corresponding upper bound is clipped at the bound.}

\item{\code{add.bias}}{Boolean indicating whether to add a bias term to \code{X}.
Defaults to FALSE.}

\item{\code{weights}}{Numeric vector of observation weights of the same length as
\code{y}.}

\item{\code{weights.upper.bound}}{Numeric value representing the global or public
upper bound on the weights.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-WeightedERMDP.CMS-predict"></a>}}
\if{latex}{\out{\hypertarget{method-WeightedERMDP.CMS-predict}{}}}
\subsection{Method \code{predict()}}{
Predict label(s) for given \code{X} using the fitted
coefficients.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{WeightedERMDP.CMS$predict(X, add.bias = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Dataframe of data on which to make predictions. Must be of same
form as \code{X} used to fit coefficients.}

\item{\code{add.bias}}{Boolean indicating whether to add a bias term to \code{X}.
Defaults to FALSE. If add.bias was set to TRUE when fitting the
coefficients, add.bias should be set to TRUE for predictions.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Matrix of predicted labels corresponding to each row of \code{X}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-WeightedERMDP.CMS-clone"></a>}}
\if{latex}{\out{\hypertarget{method-WeightedERMDP.CMS-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{WeightedERMDP.CMS$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
