# "read_sol() Single profile"

  test_sol <- tempfile()

    write(
      c("*SOILS: General DSSAT Soil Input File",
        "",
        "*IB00000001  IBSNAT      SIC     210 DEFAULT - DEEP SILTY CLAY",
        "@SITE        COUNTRY          LAT     LONG SCS FAMILY",
        " Generic     Generic          -99      -99 Generic",
        "@ SCOM  SALB  SLU1  SLDR  SLRO  SLNF  SLPF  SMHB  SMPX  SMKE",
        "   -99  0.11   6.0  0.30  85.0  1.00  1.00 IB001 IB001 IB001",
        "@  SLB  SLMH  SLLL  SDUL  SSAT  SRGF  SSKS  SBDM  SLOC  SLCL  SLSI  SLCF  SLNI  SLHW  SLHB  SCEC  SADC",
        "     5   -99 0.228 0.385 0.481 1.000   -99  1.30  1.75  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    15   -99 0.228 0.385 0.481 1.000   -99  1.30  1.75  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    30   -99 0.249 0.406 0.482 0.638   -99  1.30  1.60  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    45   -99 0.249 0.406 0.465 0.472   -99  1.35  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    60   -99 0.249 0.406 0.465 0.350   -99  1.35  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    90   -99 0.308 0.456 0.468 0.223   -99  1.35  1.10  50.0  45.0   0.0 0.110   6.5   -99   -99   -99 ",
        "   120   -99 0.207 0.341 0.452 0.122   -99  1.40  0.65  50.0  45.0   0.0 0.060   6.5   -99   -99   -99 ",
        "   150   -99 0.243 0.365 0.455 0.067   -99  1.40  0.30  50.0  45.0   0.0 0.030   6.5   -99   -99   -99 ",
        "   180   -99 0.259 0.361 0.457 0.037   -99  1.40  0.10  50.0  45.0   0.0 0.010   6.5   -99   -99   -99 ",
        "   210   -99 0.259 0.361 0.457 0.020   -99  1.40  0.01  50.0  45.0   0.0 0.000   6.5   -99   -99   -99 ",
        ""),
      test_sol
      )

    sol <- DSSAT::read_sol(test_sol)

file.remove(test_sol)

  info_prefix <- "read_sol() Single profile"
    actual <- sol
    char_cols <- c("PEDON", "SOURCE", "TEXTURE", "DESCRIPTION",
                  "SITE", "COUNTRY", "SCS FAMILY", "SCOM",
                  "SMHB", "SMPX", "SMKE", "SLMH")
    list_cols <- c("SLB", "SLMH", "SLLL", "SDUL", "SSAT",
                  "SRGF", "SSKS", "SBDM", "SLOC", "SLCL",
                  "SLSI", "SLCF", "SLNI", "SLHW", "SLHB",
                  "SCEC", "SADC")
    expected_vals <- list(PEDON = "IB00000001",
                         SOURCE = "IBSNAT",
                         TEXTURE = "SIC",
                         DEPTH = 210,
                         DESCRIPTION = "DEFAULT - DEEP SILTY CLAY",
                         SITE = "Generic",
                         COUNTRY = "Generic",
                         LAT = NA_real_,
                         LONG = NA_real_,
                         "SCS FAMILY" = "Generic",
                         SCOM = NA_character_,
                         SALB = 0.11,
                         SLU1 = 6,
                         SLDR = 0.3,
                         SLRO = 85,
                         SLNF = 1,
                         SLPF = 1,
                         SMHB = "IB001",
                         SMPX = "IB001",
                         SMKE = "IB001",
                         SLB = list(c(5, 15, 30, 45, 60, 90,
                                      120, 150, 180, 210)),
                         SLMH = list(c(NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_)),
                         SLLL = list(c(0.228, 0.228, 0.249, 0.249,
                                       0.249, 0.308, 0.207, 0.243,
                                       0.259, 0.259)),
                         SDUL = list(c(0.385, 0.385, 0.406, 0.406,
                                       0.406, 0.456, 0.341, 0.365,
                                       0.361, 0.361)),
                         SSAT = list(c(0.481, 0.481, 0.482, 0.465,
                                       0.465, 0.468, 0.452, 0.455,
                                       0.457, 0.457)),
                         SRGF = list(c(1, 1, 0.638, 0.472, 0.35,
                                       0.223, 0.122, 0.067,
                                       0.037, 0.02)),
                         SSKS = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_)),
                         SBDM = list(c(1.3, 1.3, 1.3, 1.35,
                                       1.35, 1.35, 1.4, 1.4,
                                       1.4, 1.4)),
                         SLOC = list(c(1.75, 1.75, 1.6, 1.45,
                                       1.45, 1.1, 0.65, 0.3,
                                       0.1, 0.01)),
                         SLCL = list(c(50, 50, 50, 50, 50,
                                       50, 50, 50, 50, 50)),
                         SLSI = list(c(45, 45, 45, 45, 45,
                                       45, 45, 45, 45, 45)),
                         SLCF = list(c(0, 0, 0, 0, 0, 0,
                                       0, 0, 0, 0)),
                         SLNI = list(c(0.17, 0.17, 0.17, 0.14,
                                       0.14, 0.11, 0.06, 0.03,
                                       0.01, 0)),
                         SLHW = list(c(6.5, 6.5, 6.5, 6.5,
                                       6.5, 6.5, 6.5, 6.5,
                                       6.5, 6.5)),
                         SLHB = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_)),
                         SCEC = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_)),
                         SADC = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_)))

  # Check for all missing variables
  for(nm in names(actual)){
    if("missing" %in% objects() && nm %in% missing){
      if(exists("char_cols") && nm %in% char_cols){
        na_val <- NA_character_
      }else if("date_cols" %in% objects() && !is.null(date_cols) && nm == date_cols){
        na_val <- as.POSIXct(NA, tz="UTC")
      }else{
        na_val <- NA_real_
      }
      if("list_cols" %in% objects() && !is.null(list_cols) && nm %in% list_cols){
        expect_equal(unlist(actual[[nm]]),
                               rep(na_val, length(unlist(actual[[nm]]))),
                               info = paste0(info_prefix, ": ", nm))
      }else{
        expect_equal(actual[[nm]],
                     rep(na_val, nrow(actual)),
                     info = paste0(info_prefix, ": ", nm))
      }
    }
  }

  # Check for specific expected values
  if("expected_vals" %in% objects() && !is.null(expected_vals)){
    for(nm in names(expected_vals)){
        expect_equal(actual[[nm]],
                     expected_vals[[nm]],
                     info = paste0(info_prefix, ": ", nm))
    }
  }

  # Check list column type and dimensions
  if("list_cols" %in% objects() && !is.null(list_cols)){
    for(nm in list_cols){
        expect_true(is.list(actual[[nm]]),
                    info = paste0(info_prefix, ": ", nm))
      if(exists("list_col_length") && !is.null(list_col_length)){
        if(length(list_col_length) == 1){
          list_col_length <- rep(list_col_length, length(actual[[nm]]))
        }
        for(i in 1:length(actual[[nm]])){
            expect_equal(length(unlist(actual[[nm]][i])),
                       list_col_length[i],
                       info = paste0(info_prefix, nm))
        }
      }
      if("list_col_groups" %in% objects() && !is.null(list_col_groups)){
        for(i in 1:nrow(actual)){
          for(g in 1:length(list_col_groups)){
            length_range <- range(
              sapply(actual[i,][list_col_groups[[g]]],
                     function(x)length(unlist(x)))
              )
            lbl = paste0(paste0(list_col_groups[[g]],collapse = ", ")," - row ", i)
              expect_equal(length_range[1],
                         length_range[2],
                         info = paste0(info_prefix, ": ", lbl))
          }
        }
      }
    }
  }
  for(arg in c("char_cols", "list_cols", "date_cols",
               "missing", "list_col_length",
               "list_col_groups", "expected_vals")){
    if(arg %in% objects()) rm(list = arg)
  }




# "read_sol() Three profiles"

  test_sol <- tempfile()

    write(
      c("*SOILS: General DSSAT Soil Input File",
        "",
        "*IB00000001  IBSNAT      SIC     210 DEFAULT - DEEP SILTY CLAY",
        "@SITE        COUNTRY          LAT     LONG SCS FAMILY",
        " Generic     Generic          -99      -99 Generic",
        "@ SCOM  SALB  SLU1  SLDR  SLRO  SLNF  SLPF  SMHB  SMPX  SMKE",
        "   -99  0.11   6.0  0.30  85.0  1.00  1.00 IB001 IB001 IB001",
        "@  SLB  SLMH  SLLL  SDUL  SSAT  SRGF  SSKS  SBDM  SLOC  SLCL  SLSI  SLCF  SLNI  SLHW  SLHB  SCEC  SADC",
        "     5   -99 0.228 0.385 0.481 1.000   -99  1.30  1.75  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    15   -99 0.228 0.385 0.481 1.000   -99  1.30  1.75  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    30   -99 0.249 0.406 0.482 0.638   -99  1.30  1.60  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    45   -99 0.249 0.406 0.465 0.472   -99  1.35  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    60   -99 0.249 0.406 0.465 0.350   -99  1.35  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    90   -99 0.308 0.456 0.468 0.223   -99  1.35  1.10  50.0  45.0   0.0 0.110   6.5   -99   -99   -99 ",
        "   120   -99 0.207 0.341 0.452 0.122   -99  1.40  0.65  50.0  45.0   0.0 0.060   6.5   -99   -99   -99 ",
        "   150   -99 0.243 0.365 0.455 0.067   -99  1.40  0.30  50.0  45.0   0.0 0.030   6.5   -99   -99   -99 ",
        "   180   -99 0.259 0.361 0.457 0.037   -99  1.40  0.10  50.0  45.0   0.0 0.010   6.5   -99   -99   -99 ",
        "   210   -99 0.259 0.361 0.457 0.020   -99  1.40  0.01  50.0  45.0   0.0 0.000   6.5   -99   -99   -99 ",
        "",
        "*IB00000002  IBSNAT      SIC     150 DEFAULT - MEDIUM SILTY CLAY",
        "@SITE        COUNTRY          LAT     LONG SCS FAMILY",
        " Generic     Generic          -99      -99 Generic",
        "@ SCOM  SALB  SLU1  SLDR  SLRO  SLNF  SLPF  SMHB  SMPX  SMKE",
        "   -99  0.11   6.0  0.20  87.0  1.00  1.00 IB001 IB001 IB001",
        "@  SLB  SLMH  SLLL  SDUL  SSAT  SRGF  SSKS  SBDM  SLOC  SLCL  SLSI  SLCF  SLNI  SLHW  SLHB  SCEC  SADC",
        "     5   -99 0.228 0.385 0.463 1.000   -99  1.35  1.74  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    15   -99 0.228 0.385 0.463 1.000   -99  1.35  1.74  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    30   -99 0.228 0.385 0.459 0.638   -99  1.36  1.66  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    45   -99 0.249 0.406 0.461 0.472   -99  1.36  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    60   -99 0.249 0.406 0.461 0.350   -99  1.36  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    90   -99 0.308 0.449 0.460 0.223   -99  1.37  1.09  50.0  45.0   0.0 0.110   6.5   -99   -99   -99 ",
        "   120   -99 0.207 0.341 0.460 0.122   -99  1.38  0.65  50.0  45.0   0.0 0.060   6.5   -99   -99   -99 ",
        "   150   -99 0.256 0.373 0.463 0.067   -99  1.38  0.29  50.0  45.0   0.0 0.030   6.5   -99   -99   -99 ",
        "",
        "*IB00000003  IBSNAT      SIC      60 DEFAULT - SHALLOW SILTY CLAY",
        "@SITE        COUNTRY          LAT     LONG SCS FAMILY",
        " Generic     Generic          -99      -99 Generic",
        "@ SCOM  SALB  SLU1  SLDR  SLRO  SLNF  SLPF  SMHB  SMPX  SMKE",
        "   -99  0.11   6.0  0.10  89.0  1.00  1.00 IB001 IB001 IB001",
        "@  SLB  SLMH  SLLL  SDUL  SSAT  SRGF  SSKS  SBDM  SLOC  SLCL  SLSI  SLCF  SLNI  SLHW  SLHB  SCEC  SADC",
        "     5   -99 0.228 0.385 0.463 1.000   -99  1.35  1.74  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    15   -99 0.228 0.385 0.463 1.000   -99  1.35  1.74  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    30   -99 0.228 0.385 0.459 0.638   -99  1.36  1.66  50.0  45.0   0.0 0.170   6.5   -99   -99   -99 ",
        "    45   -99 0.249 0.406 0.461 0.472   -99  1.36  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        "    60   -99 0.249 0.406 0.461 0.350   -99  1.36  1.45  50.0  45.0   0.0 0.140   6.5   -99   -99   -99 ",
        ""),
      test_sol
      )

    sol <- DSSAT::read_sol(test_sol)

file.remove(test_sol)

  info_prefix <- "read_sol() Three profiles"
    actual <- sol
    char_cols <- c("PEDON", "SOURCE", "TEXTURE", "DESCRIPTION",
                  "SITE", "COUNTRY", "SCS FAMILY", "SCOM",
                  "SMHB", "SMPX", "SMKE", "SLMH")
    list_cols <- c("SLB", "SLMH", "SLLL", "SDUL", "SSAT",
                  "SRGF", "SSKS", "SBDM", "SLOC", "SLCL",
                  "SLSI", "SLCF", "SLNI", "SLHW", "SLHB",
                  "SCEC", "SADC")
    expected_vals <- list(PEDON = c("IB00000001", "IB00000002",
                                   "IB00000003"),
                         SOURCE = c("IBSNAT", "IBSNAT", "IBSNAT"),
                         TEXTURE = c("SIC", "SIC", "SIC"),
                         DEPTH = c(210, 150, 60),
                         DESCRIPTION = c("DEFAULT - DEEP SILTY CLAY",
                                         "DEFAULT - MEDIUM SILTY CLAY",
                                         "DEFAULT - SHALLOW SILTY CLAY"),
                         SITE = c("Generic", "Generic", "Generic"),
                         COUNTRY = c("Generic", "Generic", "Generic"),
                         LAT = c(NA_real_, NA_real_, NA_real_),
                         LONG = c(NA_real_, NA_real_, NA_real_),
                         "SCS FAMILY" = c("Generic", "Generic",
                                          "Generic"),
                         SCOM = c(NA_character_, NA_character_,
                                  NA_character_),
                         SALB = c(0.11, 0.11, 0.11),
                         SLU1 = c(6, 6, 6),
                         SLDR = c(0.3, 0.2, 0.1),
                         SLRO = c(85, 87, 89),
                         SLNF = c(1, 1, 1),
                         SLPF = c(1, 1, 1),
                         SMHB = c("IB001", "IB001", "IB001"),
                         SMPX = c("IB001", "IB001", "IB001"),
                         SMKE = c("IB001", "IB001", "IB001"),
                         SLB = list(c(5, 15, 30, 45, 60, 90,
                                      120, 150, 180, 210),
                                    c(5, 15, 30, 45, 60, 90,
                                      120, 150),
                                    c(5, 15, 30, 45, 60)),
                         SLMH = list(c(NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_),
                                     c(NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_, NA_character_),
                                     c(NA_character_, NA_character_,
                                       NA_character_, NA_character_,
                                       NA_character_)),
                         SLLL = list(c(0.228, 0.228, 0.249, 0.249,
                                       0.249, 0.308, 0.207, 0.243,
                                       0.259, 0.259),
                                     c(0.228, 0.228, 0.228, 0.249,
                                       0.249, 0.308, 0.207, 0.256),
                                     c(0.228, 0.228, 0.228, 0.249,
                                       0.249)),
                         SDUL = list(c(0.385, 0.385, 0.406,
                                       0.406, 0.406, 0.456,
                                       0.341, 0.365, 0.361,
                                       0.361),
                                     c(0.385, 0.385, 0.385,
                                       0.406, 0.406, 0.449,
                                       0.341, 0.373),
                                     c(0.385, 0.385, 0.385,
                                       0.406, 0.406)),
                         SSAT = list(c(0.481, 0.481, 0.482,
                                       0.465, 0.465, 0.468,
                                       0.452, 0.455, 0.457,
                                       0.457),
                                     c(0.463, 0.463, 0.459,
                                       0.461, 0.461, 0.46,
                                       0.46, 0.463),
                                     c(0.463, 0.463, 0.459,
                                       0.461, 0.461)),
                         SRGF = list(c(1, 1, 0.638, 0.472,
                                       0.35, 0.223, 0.122,
                                       0.067, 0.037, 0.02),
                                     c(1, 1, 0.638, 0.472,
                                       0.35, 0.223, 0.122,
                                       0.067),
                                     c(1, 1, 0.638, 0.472,
                                       0.35)),
                         SSKS = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_)),
                         SBDM = list(c(1.3, 1.3, 1.3, 1.35,
                                       1.35, 1.35, 1.4, 1.4,
                                       1.4, 1.4),
                                     c(1.35, 1.35, 1.36, 1.36,
                                       1.36, 1.37, 1.38, 1.38),
                                     c(1.35, 1.35, 1.36, 1.36,
                                       1.36)),
                         SLOC = list(c(1.75, 1.75, 1.6, 1.45,
                                       1.45, 1.1, 0.65, 0.3,
                                       0.1, 0.01),
                                     c(1.74, 1.74, 1.66, 1.45,
                                       1.45, 1.09, 0.65, 0.29),
                                     c(1.74, 1.74, 1.66, 1.45,
                                       1.45)),
                         SLCL = list(c(50, 50, 50, 50, 50, 50,
                                       50, 50, 50, 50),
                                     c(50, 50, 50, 50, 50, 50,
                                       50, 50),
                                     c(50, 50, 50, 50, 50)),
                         SLSI = list(c(45, 45, 45, 45, 45, 45,
                                       45, 45, 45, 45),
                                     c(45, 45, 45, 45, 45, 45,
                                       45, 45),
                                     c(45, 45, 45, 45, 45)),
                         SLCF = list(c(0, 0, 0, 0, 0, 0, 0,
                                       0, 0, 0),
                                     c(0, 0, 0, 0, 0, 0,
                                       0, 0),
                                     c(0, 0, 0, 0, 0)),
                         SLNI = list(c(0.17, 0.17, 0.17,
                                       0.14, 0.14, 0.11,
                                       0.06, 0.03, 0.01, 0),
                                     c(0.17, 0.17, 0.17,
                                       0.14, 0.14, 0.11,
                                       0.06, 0.03),
                                     c(0.17, 0.17, 0.17,
                                       0.14, 0.14)),
                         SLHW = list(c(6.5, 6.5, 6.5, 6.5,
                                       6.5, 6.5, 6.5, 6.5,
                                       6.5, 6.5),
                                     c(6.5, 6.5, 6.5, 6.5,
                                       6.5, 6.5, 6.5, 6.5),
                                     c(6.5, 6.5, 6.5, 6.5,
                                       6.5)),
                         SLHB = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_)),
                         SCEC = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_)),
                         SADC = list(c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_, NA_real_),
                                     c(NA_real_, NA_real_,
                                       NA_real_, NA_real_,
                                       NA_real_)))

  # Check for all missing variables
  for(nm in names(actual)){
    if("missing" %in% objects() && nm %in% missing){
      if(exists("char_cols") && nm %in% char_cols){
        na_val <- NA_character_
      }else if("date_cols" %in% objects() && !is.null(date_cols) && nm == date_cols){
        na_val <- as.POSIXct(NA, tz="UTC")
      }else{
        na_val <- NA_real_
      }
      if("list_cols" %in% objects() && !is.null(list_cols) && nm %in% list_cols){
        expect_equal(unlist(actual[[nm]]),
                               rep(na_val, length(unlist(actual[[nm]]))),
                               info = paste0(info_prefix, ": ", nm))
      }else{
        expect_equal(actual[[nm]],
                     rep(na_val, nrow(actual)),
                     info = paste0(info_prefix, ": ", nm))
      }
    }
  }

  # Check for specific expected values
  if("expected_vals" %in% objects() && !is.null(expected_vals)){
    for(nm in names(expected_vals)){
        expect_equal(actual[[nm]],
                     expected_vals[[nm]],
                     info = paste0(info_prefix, ": ", nm))
    }
  }

  # Check list column type and dimensions
  if("list_cols" %in% objects() && !is.null(list_cols)){
    for(nm in list_cols){
        expect_true(is.list(actual[[nm]]),
                    info = paste0(info_prefix, ": ", nm))
      if(exists("list_col_length") && !is.null(list_col_length)){
        if(length(list_col_length) == 1){
          list_col_length <- rep(list_col_length, length(actual[[nm]]))
        }
        for(i in 1:length(actual[[nm]])){
            expect_equal(length(unlist(actual[[nm]][i])),
                       list_col_length[i],
                       info = paste0(info_prefix, nm))
        }
      }
      if("list_col_groups" %in% objects() && !is.null(list_col_groups)){
        for(i in 1:nrow(actual)){
          for(g in 1:length(list_col_groups)){
            length_range <- range(
              sapply(actual[i,][list_col_groups[[g]]],
                     function(x)length(unlist(x)))
              )
            lbl = paste0(paste0(list_col_groups[[g]],collapse = ", ")," - row ", i)
              expect_equal(length_range[1],
                         length_range[2],
                         info = paste0(info_prefix, ": ", lbl))
          }
        }
      }
    }
  }
  for(arg in c("char_cols", "list_cols", "date_cols",
               "missing", "list_col_length",
               "list_col_groups", "expected_vals")){
    if(arg %in% objects()) rm(list = arg)
  }

# "read_sol() TEST.SOL"

  rds_file <- system.file("tinytest/test_data/SOL/TEST_SOL.rds",
                          package = "DSSAT")

  sol_file <- file.path(tempdir(), "TEST.SOL")

  write(readRDS(rds_file), sol_file)

  sol <- DSSAT::read_sol(sol_file)

  file.remove(sol_file)

  info_prefix <- "read_sol() TEST.SOL"
    actual <- sol
    `char_cols` <- c("PEDON", "SOURCE", "TEXTURE", "DESCRIPTION",
                    "SITE", "COUNTRY", "SCS FAMILY", "SCOM", "SMHB",
                    "SMPX", "SMKE", "SLMH")
    `list_cols` <- c("SLB", "SLMH", "SLLL", "SDUL", "SSAT", "SRGF",
                    "SSKS", "SBDM", "SLOC", "SLCL", "SLSI", "SLCF",
                    "SLNI", "SLHW", "SLHB", "SCEC", "SADC", "SLPX",
                    "SLPT", "SLPO", "CACO3", "SLAL", "SLFE", "SLMN",
                    "SLBS", "SLPA", "SLPB", "SLKE", "SLMG", "SLNA",
                    "SLSU", "SLEC", "SLCA", "ALFVG", "MVG", "NVG",
                    "WCRES")
    expected_vals <- list(`PEDON` = c("AGSP209115", "BRPA080004",
                                     "CORD860001", "ET04197341",
                                     "GAGI810014", "GH00970001",
                                     "GHWA040001", "HC_GEN0001",
                                     "IRRI200701", "IUBF950107",
                                     "KZ01826030", "KZ02422280",
                                     "QUKI100004", "SUMO930001",
                                     "SUMO930002", "UFGA950002",
                                     "WI_CMFI002", "WI_CMZR031",
                                     "WI_GLPH024", "WI_SNET011",
                                     "WI_VRIN047"),
                         `SOURCE` = c("EUROFINS", "UFRA", "SPAIN",
                                      "ETH    Cl", "Perkinsetal",
                                      "Naab site1D", "unknown", "IFPRI",
                                      "IRRI      -", "AMES, IA",
                                      "KAZ", "KAZ", "-99", "Morogoro,TZ",
                                      "MOROGORO, T", "Alachua cou",
                                      "WISE", "WISE", "WISE", "WISE",
                                      "WISE"),
                         `TEXTURE` = c("SIC", "C", "SALO", "yLoam",
                                       "LS", "SALO", "LS", "C", "9",
                                       "", "Loam", "Loam", "C", "SACLL",
                                       "SALO", "ty", "SIL", "L",
                                       "SIC", "CL", "SL"),
                         `DEPTH` = c(12, 30, 195, 200, 183, 9,
                                     90, 180, 211, 0, 200, 200, 100,
                                     100, 100, 20, 120, 108, 87,
                                     130, 103),
                         `DESCRIPTION` = c("ALFALFA, HYDRANT 209b",
                                           "Oxisol", "DEFAULT - DEEP SANDY LOAM",
                                           "ISRIC soilgrids + HC27",
                                           "Tifton loamy sand", "SAVANNA OCHROSOL",
                                           "Loamy Silt", "HarvestChoice HC27, Clay HF180",
                                           "Nanjing - use default shallow silty clay for base",
                                           "Clarion, well drained",
                                           "ISRIC soilgrids + HC27",
                                           "Almaty", "Kingaroy",
                                           "ALFSOL, ISOHYPERTHEMIC,  TYPIC TROPUSTIC",
                                           "OXISOL ISOHYPERTHEMIC, ARIDIC  TROPUSTIC",
                                           "Millhopper Fine Sand (Compacted layer)",
                                           "WISE DATABASE, SOIL FI002",
                                           "WISE DATABASE, SOIL ZR031",
                                           "WISE DATABASE, SOIL PH024",
                                           "WISE DATABASE, SOIL ET011",
                                           "WISE DATABASE, SOIL IN047"
                         ),
                         `SITE` = c("LA VIOLADA", "Paragominas",
                                    "Generic", NA, "GibbsFarm", "NYANKPALA",
                                    "Wa", NA, "Nanjing", NA, NA,
                                    NA, "Kingaroy Aus", "MOROGRORO",
                                    "MOROGORO", "-99         -",
                                    NA, NA, NA, NA, NA),
                         `COUNTRY` = c("SPAIN", "Brazil", "Generic",
                                       "ET", "USA", "GHANA", "Ghana",
                                       "Generic", "China", NA, "KZ",
                                       "KZ", "tralia", "TANZANIA",
                                       "TANZANIA", "99", "Finland",
                                       "Democratic", "Philippines",
                                       "Ethiopia", "India"),
                         `LAT` = c(41, -3, NA, 9.042, 31.43, 9.416,
                                   10.005, 0, NA, 42, 54.792, 43.292,
                                   NA, -6.5, -6.5, NA, 61.35, -3.867,
                                   14.433, 9.017, 17.667),
                         `LONG` = c(0, -47, NA, 38.458, -83.58,
                                    0.966, -2.496, 0, NA, NA, 69.208,
                                    76.709, NA, 37.3, 37.3, NA, 24.267,
                                    27.3, 121.45, 40.167, 78.083),
                         `SCS FAMILY` = c("PETROCALCIC CALCIXEREPT",
                                          "Oxisols", "Generic", "HC_GEN0011",
                                          "Fine-loamy,siliceous,thermic Plinthic Paleudults",
                                          "PALEUSTALF", "unknown",
                                          "Clay Deep (HF)", "", "",
                                          "HC_GEN0011", "HC_GEN0011",
                                          "Brown Ferrasol", "DARK GRAYISH BROWN  SAND CLAY LOAM ISOHYPERTHEMIC,TYPIC TROPUSTIC",
                                          "REDDISH BROWN SAND LOAM",
                                          "Loamy, siliceous hyperthermic Grossarenic Paleudults",
                                          "Dystric Cambisol (CMd)",
                                          "Ferralic Cambisol (CMo)",
                                          "Mollic Gleysol (GLm)",
                                          "Haplic Solonetz (SNh)",
                                          "Calcic Vertisol (VRk)"
                         ),
                         `SCOM` = c("G", "Y", NA, "BK", "BN", "BN",
                                    "BN", "BK", NA, NA, "BK", "BK",
                                    "BL", "2.5YR", "2.5YR", NA, "BN",
                                    "BK", "BK", "BN", "BN"),
                         `SALB` = c(0.13, 0.17, 0.21, 0.1, 0.13,
                                    0.13, 0.13, 0.05, 0.11, 0.16,
                                    0.1, 0.1, 0.12, 0.13, 0.12, 0.18,
                                    0.13, 0.09, 0.09, 0.13, 0.13),
                         `SLU1` = c(6, 6, 6, 6, 7.3, 6, 6, 8, NA,
                                    9.8, 6, 6, 6, 6, 6, 5, 10.16,
                                    9.52, 7.94, 10.32, 9.6),
                         `SLDR` = c(0.6, 0.4, 0.6, 0.5, 0.6, 0.4,
                                    0.6, 0.2, 0.1, 0.55, 0.5, 0.5,
                                    0.3, 0.35, 0.5, 0.5, 0.4, 0.6,
                                    0.01, 0.4, 0.4),
                         `SLRO` = c(73, 73, 60, 75, 76, 76, 73,
                                    85, 89, 91, 75, 75, 73, 84, 84,
                                    66, 75, 75, 73, 75, 75),
                         `SLNF` = c(1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    0.9, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1),
                         `SLPF` = c(1, 1, 1, 1, 0.93, 0.84, 1,
                                    1, 1, 1, 1, 1, 1, 0.88, 0.8,
                                    0.92, 1, 1, 1, 1, 1),
                         `SMHB` = c("IB001", "IB001", "IB001",
                                    "SA001", "IB001", "IB001", "IB001",
                                    "SA001", NA, "IB001", "SA001",
                                    "SA001", "IB00", "IB001", "IB001",
                                    "IB001", "SA001", "SA001", "SA001",
                                    "SA001", "SA001"),
                         `SMPX` = c("IB001", "IB001", "IB001",
                                    "SA001", "IB001", "IB001", "SA002",
                                    "SA001", NA, "IB001", "SA001",
                                    "SA001", "IB00", "IB002", "IB002",
                                    "IB001", "SA001", "SA001", "SA001",
                                    "SA001", "SA001"),
                         `SMKE` = c("IB001", "IB001", "IB001",
                                    "SA001", "IB001", "IB001", "IB001",
                                    "SA001", NA, "IB001", "SA001",
                                    "SA001", "IB00", "IB001", "IB001",
                                    "IB001", "SA001", "SA001", "SA001",
                                    "SA001", "SA001"),
                         `SLB` = list(c(30, 60, 90, 120),
                                      c(5,
                                        15, 30, 40, 50, 60, 70, 100, 125,
                                        150, 160, 300),
                                      c(5, 15, 45, 75,
                                        105, 135, 165, 195),
                                      c(5, 15,
                                        30, 60, 100, 200),
                                      c(30, 51, 76,
                                        104, 135, 183),
                                      c(5, 15, 30, 45,
                                        60, 90),
                                      c(5, 20, 40, 60, 90),
                                      c(10, 30, 60, 90, 120, 150,
                                        180),
                                      c(10, 20, 30, 40, 50,
                                        60),
                                      c(5, 15, 25, 30, 50,
                                        70, 110),
                                      c(5, 15, 30, 60,
                                        100, 200),
                                      c(5, 15, 30, 60,
                                        100, 200),
                                      c(5, 10, 20, 30,
                                        40, 50, 60, 70, 80, 90),
                                      c(10,
                                        20, 30, 40, 50, 60, 70, 80,
                                        90, 100),
                                      c(10, 20, 30, 40,
                                        50, 60, 70, 80, 90, 100),
                                      c(15, 33, 89, 147, 157, 175,
                                        203),
                                      c(4, 18, 35, 50, 80,
                                        120),
                                      c(17, 42, 68, 86, 108
                                      ),
                                      c(18, 26, 48, 64, 87),
                                      c(30, 40, 55, 84, 130),
                                      c(2,
                                        19, 38, 66, 86, 103)),
                         `SLMH` = list(c(NA_character_, NA_character_,
                                         NA_character_, NA_character_),
                                       c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_
                                       ),
                                       c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_
                                       ),
                                       c("A", "A", "AB", "BA",
                                         "B", "BC"),
                                       c("Apc", "Btc1",
                                         "Btc2", "Btv1", "Btv2", "Bt"
                                       ),
                                       c("AP", "AP", "B1", "B2",
                                         "B3", "B3"),
                                       c("A", "A",
                                         "B", "B", "B"),
                                       c("A", "AB",
                                         "BA", "B", "BC1", "BC2",
                                         "C"),
                                       c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_
                                       ),
                                       c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_),
                                       c("A", "A",
                                         "AB", "BA", "B", "BC"),
                                       c("A",
                                         "A", "AB", "BA", "B", "BC"
                                       ),
                                       c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_
                                       ),
                                       c("Ap", "Ap", "AB", "B1",
                                         "B1", "B2", "B2", "B2", "B3",
                                         "B3"),
                                       c("Ap", "Ap", "E",
                                         "AB", "B1", "B2", "B2", "B2",
                                         "B3", "B3"),
                                       c(NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_
                                       ),
                                       c("-", "-", "-", "-",
                                         "-", "-"),
                                       c("A11", "A12",
                                         "A3", "(Bc)", "B/C"),
                                       c("Apg1",
                                         "Apg2", "Bg2", "Bg2", "Btg"
                                       ),
                                       c("-", "-", "-", "-",
                                         "-"),
                                       c("Ap1", "Ap2", "A12",
                                         "A13", "A14", "ACca")),
                         `SLLL` = list(c(0.117, 0.22, 0.219, 0.223
                         ),
                         c(0.24, 0.25, 0.301, 0.307,
                           0.307, 0.307, 0.307, 0.307, 0.307,
                           0.307, 0.307, 0.307),
                         c(0.07,
                           0.07, 0.07, 0.07, 0.07, 0.07,
                           0.07, 0.07),
                         c(0.233, 0.244,
                           0.259, 0.271, 0.272, 0.263),
                         c(0.067, 0.148, 0.192, 0.194,
                           0.177, 0.19),
                         c(0.095, 0.06,
                           0.06, 0.105, 0.12, 0.13),
                         c(0.085, 0.085, 0.122, 0.124,
                           0.059),
                         c(0.21, 0.23, 0.26,
                           0.29, 0.32, 0.36, 0.4),
                         c(0.228,
                           0.228, 0.228, 0.249, 0.249,
                           0.249),
                         c(0.181, 0.177, 0.192,
                           0.204, 0.15, 0.15, 0.127),
                         c(0.103, 0.113, 0.127, 0.141,
                           0.14, 0.13),
                         c(0.113, 0.124,
                           0.137, 0.151, 0.151, 0.141
                         ),
                         c(0.18, 0.18, 0.19, 0.2,
                           0.21, 0.21, 0.22, 0.22, 0.22,
                           0.22),
                         c(0.081, 0.097, 0.102,
                           0.101, 0.082, 0.129, 0.096,
                           0.056, 0.056, 0.056),
                         c(0.091,
                           0.066, 0.104, 0.125, 0.088,
                           0.165, 0.053, 0.159, 0.154,
                           0.089),
                         c(0.024, 0.016, 0.011,
                           0.021, 0.022, 0.07, 0.105
                         ),
                         c(0.155, 0.148, 0.184,
                           0.163, 0.188, 0.188),
                         c(0.073,
                           0.11, 0.172, 0.213, 0.21),
                         c(0.183, 0.185, 0.21, 0.211,
                           0.229),
                         c(0.162, 0.214, 0.226,
                           0.124, 0.241),
                         c(0.124, 0.124,
                           0.116, 0.121, 0.121, 0.134
                         )),
                         `SDUL` = list(c(0.433, 0.449, 0.47, 0.464
                         ),
                         c(0.488, 0.455, 0.457, 0.434,
                           0.434, 0.434, 0.434, 0.434, 0.434,
                           0.434, 0.434, 0.434),
                         c(0.27,
                           0.27, 0.25, 0.22, 0.2, 0.195,
                           0.18, 0.22),
                         c(0.355, 0.367,
                           0.383, 0.396, 0.396, 0.386),
                         c(0.141, 0.241, 0.276, 0.278,
                           0.263, 0.273),
                         c(0.257, 0.227,
                           0.228, 0.229, 0.205, 0.2),
                         c(0.155, 0.155, 0.19, 0.17,
                           0.079),
                         c(0.39, 0.41, 0.415,
                           0.42, 0.425, 0.43, 0.43),
                         c(0.385, 0.385, 0.385, 0.406,
                           0.406, 0.406),
                         c(0.239, 0.241,
                           0.266, 0.261, 0.262, 0.262,
                           0.247),
                         c(0.229, 0.24, 0.254,
                           0.268, 0.266, 0.253),
                         c(0.259,
                           0.271, 0.284, 0.298, 0.297,
                           0.284),
                         c(0.31, 0.31, 0.29,
                           0.28, 0.32, 0.29, 0.29, 0.29,
                           0.29, 0.29),
                         c(0.217, 0.233,
                           0.238, 0.237, 0.218, 0.265,
                           0.232, 0.192, 0.192, 0.192
                         ),
                         c(0.227, 0.202, 0.24,
                           0.261, 0.224, 0.301, 0.189,
                           0.295, 0.29, 0.225),
                         c(0.161,
                           0.104, 0.113, 0.08, 0.101,
                           0.196, 0.248),
                         c(0.337, 0.337,
                           0.308, 0.292, 0.317, 0.317
                         ),
                         c(0.189, 0.269, 0.276,
                           0.319, 0.312),
                         c(0.353, 0.354,
                           0.347, 0.313, 0.306),
                         c(0.298,
                           0.365, 0.382, 0.286, 0.374
                         ),
                         c(0.234, 0.234, 0.247,
                           0.253, 0.253, 0.254)),
                         `SSAT` = list(c(0.497, 0.553, 0.507, 0.546
                         ),
                         c(0.582, 0.553, 0.502, 0.489,
                           0.489, 0.489, 0.489, 0.489, 0.489,
                           0.489, 0.489, 0.489),
                         c(0.39,
                           0.39, 0.35, 0.32, 0.3, 0.285,
                           0.28, 0.31),
                         c(0.436, 0.442,
                           0.451, 0.459, 0.458, 0.451),
                         c(0.323, 0.309, 0.384, 0.314,
                           0.302, 0.329),
                         c(0.359, 0.359,
                           0.34, 0.342, 0.342, 0.347
                         ),
                         c(0.383, 0.383, 0.362,
                           0.204, 0.088),
                         c(0.44, 0.467,
                           0.467, 0.47, 0.475, 0.48,
                           0.48),
                         c(0.463, 0.459, 0.459,
                           0.461, 0.461, 0.461),
                         c(0.314,
                           0.323, 0.35, 0.325, 0.334,
                           0.329, 0.322),
                         c(0.39, 0.392,
                           0.396, 0.4, 0.399, 0.395),
                         c(0.395, 0.399, 0.405, 0.411,
                           0.411, 0.405),
                         c(0.49, 0.49,
                           0.51, 0.51, 0.52, 0.55, 0.55,
                           0.55, 0.55, 0.55),
                         c(0.434,
                           0.382, 0.382, 0.382, 0.399,
                           0.295, 0.382, 0.451, 0.451,
                           0.451),
                         c(0.382, 0.382, 0.399,
                           0.365, 0.382, 0.347, 0.417,
                           0.417, 0.434, 0.472),
                         c(0.487,
                           0.407, 0.408, 0.385, 0.404,
                           0.374, 0.351),
                         c(0.574, 0.557,
                           0.477, 0.473, 0.481, 0.575
                         ),
                         c(0.445, 0.449, 0.455,
                           0.452, 0.456),
                         c(0.601, 0.583,
                           0.597, 0.513, 0.533),
                         c(0.475,
                           0.509, 0.51, 0.436, 0.512
                         ),
                         c(0.409, 0.404, 0.399,
                           0.398, 0.398, 0.395)),
                         `SRGF` = list(c(1, 0.407, 0.223, 0.122
                         ),
                         c(1, 1, 0.42, 0.34, 0.23,
                           0.2, 0.18, 0.16, 0.04, 0.04,
                           0, 0),
                         c(1, 1, 0.6, 0.3, 0.22,
                           0.22, 0.22, 0.22),
                         c(1, 0.85,
                           0.7, 0.5, 0.38, 0.05),
                         c(1, 0.7,
                           0.5, 0.3, 0.2, 0.1),
                         c(1, 1,
                           0.7, 0.35, 0.25, 0.05),
                         c(1,
                           1, 0.549, 0.368, 0),
                         c(1, 0.65,
                           0.38, 0.22, 0.12, 0.08, 0.03),
                         c(1, 0.9, 0.64, 0.53, 0.43,
                           0.35),
                         c(1, 1, 0.9, 0.8,
                           0.6, 0.4, 0.3),
                         c(1, 0.85,
                           0.7, 0.5, 0.38, 0.05),
                         c(1,
                           0.85, 0.7, 0.5, 0.38, 0.05
                         ),
                         c(1, 1, 0.9, 0.8, 0.6,
                           0.4, 0.3, 0.06, 0.03, 0.01
                         ),
                         c(1, 0.819, 0.268, 0.268,
                           0.138, 0.128, 0.128, 0.128,
                           0.1, 0.1),
                         c(1, 0.841, 0.499,
                           0.338, 0.166, 0.134, 0.019,
                           0.016, 0.012, 0.003),
                         c(1,
                           0.3, 0, 0.05, 0, 0, 0),
                         c(1,
                           1, 0.59, 0.43, 0.27, 0.14
                         ),
                         c(1, 0.55, 0.33, 0.21,
                           0.14),
                         c(1, 0.64, 0.48, 0.33,
                           0.22),
                         c(0.74, 0.5, 0.39,
                           0.25, 0.12),
                         c(1, 1, 0.57,
                           0.35, 0.22, 0.15)),
                         `SSKS` = list(c(0.09, 0.68, 0.68, 0.68
                         ),
                         c(0.5, 0.5, 0.5, 0.5, 0.5,
                           0.5, 0.5, 0.5, 0.5, 0.5, 0.5,
                           0.5),
                         c(NA_real_, NA_real_, NA_real_,
                           NA_real_, NA_real_, NA_real_,
                           NA_real_, NA_real_),
                         c(0.14,
                           0.11, 0.08, 0.06, 0.06, 0.07),
                         c(5.36, 7.06, 3.59, 0.11,
                           0.03, 0.03),
                         c(NA_real_,
                           NA_real_, NA_real_, NA_real_,
                           NA_real_, NA_real_),
                         c(2.59,
                           2.59, 2.59, 0.12, 0.06),
                         c(1, 0.8, 0.6, 0.5, 0.4,
                           0.3, 0.2),
                         c(0.014, 0.09,
                           0.725, 0.725, 0.04, 0.105
                         ),
                         c(0.81, 2.86, 5.28, 3.71,
                           4.99, 7.08, 10.38),
                         c(1.18,
                           1, 0.79, 0.63, 0.65, 0.78
                         ),
                         c(0.75, 0.63, 0.51, 0.41,
                           0.42, 0.51),
                         c(0.1, 0.1,
                           0.1, 0.1, 0.1, 0.05, 0.03,
                           0.01, 0.01, 0.01),
                         c(5.01,
                           6.98, 7.08, 5.01, 5.01, 5.01,
                           5.02, 6.99, 6.25, 6.84),
                         c(16.56, 16.56, 16.16, 16.56,
                           6.98, 5.98, 5.98, 16.55,
                           5.99, 5.99),
                         c(9.3, 24.4,
                           0.001, 32.9, 19.8, 4, 0.9
                         ),
                         c(1.96, 1.71, 1.24, 1.42,
                           0.99, 2.36),
                         c(6.85, 1.77,
                           1.52, 0.66, 0.8),
                         c(1.78,
                           1.53, 1.89, 3.25, 4.31),
                         c(1.31, 0.6, 0.45, 1.14,
                           0.53),
                         c(2.4, 2.27, 1.69,
                           1.51, 1.51, 1.42)),
                         `SBDM` = list(c(1.5, 1.5, 1.5, 1.4),
                                       c(1.02,
                                         1.09, 1.3, 1.32, 1.32, 1.32,
                                         1.32, 1.32, 1.32, 1.32, 1.32,
                                         1.32),
                                       c(1.61, 1.61, 1.61, 1.61,
                                         1.61, 1.62, 1.62, 1.62),
                                       c(1.08,
                                         1.1, 1.13, 1.18, 1.24, 1.3),
                                       c(1.73, 1.76, 1.57, 1.77,
                                         1.8, 1.73),
                                       c(1.39, 1.39,
                                         1.59, 1.59, 1.63, 1.45),
                                       c(1.54, 1.54, 1.57, 1.52,
                                         1.38),
                                       c(1.2, 1.2, 1.2, 1.2,
                                         1.2, 1.2, 1.2),
                                       c(1.28, 1.28,
                                         1.68, 1.68, 1.68, 1.68),
                                       c(1.52, 1.47, 1.39, 1.53,
                                         1.48, 1.51, 1.5),
                                       c(1.28,
                                         1.3, 1.33, 1.38, 1.44, 1.5
                                       ),
                                       c(1.32, 1.34, 1.37, 1.42,
                                         1.48, 1.53),
                                       c(1.23, 1.23,
                                         1.3, 1.33, 1.33, 1.33, 1.33,
                                         1.33, 1.33, 1.31),
                                       c(1.4,
                                         1.55, 1.55, 1.55, 1.5, 1.5,
                                         1.55, 1.35, 1.35, 1.35),
                                       c(1.55, 1.55, 1.5, 1.6, 1.55,
                                         1.65, 1.45, 1.45, 1.4, 1.29
                                       ),
                                       c(1.36, 1.57, 1.57, 1.63,
                                         1.58, 1.66, 1.72),
                                       c(0.88,
                                         1.02, 1.32, 1.35, 1.34, 1.08
                                       ),
                                       c(1.39, 1.39, 1.4, 1.41,
                                         1.4),
                                       c(0.95, 1.01, 1.01,
                                         0.97, 0.88),
                                       c(1.35, 1.26,
                                         1.26, 1.43, 1.25),
                                       c(1.47,
                                         1.49, 1.5, 1.5, 1.5, 1.51
                                       )),
                         `SLOC` = list(c(1.8, 0.5, 0.8, 0.4),
                                       c(0.9,
                                         0.9, 0.7, 0.3, 0.3, 0.3, 0.3,
                                         0.3, 0.2, 0.2, 0.2, 0.2),
                                       c(0.7,
                                         0.7, 0.58, 0.58, 0.43, 0.26,
                                         0.12, 0.04),
                                       c(1.4, 0.87, 0.69,
                                         0.63, 0.61, 0.6),
                                       c(0.74, 1.08,
                                         0.34, 0.19, 0.25, 0.04),
                                       c(0.41,
                                         0.41, 0.32, 0.28, 0.28, 0.17),
                                       c(0.49, 0.49, 0.48, 0.51,
                                         0.43),
                                       c(1.4, 0.87, 0.69,
                                         0.63, 0.6, 0.53, 0.3),
                                       c(1.65,
                                         1.23, 0.78, 0.78, 0.78, 0.39
                                       ),
                                       c(2.03, 2.03, 2.03, 2.03,
                                         0.44, 0.44, 0.15),
                                       c(1.4,
                                         0.87, 0.69, 0.63, 0.61, 0.6
                                       ),
                                       c(1.58, 1.34, 1.02, 0.66,
                                         0.37, 0.21),
                                       c(1.5, 1.5,
                                         1.4, 1.1, 0.5, 0.3, 0.1,
                                         0.1, 0.1, 0.1),
                                       c(2.27, 2.27,
                                         1.15, 1.64, 1.8, 1.8, 1.6,
                                         1.6, 0.55, 0.55),
                                       c(1.13,
                                         1.13, 0.76, 0.76, 0.7, 0.7,
                                         0.4, 0.33, 0.33, 0.3),
                                       c(0.73,
                                         0.34, 0.19, 0.07, 0.07, 0.1,
                                         0.12),
                                       c(12.37, 4.78, 3.5,
                                         0.56, NA, NA),
                                       c(0.88, 0.57,
                                         0.37, 0.25, 0.23),
                                       c(3.99,
                                         3.07, 0.54, 0.36, 0.31),
                                       c(0.19, NA, NA, NA, NA),
                                       c(0.71, 0.59, 0.48, 0.47,
                                         0.47, 0.4)),
                         `SLCL` = list(c(42.9, 6.4, 24.4, 5.5),
                                       c(77, 78, 84.1, 84.3, 84.3,
                                         84.3, 84.3, 84.3, 84.3, 84.3,
                                         84.3, 84.3),
                                       c(10, 10, 10,
                                         10, 10, 10, 10, 10),
                                       c(38.96,
                                         40.89, 43.46, 45.62, 45.65,
                                         44.13),
                                       c(4.2, 18.6, 31.3,
                                         32.6, 28.8, 32.5),
                                       c(7, 7,
                                         11.3, 15, 15.5, 18.2),
                                       c(7.5,
                                         7.5, 14.5, 40.9, 52.9),
                                       c(60,
                                         60, 60, 60, 60, 60, 60),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(18, 18, 22, 22, 21,
                                         21, 17),
                                       c(16.82, 18.56,
                                         20.93, 23.31, 23.21, 21.46
                                       ),
                                       c(18.28, 20.15, 22.53,
                                         24.94, 24.91, 23.13),
                                       c(50,
                                         50, 55, 60, 60, 60, 60, 60,
                                         60, 28),
                                       c(55, 57, 59, 58,
                                         58, 58, 59, 48, 48, 48),
                                       c(60, 65, 65, 53, 65, 55,
                                         68, 40, 35, 38),
                                       c(2.1, 2.3,
                                         2.4, 1.9, 4.1, 16.4, 28.6
                                       ),
                                       c(27, 24, 26, 30, 33,
                                         34),
                                       c(19, 24, 29, 30, 31
                                       ),
                                       c(51, 51, 49, 41, 26),
                                       c(29, 41, 44, 31, 39),
                                       c(20,
                                         20, 20, 20, 20, 20)),
                         `SLSI` = list(c(51, 70, 60, 67),
                                       c(12.4,
                                         12.7, 11.4, 12.1, 12.1, 12.1,
                                         12.1, 12.1, 12.1, 12.1, 12.1,
                                         12.1),
                                       c(30, 30, 30, 30, 30,
                                         30, 30, 30),
                                       c(22.53, 21.77,
                                         20.73, 19.7, 19.05, 18.83),
                                       c(10.9,
                                         11.9, 12.9, 13.9, 15.6, 15.6),
                                       c(18.2, 18.2, 24.5, 27.9,
                                         27.5, 27.7),
                                       c(8.3, 8.3,
                                         8.2, 10.7, 16.9),
                                       c(25, 25,
                                         25, 25, 25, 25, 25),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(31,
                                         31, 31, 31, 28, 28, 32),
                                       c(39.45, 38.67, 37.57, 36.22,
                                         35.65, 35.4),
                                       c(48.65, 47.88,
                                         46.91, 45.65, 45.07, 44.95
                                       ),
                                       c(17, 17, 14, 10, 10,
                                         10, 10, 10, 10, 33),
                                       c(24,
                                         23, 18, 24, 24, 23, 20, 42,
                                         42, 42),
                                       c(20, 30, 15, 13,
                                         13, 10, 15, 24, 30, 22),
                                       c(6.3, 4.1, 3.3, 3.3, 3.9,
                                         6.1, 8.6),
                                       c(68, 70, 67,
                                         61, 62, 63),
                                       c(33, 38, 37,
                                         37, 38),
                                       c(41, 40, 37, 36,
                                         28),
                                       c(44, 37, 37, 19, 39
                                       ),
                                       c(22, 22, 22, 22, 22,
                                         22)),
                         `SLCF` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0, 0, 0, 0, 0, 0,
                                         0, 0),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0, 0, 0, 0, 0, 0
                                       ),
                                       c(6.2, 6.2, 11.4, 47.1, 47.1,
                                         47.5),
                                       c(4.3, 4.3, 6.4, 49.2,
                                         80.7),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(2.5, 2.5,
                                         2.5, 2.5, 5, 7.5, 7.5),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(39, 39,
                                         39, 39, 39, 39, 39, 39, 39, 39
                                       ),
                                       c(21, 20, 23, 22, 18, 19,
                                         21, 10, 10, 10),
                                       c(20, 15, 30,
                                         24, 22, 35, 17, 36, 35, 20),
                                       c(0, 0, 0, 9, 0, 0, 0),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(78, 78, 78, 78, 78,
                                         78)),
                         `SLNI` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.08, 0.08, 0.06,
                                         0.03, 0.03, 0.03, 0.03, 0.03,
                                         0.02, 0.02, 0.02, 0.02),
                                       c(0.07,
                                         0.07, 0.058, 0.058, 0.043, 0.026,
                                         0.012, 0.004),
                                       c(0.12, 0.09,
                                         0.07, 0.06, 0.05, 0.05),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0.06,
                                         0.06, 0.06, 0.04, 0.04),
                                       c(0.12,
                                         0.07, 0.06, 0.05, 0.05, 0.04,
                                         0.03),
                                       c(0.1, 0.08, 0.05, 0.05,
                                         0.05, 0.02),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0.12,
                                         0.09, 0.07, 0.06, 0.05, 0.05),
                                       c(0.12, 0.09, 0.07, 0.06,
                                         0.05, 0.05),
                                       c(0.15, 0.15,
                                         0.14, 0.11, 0.05, 0.03, 0.01,
                                         0.01, 0.01, 0.01),
                                       c(0.09,
                                         0.09, 0.08, 0.07, 0.07, 0.07,
                                         0.09, 0.06, 0.05, 0.05),
                                       c(0.06, 0.06, 0.05, 0.06,
                                         0.05, 0.04, 0.04, 0.05, 0.05,
                                         0.03),
                                       c(0.08, 0.03, 0.02,
                                         0.01, 0.01, 0.01, 0.01),
                                       c(1.23, 0.62, 0.35, 0.07,
                                         NA, NA),
                                       c(0.06, 0.05, 0.05,
                                         0.05, 0.04),
                                       c(0.33, 0.26,
                                         0.05, 0.04, 0.03),
                                       c(0.02,
                                         NA, NA, NA, NA),
                                       c(0.07,
                                         0.06, 0.05, 0.05, 0.05, 0.04
                                       )),
                         `SLHW` = list(c(8, 8, 8, 8),
                                       c(5.8, 5.8,
                                         5.5, 4.9, 4.9, 4.9, 4.9, 4.9,
                                         4.9, 4.9, 4.9, 4.9),
                                       c(7.1, 7.1,
                                         7.1, 7.1, 7.1, 7.1, 7.1, 7.1),
                                       c(6.22, 6.29, 6.38, 6.5,
                                         6.64, 6.82),
                                       c(5.9, 4.8,
                                         5.1, 4.9, 4.8, 4.6),
                                       c(6.7,
                                         6.7, 6.4, 6.1, 6.1, 6),
                                       c(6.3,
                                         6.3, 6.3, 5.9, 6),
                                       c(6.5,
                                         6.5, 6.5, 6.5, 6.5, 6.5,
                                         6.5),
                                       c(6.5, 6.5, 6.5, 6.5,
                                         6.5, 6.5),
                                       c(6.4, 6.4, 6.4,
                                         6.4, 6.7, 6.7, 7.9),
                                       c(7.5,
                                         7.57, 7.66, 7.78, 7.91, 8.1
                                       ),
                                       c(7.91, 7.98, 8.08, 8.19,
                                         8.32, 8.5),
                                       c(7.3, 7.2, 7.5,
                                         7.9, 8.1, 8.2, 8.3, 8.5,
                                         8.5, 8.5),
                                       c(5.43, 5.09,
                                         4.25, 5.03, 5.03, 5.04, 5.13,
                                         4.83, 4.8, 4.61),
                                       c(4.56,
                                         4.5, 5.38, 5.5, 5.69, 4.59,
                                         4.23, 5.14, 4.46, 4.61),
                                       c(5.2, 5.4, 5.2, 5.3, 5.2,
                                         4.9, 5),
                                       c(4.9, 4.9, 5.5,
                                         5.8, 5.9, 6.1),
                                       c(5.2, 5,
                                         4.8, 4.9, 4.9),
                                       c(7, 6.8,
                                         6.8, 6.8, 7.2),
                                       c(8.1, NA,
                                         NA, NA, NA),
                                       c(8, 8.1, 8.1,
                                         8.1, 8.2, 8.3)),
                         `SLHB` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(5.3,
                                         5.2, 5.2, 5, 5, 5, 5.05,
                                         5, 5.1, 5.1),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(4.1,
                                         3.8, 3.8, 4.1, 4.1, 4),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SCEC` = list(c(16, 13, 11, 14),
                                       c(7.9,
                                         7.9, 7.5, 6.4, 6.4, 6.4, 6.4,
                                         6.4, 5.4, 5.4, 5.4, 5.4),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(34.6, 30.3, 29.4,
                                         30.7, 30.9, 30.8),
                                       c(2.5, 3.2,
                                         3.8, 4.5, 3.9, 4.4),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(1, 1,
                                         1, 0.9, 0.8),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(26.6, 23.3, 22.5, 23.5,
                                         23.8, 23.7),
                                       c(21.1, 18.5,
                                         17.9, 18.8, 18.9, 18.8),
                                       c(71, 71, 71, 71, 71, 71,
                                         65, 65, 65, 65),
                                       c(5.43,
                                         5.11, 3.74, 3.63, 3.33, 3.04,
                                         2.44, 2.34, 2.14, 1.82),
                                       c(8.97, 8.89, 5.65, 5.31,
                                         5.44, 3.77, 3.31, 2.03, 1.91,
                                         1.94),
                                       c(5.4, 2.9, 1.9, 1.1,
                                         2, 6, 10.4),
                                       c(41, 21.1,
                                         9.5, NA, NA, 10.4),
                                       c(6.5,
                                         6.5, 6.5, 6.3, 6.3),
                                       c(43.8,
                                         43, 43.1, 51.1, 52.6),
                                       c(28.2,
                                         NA, NA, NA, NA),
                                       c(35.7,
                                         38.3, 41.2, 42.9, 43.2, 34.8
                                       )),
                         `SADC` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLPX` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(6.12,
                                         5.85, 3.97, 2.91, 2.02, 1.47,
                                         2.05, 1.64, 1.71, 1.54),
                                       c(5.42, 5.03, 4.33, 4.01,
                                         3.31, 3.14, 2.89, 3.33, 2.42,
                                         2.41),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLPT` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(360,
                                         360, 309, 309, 309, 309,
                                         309, 143, 143, 133),
                                       c(270,
                                         270, 242, 170, 170, 170,
                                         170, 75, 75, 75),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(4, NA, NA, NA, NA),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       )),
                         `SLPO` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(280,
                                         280, 30, 30, 30, 30, 30,
                                         40, 40, 40),
                                       c(120, 120,
                                         120, 80, 80, 80, 80, 50,
                                         50, 50),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `CACO3` = list(c(NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(2.03,
                                          2.01, 1.94, 1.93, 1.93, 1.64,
                                          1.04, 0.94, 0.84, 0.52),
                                        c(1.77,
                                          1.59, 1.45, 1.31, 1.24, 1.07,
                                          0.61, 0.33, 0.34, 0.34),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(6.5, NA, NA, NA,
                                          NA),
                                        c(2.2, 3, 3.3, 3.8, 4.4,
                                          7.9)),
                         `SLAL` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0, 0, 0, 0, 0, 0,
                                         0.2, 0.2, 0.3, 0.3, 0.3, NA),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0,
                                         0.3, 0.1, 0.2, 0.7, 1.4),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(1.1,
                                         1.1, 0.8, 0.8, 0.3, 0.3,
                                         0.3, 0.3, 0.3, 0.3),
                                       c(1.5,
                                         1.5, 1.4, 1.3, 1.3, 1.3,
                                         1.3, 1.3, 1.3, 1.2),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLFE` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(1.2,
                                         1.2, 1.6, 1.1, 1.1, 1.2,
                                         1.2, 1.1, 1.1, 1),
                                       c(3.2,
                                         3.2, 3, 2.87, 2.86, 2.2,
                                         2.2, 1.29, 1.87, 1.1),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLMN` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLBS` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(43, 24, 36, NA, NA,
                                         79),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(100, 100, 100, 90, 100
                                       ),
                                       c(100, 100, 100, 100,
                                         100),
                                       c(100, 100, 100, 100,
                                         100, 100)),
                         `SLPA` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLPB` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.3, NA, NA,
                                         NA, NA),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLKE` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.1, 0.1, 0.1, 0.1,
                                         0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                                         0.1, NA),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(0.1, 0.2, 0.2, 0.1,
                                         0.1, 0),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.1, 0.1, 0.1,
                                         0.1, 0.1),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0.6,
                                         0.6, 0.3, 0.4, 0.3, 0.3,
                                         0.3, 0.3, 0.3, 0.3),
                                       c(1.6,
                                         1.6, 1.4, 1.4, 1.4, 0.4,
                                         0.4, 0.4, 0.4, 0.4),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(0.6, 0.2, 0.2, NA, NA,
                                         0.2),
                                       c(NA, 0.1, 0.1, 0.1,
                                         0.1),
                                       c(0.3, 0.2, 0.2, 0.3,
                                         0.7),
                                       c(1.2, NA, NA, NA,
                                         NA),
                                       c(0.1, 0.1, 0.1, 0.1,
                                         0.1, 0.1)),
                         `SLMG` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.3, 0.3, 0.5,
                                         0.7, 0.9),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(2.8,
                                         2.5, 1.5, 1.3, 1.1, 1.1,
                                         1.1, 1.1, 1, 1),
                                       c(5.6, 5.7,
                                         2.8, 2.6, 2.8, 2.3, 2.3,
                                         1.3, 1.2, 1.2),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(4.8, 1.8, 1.3, NA, NA,
                                         4),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(14.9, 14.6, 17.5, 19,
                                         18.5),
                                       c(2.4, NA, NA, NA,
                                         NA),
                                       c(1, 1, 1, 1, 1, 1)),
                         `SLNA` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.5, 0.5, 0.5,
                                         0.5, 0.5),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0.71,
                                         0.71, 0.61, 0.61, 0.61, 0.61,
                                         0.61, 0.65, 0.65, 0.65),
                                       c(0.19, 0.19, 0.22, 0.22,
                                         0.22, 0.22, 0.16, 0.16, 0.16,
                                         0.16),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(0.1,
                                         0.1, 0.1, NA, NA, 0.1),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(0.9, 1, 1, 1.3,
                                         1.3),
                                       c(9.1, NA, NA, NA,
                                         NA),
                                       c(0.2, 0.2, 0.2, 0.2,
                                         0.2, 0.2)),
                         `SLSU` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLEC` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(1.2, 0.7, 0.3, 0.3,
                                         NA, 0.3),
                                       c(0.9, NA, NA,
                                         NA, NA),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(15.6, NA, NA, NA, NA
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SLCA` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(3.9, 3.9, 3.9, 3.9,
                                         2.9, 2.9, 1.8, 1.8, 1.2, 1.2,
                                         1.2, NA),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_),
                                       c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(1.5, 1.5, 1.2,
                                         1.6, 1.9),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_),
                                       c(NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_
                                       ),
                                       c(12.1, 2.9, 1.8, NA,
                                         NA, 3.9),
                                       c(NA, 0.5, 0.5,
                                         0.7, 0.4),
                                       c(29.2, 28.1,
                                         28.2, 30.2, 30.5),
                                       c(53.8,
                                         NA, NA, NA, NA),
                                       c(34.4,
                                         37, 39.9, 41.6, 41.9, 33.5
                                       )),
                         `ALFVG` = list(c(NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(0.001, 0.011, 0.133, 0.133,
                                          0.047, 0.127),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        )),
                         `MVG` = list(c(NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(0.143, 0.143, 0.095,
                                        0.122, 0.081, 0.084),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_)),
                         `NVG` = list(c(NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(1.167, 1.167, 1.105,
                                        1.139, 1.089, 1.091),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_
                                      ),
                                      c(NA_real_, NA_real_, NA_real_,
                                        NA_real_, NA_real_),
                                      c(NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_),
                                      c(NA_real_, NA_real_,
                                        NA_real_, NA_real_, NA_real_,
                                        NA_real_)),
                         `WCRES` = list(c(NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(0.219, 0.204, 0.172, 0.172,
                                          0.138, 0.172),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_),
                                        c(NA_real_,
                                          NA_real_, NA_real_, NA_real_,
                                          NA_real_),
                                        c(NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        ),
                                        c(NA_real_, NA_real_, NA_real_,
                                          NA_real_, NA_real_, NA_real_
                                        )))

  # Check for all missing variables
  for(nm in names(actual)){
    if("missing" %in% objects() && nm %in% missing){
      if(exists("char_cols") && nm %in% char_cols){
        na_val <- NA_character_
      }else if("date_cols" %in% objects() && !is.null(date_cols) && nm == date_cols){
        na_val <- as.POSIXct(NA, tz="UTC")
      }else{
        na_val <- NA_real_
      }
      if("list_cols" %in% objects() && !is.null(list_cols) && nm %in% list_cols){
        expect_equal(unlist(actual[[nm]]),
                               rep(na_val, length(unlist(actual[[nm]]))),
                               info = paste0(info_prefix, ": ", nm))
      }else{
        expect_equal(actual[[nm]],
                     rep(na_val, nrow(actual)),
                     info = paste0(info_prefix, ": ", nm))
      }
    }
  }

  # Check for specific expected values
  if("expected_vals" %in% objects() && !is.null(expected_vals)){
    for(nm in names(expected_vals)){
        expect_equal(actual[[nm]],
                     expected_vals[[nm]],
                     info = paste0(info_prefix, ": ", nm))
    }
  }

  # Check list column type and dimensions
  if("list_cols" %in% objects() && !is.null(list_cols)){
    for(nm in list_cols){
        expect_true(is.list(actual[[nm]]),
                    info = paste0(info_prefix, ": ", nm))
      if(exists("list_col_length") && !is.null(list_col_length)){
        if(length(list_col_length) == 1){
          list_col_length <- rep(list_col_length, length(actual[[nm]]))
        }
        for(i in 1:length(actual[[nm]])){
            expect_equal(length(unlist(actual[[nm]][i])),
                       list_col_length[i],
                       info = paste0(info_prefix, nm))
        }
      }
      if("list_col_groups" %in% objects() && !is.null(list_col_groups)){
        for(i in 1:nrow(actual)){
          for(g in 1:length(list_col_groups)){
            length_range <- range(
              sapply(actual[i,][list_col_groups[[g]]],
                     function(x)length(unlist(x)))
              )
            lbl = paste0(paste0(list_col_groups[[g]],collapse = ", ")," - row ", i)
              expect_equal(length_range[1],
                         length_range[2],
                         info = paste0(info_prefix, ": ", lbl))
          }
        }
      }
    }
  }
  for(arg in c("char_cols", "list_cols", "date_cols",
               "missing", "list_col_length",
               "list_col_groups", "expected_vals")){
    if(arg %in% objects()) rm(list = arg)
  }


# "read_sol() TEST.SOL UFGA950002"

    rds_file <- system.file("tinytest/test_data/SOL/TEST_SOL.rds",
                            package = "DSSAT")

    sol_file <- file.path(tempdir(), "TEST.SOL")

    write(readRDS(rds_file), sol_file)

    sol <- DSSAT::read_sol(sol_file, id_soil = "UFGA950002")

    file.remove(sol_file)

    info_prefix <- "read_sol() TEST.SOL UFGA950002"

    actual <- sol
    `char_cols` <- c("PEDON", "SOURCE", "TEXTURE", "DESCRIPTION",
                    "SITE", "COUNTRY", "SCS FAMILY", "SCOM", "SMHB",
                    "SMPX", "SMKE", "SLMH")
    `list_cols` <- c("SLB", "SLMH", "SLLL", "SDUL", "SSAT", "SRGF",
                    "SSKS", "SBDM", "SLOC", "SLCL", "SLSI", "SLCF",
                    "SLNI", "SLHW", "SLHB", "SCEC", "SADC")
    expected_vals <- list(`PEDON` = "UFGA950002",
                         `SOURCE` = "Alachua cou",
                         `TEXTURE` = "ty",
                         `DEPTH` = 20,
                         `DESCRIPTION` = "Millhopper Fine Sand (Compacted layer)",
                         `SITE` = "-99         -",
                         `COUNTRY` = "99",
                         `LAT` = NA_real_,
                         `LONG` = NA_real_,
                         `SCS FAMILY` = "Loamy, siliceous hyperthermic Grossarenic Paleudults",
                         `SCOM` = NA_character_,
                         `SALB` = 0.18,
                         `SLU1` = 5,
                         `SLDR` = 0.5,
                         `SLRO` = 66,
                         `SLNF` = 1,
                         `SLPF` = 0.92,
                         `SMHB` = "IB001",
                         `SMPX` = "IB001",
                         `SMKE` = "IB001",
                         `SLB` = list(c(15, 33, 89, 147, 157, 175,
                                        203)),
                         `SLMH` = list(c(NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_, NA_character_,
                                         NA_character_)),
                         `SLLL` = list(c(0.024, 0.016, 0.011, 0.021,
                                         0.022, 0.07, 0.105)),
                         `SDUL` = list(c(0.161, 0.104, 0.113, 0.08,
                                         0.101, 0.196, 0.248)),
                         `SSAT` = list(c(0.487, 0.407, 0.408, 0.385,
                                         0.404, 0.374, 0.351)),
                         `SRGF` = list(c(1, 0.3, 0, 0.05, 0, 0,
                                         0)),
                         `SSKS` = list(c(9.3, 24.4, 0.001, 32.9,
                                         19.8, 4, 0.9)),
                         `SBDM` = list(c(1.36, 1.57, 1.57, 1.63,
                                         1.58, 1.66, 1.72)),
                         `SLOC` = list(c(0.73, 0.34, 0.19, 0.07,
                                         0.07, 0.1, 0.12)),
                         `SLCL` = list(c(2.1, 2.3, 2.4, 1.9, 4.1,
                                         16.4, 28.6)),
                         `SLSI` = list(c(6.3, 4.1, 3.3, 3.3, 3.9,
                                         6.1, 8.6)),
                         `SLCF` = list(c(0L, 0L, 0L, 9L, 0L, 0L,
                                         0L)),
                         `SLNI` = list(c(0.08, 0.03, 0.02, 0.01,
                                         0.01, 0.01, 0.01)),
                         `SLHW` = list(c(5.2, 5.4, 5.2, 5.3, 5.2,
                                         4.9, 5)),
                         `SLHB` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)),
                         `SCEC` = list(c(5.4, 2.9, 1.9, 1.1, 2,
                                         6, 10.4)),
                         `SADC` = list(c(NA_real_, NA_real_, NA_real_,
                                         NA_real_, NA_real_, NA_real_,
                                         NA_real_)))

  # Check for all missing variables
  for(nm in names(actual)){
    if("missing" %in% objects() && nm %in% missing){
      if(exists("char_cols") && nm %in% char_cols){
        na_val <- NA_character_
      }else if("date_cols" %in% objects() && !is.null(date_cols) && nm == date_cols){
        na_val <- as.POSIXct(NA, tz="UTC")
      }else{
        na_val <- NA_real_
      }
      if("list_cols" %in% objects() && !is.null(list_cols) && nm %in% list_cols){
        expect_equal(unlist(actual[[nm]]),
                               rep(na_val, length(unlist(actual[[nm]]))),
                               info = paste0(info_prefix, ": ", nm))
      }else{
        expect_equal(actual[[nm]],
                     rep(na_val, nrow(actual)),
                     info = paste0(info_prefix, ": ", nm))
      }
    }
  }

  # Check for specific expected values
  if("expected_vals" %in% objects() && !is.null(expected_vals)){
    for(nm in names(expected_vals)){
        expect_equal(actual[[nm]],
                     expected_vals[[nm]],
                     info = paste0(info_prefix, ": ", nm))
    }
  }

  # Check list column type and dimensions
  if("list_cols" %in% objects() && !is.null(list_cols)){
    for(nm in list_cols){
        expect_true(is.list(actual[[nm]]),
                    info = paste0(info_prefix, ": ", nm))
      if(exists("list_col_length") && !is.null(list_col_length)){
        if(length(list_col_length) == 1){
          list_col_length <- rep(list_col_length, length(actual[[nm]]))
        }
        for(i in 1:length(actual[[nm]])){
            expect_equal(length(unlist(actual[[nm]][i])),
                       list_col_length[i],
                       info = paste0(info_prefix, nm))
        }
      }
      if("list_col_groups" %in% objects() && !is.null(list_col_groups)){
        for(i in 1:nrow(actual)){
          for(g in 1:length(list_col_groups)){
            length_range <- range(
              sapply(actual[i,][list_col_groups[[g]]],
                     function(x)length(unlist(x)))
              )
            lbl = paste0(paste0(list_col_groups[[g]],collapse = ", ")," - row ", i)
              expect_equal(length_range[1],
                         length_range[2],
                         info = paste0(info_prefix, ": ", lbl))
          }
        }
      }
    }
  }
  for(arg in c("char_cols", "list_cols", "date_cols",
               "missing", "list_col_length",
               "list_col_groups", "expected_vals")){
    if(arg %in% objects()) rm(list = arg)
  }



if(FALSE){
  library(tidyverse)

  sol_tests <- tibble(files = list.files("~/active/dssat-csm-data/Soil/",
                                         pattern = "SOL$",
                                         full.names = TRUE)) %>%
    mutate(contents = lapply(files, readLines)) %>%
    mutate(pstart = map(contents, ~str_which(.x, "^\\*(?![Ss][Oo][Ii][Ll])"))) %>%
    mutate(pend = map(contents, ~{c(tail(
      str_which(.x, "^\\*(?![Ss][Oo][Ii][Ll])"),
      -1),
      length(.x)
    ) - 1})) %>%
    mutate(pedon = pmap(list(contents, pstart),
                        \(lines, start) str_extract(lines[start], "(?<=^\\*)[^ ]+"))) %>%
    unnest(c(pedon, pstart, pend)) %>%
    mutate(headers = pmap(list(contents, pstart, pend),
                          \(lines, start, end) str_subset(lines[start:end], "^@"))) %>%
    unnest(c(headers)) %>%
    filter(!duplicated(headers)) %>%
    filter(!duplicated(pedon)) %>%
    mutate(lines = pmap(list(contents, pstart, pend),
                        \(lines, start, end) lines[start:end])) %>%
    pull(lines) %>%
    unlist()

  sol_tests %>%
    c("*SOILS: General DSSAT Soil Input File", .) %>%
    write("inst/tinytest/TEST.SOL")

  sol_orig <- read_sol("inst/tinytest/TEST.SOL")

  DSSAT:::create_call_test_cols_check("sol", sol_orig) %>%
    clipr::write_clip()

}
