% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_node_attrs_ws.R
\name{mutate_node_attrs_ws}
\alias{mutate_node_attrs_ws}
\title{Mutate node attribute values for a selection of nodes}
\usage{
mutate_node_attrs_ws(graph, ...)
}
\arguments{
\item{graph}{a graph object of class
\code{dgr_graph}.}

\item{...}{expressions used for the mutation
of node attributes. LHS of each expression is
either an existing or new node attribute name.
The RHS can consist of any valid R code that
uses node attributes as variables. Expressions
are evaluated in the order provided, so, node
attributes created or modified are ready to
use in subsequent expressions.}
}
\value{
a graph object of class
\code{dgr_graph}.
}
\description{
Within a graph's internal node
data frame (ndf), mutate node attribute
values only for nodes in a selection by
using one or more expressions.
}
\examples{
# Create a graph with 3 nodes
# and then select node `1`
graph <-
  create_graph() \%>\%
  add_path(n = 3) \%>\%
  set_node_attrs(
    node_attr = width,
    values = c(1.4, 0.3, 1.1)) \%>\%
  select_nodes(nodes = 1)

# Get the graph's internal ndf
# to show which node attributes
# are available
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1   1.4
#> 2  2 <NA>     2   0.3
#> 3  3 <NA>     3   1.1

# Mutate the `width` node
# attribute for the nodes
# only in the active selection
# of nodes (node `1`); here,
# we divide each value in the
# selection by 2
graph <-
  graph \%>\%
  mutate_node_attrs_ws(
    width = width / 2)

# Get the graph's internal
# ndf to show that the node
# attribute `width` was
# mutated only for node `1`
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1  0.7
#> 2  2 <NA>     2  0.3
#> 3  3 <NA>     3  1.1

# Create a new node attribute,
# called `length`, that is the
# log of values in `width` plus
# 2 (and, also, round all values
# to 2 decimal places)
graph <-
  graph \%>\%
  clear_selection() \%>\%
  select_nodes(nodes = 2:3) \%>\%
  mutate_node_attrs_ws(
    length = (log(width) + 2) \%>\%
               round(2))

# Get the graph's internal ndf
# to show that the node attribute
# values had been mutated only
# for nodes `2` and `3` (since
# node `1` is excluded, an NA
# value is applied)
get_node_df(graph)
#>   id type label width length
#> 1  1 <NA>     1   0.7     NA
#> 2  2 <NA>     2   0.3    0.8
#> 3  3 <NA>     3   1.1    2.1

# Create a new node attribute
# called `area`, which is the
# product of the `width` and
# `length` attributes
graph <-
  graph \%>\%
  mutate_node_attrs_ws(
    area = width * length)

# Get the graph's internal ndf
# to show that the node attribute
# values had been multiplied
# together (with new attr `area`)
# for nodes `2` and `3`
get_node_df(graph)
#>   id type label width length area
#> 1  1 <NA>     1   0.7     NA   NA
#> 2  2 <NA>     2   0.3    0.8 0.24
#> 3  3 <NA>     3   1.1    2.1 2.31

# We can invert the selection
# and mutate node `1` several
# times to get an `area` value
# for that node
graph <-
  graph \%>\%
  invert_selection() \%>\%
  mutate_node_attrs_ws(
    length = (log(width) + 5) \%>\%
               round(2),
    area = width * length)

# Get the graph's internal ndf
# to show that the 2 mutations
# occurred for node `1`, yielding
# non-NA values for its node
# attributes without changing
# those of the other nodes
get_node_df(graph)
#>   id type label width length  area
#> 1  1 <NA>     1   0.7   4.64 3.248
#> 2  2 <NA>     2   0.3   0.80 0.240
#> 3  3 <NA>     3   1.1   2.10 2.310
}
