% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynEGA.R
\name{dynEGA}
\alias{dynEGA}
\title{Dynamic Exploratory Graph Analysis}
\usage{
dynEGA(
  data,
  n.embed,
  tau = 1,
  delta = 1,
  level = c("individual", "group", "population"),
  id = NULL,
  group = NULL,
  use.derivatives = 1,
  model = c("glasso", "TMFG"),
  model.args = list(),
  algorithm = c("walktrap", "leiden", "louvain"),
  algorithm.args = list(),
  corr = c("cor_auto", "pearson", "spearman"),
  uni.method = c("expand", "LE", "louvain"),
  ncores,
  progress = TRUE,
  ...
)
}
\arguments{
\item{data}{A dataframe with the variables to be used in the analysis. The dataframe should be in a long format (i.e. observations for the same individual (for example, individual 1) are placed in order, from time 1 to time t, followed by the observations from individual 2, also ordered from time 1 to time t.)}

\item{n.embed}{Integer.
Number of embedded dimensions (the number of observations to be used in the \code{\link[EGAnet]{Embed}} function). For example,
an \code{"n.embed = 5"} will use five consecutive observations to estimate a single derivative.}

\item{tau}{Integer.
Number of observations to offset successive embeddings in the \code{\link[EGAnet]{Embed}} function. A tau of one uses adjacent observations.
Default is \code{"tau = 1"}.}

\item{delta}{Integer.
The time between successive observations in the time series.
Default is \code{"delta = 1"}.}

\item{level}{Character.
A string indicating the level of analysis. If the interest is
in modeling the intraindividual structure only (one dimensionality structure per individual), then \code{level} should be set to \code{"individual"}.
If the interest is in the structure of a group of individuals, then \code{level} should be set to \code{"group"}.
Finally, if the interest is in the population structure, then \code{level} should be set to \code{"population"}.

Current options are:

\itemize{

\item{\strong{\code{individual}}}
{Estimates the dynamic factors per individual. This should be the prefered method is one is interested in
in the factor structure of individuals. An additional parameter (\code{"id"}) needs to be provided identifying each individual.}

\item{\strong{\code{group}}}
{Estimates the dynamic factors for each group.
An additional parameter (\code{"group"}) needs to be provided identifying the group membership.}

\item{\strong{\code{population}}}
{Estimates the dynamic factors of the population}

}}

\item{id}{Numeric.
Number of the column identifying each individual.}

\item{group}{Numeric or character.
Number of the column identifying group membership. Must be specified only if \code{level = "group"}.}

\item{use.derivatives}{Integer.
The order of the derivative to be used in the EGA procedure. Default to 1.}

\item{model}{Character.
A string indicating the method to use.
Current options are:

\itemize{

\item{\strong{\code{glasso}}}
{Estimates the Gaussian graphical model using graphical LASSO with
extended Bayesian information criterion to select optimal regularization parameter.
This is the default method}

\item{\strong{\code{TMFG}}}
{Estimates a Triangulated Maximally Filtered Graph}

}}

\item{model.args}{List.
A list of additional arguments for \code{\link[EGAnet]{EBICglasso.qgraph}}
or \code{\link[EGAnet]{TMFG}}}

\item{algorithm}{A string indicating the algorithm to use or a function from \code{\link{igraph}}
Defaults to \code{"walktrap"}.
Current options are:

\itemize{

\item{\strong{\code{walktrap}}}
{Computes the Walktrap algorithm using \code{\link[igraph]{cluster_walktrap}}}

\item{\strong{\code{leiden}}}
{Computes the Leiden algorithm using \code{\link[igraph]{cluster_leiden}}.
Defaults to \code{objective_function = "modularity"}}

\item{\strong{\code{louvain}}}
{Computes the Louvain algorithm using \code{\link[igraph]{cluster_louvain}}}

}}

\item{algorithm.args}{List.
A list of additional arguments for \code{\link[igraph]{cluster_walktrap}}, \code{\link[igraph]{cluster_louvain}},
or some other community detection algorithm function (see examples)}

\item{corr}{Type of correlation matrix to compute. The default uses \code{"pearson"}.
Current options are:

\itemize{

\item{\strong{\code{cor_auto}}}
{Computes the correlation matrix using the \code{\link[qgraph]{cor_auto}} function from
\code{\link[qgraph]{qgraph}}}.

\item{\strong{\code{pearson}}}
{Computes Pearson's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.

\item{\strong{\code{spearman}}}
{Computes Spearman's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.
}}

\item{uni.method}{Character.
What unidimensionality method should be used? 
Defaults to \code{"LE"}.
Current options are:

\itemize{

\item{\strong{\code{expand}}}
{Expands the correlation matrix with four variables correlated .50.
If number of dimension returns 2 or less in check, then the data 
are unidimensional; otherwise, regular EGA with no matrix
expansion is used. This is the method used in the Golino et al. (2020)
\emph{Psychological Methods} simulation.}

\item{\strong{\code{LE}}}
{Applies the Leading Eigenvalue algorithm (\code{\link[igraph]{cluster_leading_eigen}})
on the empirical correlation matrix. If the number of dimensions is 1,
then the Leading Eigenvalue solution is used; otherwise, regular EGA
is used. This is the final method used in the Christensen, Garrido,
and Golino (2021) simulation.}

\item{\strong{\code{louvain}}}
{Applies the Louvain algorithm (\code{\link[igraph]{cluster_louvain}})
on the empirical correlation matrix using a resolution parameter = 0.95.
If the number of dimensions is 1, then the Louvain solution is used; otherwise,
regular EGA is used. This method was validated in the Christensen (2022) simulation.}

}}

\item{ncores}{Numeric.
Number of cores to use in computing results.
Defaults to \code{parallel::detectCores() / 2} or half of your
computer's processing power.
Set to \code{1} to not use parallel computing.
Recommended to use maximum number of cores minus one

If you're unsure how many cores your computer has,
then use the following code: \code{parallel::detectCores()}}

\item{progress}{Boolean.
Should progress be displayed?
Defaults to \code{TRUE}.
For Windows, \code{FALSE} is about 2x faster}

\item{...}{Additional arguments.
Used for deprecated arguments from previous versions of \code{\link{EGA}}}
}
\description{
Estimates dynamic factors in multivariate time series (i.e. longitudinal data, panel data, intensive longitudinal data) at multiple
time scales, in different levels of analysis: individuals (intraindividual structure), groups or population (structure of the population).
Exploratory graph analysis is applied in the derivatives estimated using generalized local linear approximation (\code{\link[EGAnet]{glla}}). Instead of estimating factors by modeling how variables are covarying, as in traditional
EGA, dynEGA is a dynamic model that estimates the factor structure by modeling how variables are changing together.
GLLA is a filtering method for estimating derivatives from data that uses time delay embedding and a variant of Savitzky-Golay filtering to accomplish the task.
}
\examples{
# Obtain data
sim.dynEGA <- sim.dynEGA # bypasses CRAN checks

\dontrun{
# Population structure
dyn.random <- dynEGA(
  data = sim.dynEGA, n.embed = 5, tau = 1,
  delta = 1, id = 25, group = 26, use.derivatives = 1,
  level = "population", ncores = 2, corr = "pearson"
)

# Plot population structure
plot(dyn.random)

# Group structure
dyn.group <- dynEGA(
  data = sim.dynEGA, n.embed = 5, tau = 1,
  delta = 1, id = 25, group = 26, use.derivatives = 1,
  level = "group", ncores = 2, corr = "pearson"
)

# Plot group structure
plot(dyn.group, ncol = 2, nrow = 1)

# Intraindividual structure
dyn.individual <- dynEGA(
  data = sim.dynEGA, n.embed = 5, tau = 1,
  delta = 1, id = 25, group = 26, use.derivatives = 1,
  level = "individual", ncores = 2, corr = "pearson"
)

# Plot individual structure (participant 1)
plot(dyn.individual, id = 1)}

}
\references{
Boker, S. M., Deboeck, P. R., Edler, C., & Keel, P. K. (2010)
Generalized local linear approximation of derivatives from time series. In S.-M. Chow, E. Ferrer, & F. Hsieh (Eds.),
\emph{The Notre Dame series on quantitative methodology. Statistical methods for modeling human dynamics: An interdisciplinary dialogue},
(p. 161-178). \emph{Routledge/Taylor & Francis Group}.

Deboeck, P. R., Montpetit, M. A., Bergeman, C. S., & Boker, S. M. (2009)
Using derivative estimates to describe intraindividual variability at multiple time scales.
\emph{Psychological Methods}, \emph{14(4)}, 367-386.

Golino, H., Christensen, A. P., Moulder, R. G., Kim, S., & Boker, S. M. (2021).
Modeling latent topics in social media using Dynamic Exploratory Graph Analysis: The case of the right-wing and left-wing trolls in the 2016 US elections.
\emph{Psychometrika}.

Savitzky, A., & Golay, M. J. (1964).
Smoothing and differentiation of data by simplified least squares procedures.
\emph{Analytical Chemistry}, \emph{36(8)}, 1627-1639.
}
\author{
Hudson Golino <hfg9s at virginia.edu>
}
