% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discretePB_fun.R
\name{discrete.PB}
\alias{discrete.PB}
\alias{DPB}
\alias{NDPB}
\title{Discrete Poisson-Binomial procedure}
\usage{
discrete.PB(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  adaptive = TRUE,
  critical.values = FALSE,
  exact = TRUE
)

DPB(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  critical.values = FALSE,
  exact = TRUE
)

NDPB(
  raw.pvalues,
  pCDFlist,
  alpha = 0.05,
  zeta = 0.5,
  critical.values = FALSE,
  exact = TRUE
)
}
\arguments{
\item{raw.pvalues}{vector of the raw observed p-values, as provided by the end user and before matching with their nearest neighbor in the CDFs supports.}

\item{pCDFlist}{a list of the supports of the CDFs of the p-values. Each support is represented by a vector that must be in increasing order.}

\item{alpha}{the target FDP, a number strictly between 0 and 1. For \code{*.fast} kernels, it is only necessary, if \code{stepUp = TRUE}.}

\item{zeta}{the target probability of not exceeding the desired FDP, a number strictly between 0 and 1. If \code{zeta=NULL} (the default), then \code{zeta} is chosen equal to \code{alpha}.}

\item{adaptive}{a boolean specifying whether to conduct an adaptive procedure or not.}

\item{critical.values}{a boolean. If \code{TRUE}, critical constants are computed and returned (this is computationally intensive).}

\item{exact}{a boolean specifying whether to compute the Poisson-Binomial distribution exactly or by a normal approximation.}
}
\value{
A \code{FDX} S3 class object whose elements are:
\item{Rejected}{Rejected raw p-values.}
\item{Indices}{Indices of rejected hypotheses.}
\item{Num.rejected}{Number of rejections.}
\item{Adjusted}{Adjusted p-values (only for step-down direction).}

\item{Critical.values}{Critical values (if requested).}
\item{Method}{A character string describing the used algorithm, e.g. 'Discrete Lehmann-Romano procedure (step-up)'.}
\item{FDP.threshold}{FDP threshold \code{alpha}.}
\item{Exceedance.probability}{Probability \code{zeta} of FDP exceeding \code{alpha}; thus, FDP is being controlled at level \code{alpha} with confidence \code{1 - zeta}.}


\item{Data$raw.pvalues}{The values of \code{raw.pvalues}.}
\item{Data$pCDFlist}{The values of \code{pCDFlist}.}

\item{Data$data.name}{The respective variable names of \code{raw.pvalues} and \code{pCDFlist}.}
}
\description{
Apply the [DPB] procedure, with or without computing the critical values, to
a set of p-values and their discrete support. A non-adaptive version is
available as well. Additionally, the user can choose between exact
computation of the Poisson-Binomial distribution or a refined normal
approximation.
}
\details{
\code{DPB} and \code{NDPB} are wrapper functions for \code{discrete.PB}.
The first one simply passes all its parameters to \code{discrete.PB} with
\code{adaptive = TRUE} and \code{NDPB} does the same with
\code{adaptive = FALSE}.
}
\section{References}{

S. Döhler and E. Roquain (2019). Controlling False Discovery Exceedance for
Heterogeneous Tests.
\href{https://arxiv.org/abs/1912.04607v1}{arXiv:1912.04607v1}.
}

\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

# Construction of the p-values and their supports (fisher.pvalues.support
# is from 'DiscreteFDR' package!)
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

DPB.fast <- DPB(raw.pvalues, pCDFlist)
summary(DPB.fast)

DPB.crit <- DPB(raw.pvalues, pCDFlist, critical.values = TRUE)
summary(DPB.crit)

NDPB.fast <- NDPB(raw.pvalues, pCDFlist)
summary(NDPB.fast)

NDPB.crit <- NDPB(raw.pvalues, pCDFlist, critical.values = TRUE)
summary(NDPB.crit)

}
\seealso{
\code{\link{kernel}}, \code{\link{FDX-package}}, \code{\link{continuous.LR}},
\code{\link{continuous.GR}}, \code{\link{discrete.LR}}, 
\code{\link{discrete.GR}}, \code{\link{weighted.LR}},
\code{\link{weighted.GR}}, \code{\link{weighted.PB}}
}
