% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lccrec.R
\name{lccrec}
\alias{lccrec}
\title{Level conditional coherent forecast reconciliation for genuine hierarchical/grouped time series}
\usage{
lccrec(basef, m, C, nl, weights, bnaive = NULL, const = "exogenous",
       CCC = TRUE, nn = FALSE, nn_type = "osqp", settings = list(), ...)
}
\arguments{
\item{basef}{matrix/vector of base forecasts to be reconciled:
(\mjseqn{h \times n}) matrix in the cross-sectional framework;
(\mjseqn{h(k^\ast + m) \times 1}) vector in the temporal framework;
(\mjseqn{n \times h(k^\ast+m)}) matrix in the cross-temporal framework.
\mjseqn{n} is the total number of variables, \mjseqn{m} is the highest time
frequency, \mjseqn{k^\ast} is the sum of (a subset of) (\mjseqn{p-1}) factors
of \mjseqn{m}, excluding \mjseqn{m}, and \mjseqn{h} is the forecast horizon.}

\item{m}{Highest available sampling frequency per seasonal cycle (max. order
of temporal aggregation, \mjseqn{m}), or a subset of the \mjseqn{p} factors
of \mjseqn{m}.}

\item{C}{(\mjseqn{n_a \times n_b}) cross-sectional (contemporaneous) matrix
mapping the bottom level series into the higher level ones (or a list
of matrices forming \mjseqn{\mathbf{C} = [\mathbf{C}_1' \; \mathbf{C}_2' \; ... \; \mathbf{C}_L']'},
\mjseqn{1, ..., L} being the number of the cross-sectional upper levels.}

\item{nl}{(\mjseqn{L \times 1}) vector containing the number of time series
in each level of the hierarchy (\code{nl[1] = 1}).}

\item{weights}{covariance matrix or a vector (weights used in the reconciliation:
either (\mjseqn{n_b \times 1}) for exogenous or (\mjseqn{n \times 1}) for
endogenous constraints).}

\item{bnaive}{matrix/vector of naive bts base forecasts
(e.g., seasonal averages, as in Hollyman et al., 2021):
(\mjseqn{h \times n_b}) matrix in the cross-sectional framework;
(\mjseqn{h m \times 1}) vector in the temporal framework;
(\mjseqn{n_b \times h m}) matrix in the cross-temporal framework.
Ignore it, if only basef are to be used as base forecasts.}

\item{const}{\strong{exo}genous (\emph{default}) or \strong{endo}genous
constraints}

\item{CCC}{Option to return Combined Conditional Coherent reconciled
forecasts (\emph{default} is TRUE).}

\item{nn}{Logical value: \code{TRUE} if non-negative reconciled forecasts
are wished.}

\item{nn_type}{Non-negative method: "osqp" (\emph{default}) or "sntz"
(\emph{set-negative-to-zero}, only if \code{CCC = TRUE}) with exogenous
constraints (\code{const = "exo"}); "osqp" (\emph{default}), "KAnn"
(only \code{type == "M"}) or "sntz" with endogenous constraints
(\code{const = "endo"}).}

\item{settings}{Settings for \pkg{osqp} (object \code{\link[osqp]{osqpSettings}}).
The default options are: \code{verbose = FALSE}, \code{eps_abs = 1e-5},
\code{eps_rel = 1e-5}, \code{polish_refine_iter = 100} and \code{polish = TRUE}.
For details, see the \href{https://osqp.org/}{\pkg{osqp} documentation}
(Stellato et al., 2019).}

\item{...}{Additional functional arguments passed to \link[FoReco]{htsrec} of
FoReco.}
}
\value{
The function returns the level reconciled forecasts
in case of an elementary hierarchy with one level.
Otherwise it returns a list with
\item{\code{recf}}{Level Conditional Coherent (\code{CCC = FALSE}) forecasts
matrix/vector calculated as simple averages of upper level conditional
reconciled forecasts, with either endogenous or exogenous constraints.
If \code{CCC = TRUE} then it is the Combined Conditional Coherent matrix/vector,
as weighted averages of LCC and bottom-up reconciled forecasts.}
\item{\code{levrecf}}{list of level conditional reconciled forecasts (+ BU).}
\item{\code{info} (\code{type="osqp"})}{matrix with some useful indicators (columns)
for each forecast horizon \mjseqn{h} (rows): run time (\code{run_time}), number of iteration,
norm of primal residual (\code{pri_res}), status of osqp's solution (\code{status}) and
polish's status (\code{status_polish}).}
}
\description{
\loadmathjax
Forecast reconciliation procedure built on and extending the original
proposal by Hollyman et al. (2021). Level conditional coherent
reconciled forecasts may be computed in cross-sectional, temporal, and
cross-temporal frameworks. The reconciled forecasts are conditional to (i.e.,
constrained by) the base forecasts of a specific upper level of the hierarchy
(exogenous constraints). The linear constraints linking the variables may be
dealt with endogenously as well (Di Fonzo and Girolimetto, 2021).
\emph{Combined Conditional Coherent} (CCC)
forecasts may be calculated as simple averages of LCC and bottom-up
reconciled forecasts, with either endogenous or exogenous constraints.
}
\details{
\strong{Cross-sectional hierarchies}

To be as simple as possible, we fix the forecast horizon equal to 1.
Let the base forecasts be a vector \mjseqn{\widehat{\mathbf{y}} =
\left[\widehat{\mathbf{a}}' \; \widehat{\mathbf{b}}'\right]'}, where
vector \mjseqn{\widehat{\mathbf{a}}} consists of the sub-vectors forming each
of the upper \mjseqn{L} levels of the hierarchy/grouping:
\mjsdeqn{\widehat{\mathbf{a}} = \left[\begin{array}{c}
    \widehat{a}_1 \cr \widehat{\mathbf{a}}_2 \cr \vdots \cr \widehat{\mathbf{a}}_L
    \end{array}\right],}
where \mjseqn{\widehat{\mathbf{a}}_l}, \mjseqn{l=1,\ldots, L}, has dimension
\mjseqn{(n_l \times 1)} and \mjseqn{\sum_{l=1}^{L} n_l = n_a}. Denote
\mjseqn{\mathbf{C}_l} the \mjseqn{(n_l \times n_b)} matrix mapping the
bts into the level-\mjseqn{l} uts, then the aggregation matrix
\mjseqn{\mathbf{C}} may be written as
\mjsdeqn{\mathbf{C} = \left[\begin{array}{c}
  \mathbf{C}_1 \cr\mathbf{C}_2 \cr \vdots \cr \mathbf{C}_L
  \end{array}\right],}
where the generic matrix \mjseqn{\mathbf{C}_L} is (\mjseqn{n_L \times n_b}),
\mjseqn{l=1, \ldots, L}. Given a generic level \mjseqn{l}, the reconciled
forecasts coherent with the base forecasts of that level are the solution to
a quadratic minimization problem with linear
exogenous constraints (\code{const = "exo"}):
\mjsdeqn{\begin{array}{c}\widetilde{\mathbf{b}}_{l} = \arg\min_{\mathbf{b}}
\left(\mathbf{b} - \widehat{\mathbf{b}}\right)'\mathbf{W}_b^{-1}
\left(\mathbf{b} - \widehat{\mathbf{b}}\right) \quad \mbox{ s.t. }
\mathbf{C}_l\mathbf{b} = \widehat{\mathbf{a}}_l, \quad l=1, \ldots, L ,\cr
\Downarrow \cr
\widetilde{\mathbf{b}}_{l} = \widehat{\mathbf{b}} +
\mathbf{W}_b\mathbf{C}_l'\left(\mathbf{C}_l\mathbf{W}_b\mathbf{C}_l'
\right)^{-1}\left(\widehat{\mathbf{a}}_l -\mathbf{C}_l\widehat{\mathbf{b}}
\right), \quad l=1,\ldots,L,\end{array}}
where \mjseqn{\mathbf{W}_b} is a (\mjseqn{n_b \times n_b}) p.d. matrix
(in Hollyman et al., 2021, \mjseqn{\mathbf{W}_b} is diagonal).
If endogenous constraints (\code{const = "endo"}) are considered,
denote \mjseqn{\widehat{\mathbf{y}}_l =
\left[\widehat{\mathbf{a}}_l' \; \widehat{\mathbf{b}}'\right]'} and
\mjseqn{\mathbf{U}_l' = \left[\mathbf{I}_{n_l}'  \; \mathbf{C}_l'\right]'},
then
\mjsdeqn{\begin{array}{c}\widetilde{\mathbf{y}}_{l} = \arg\min_{\mathbf{y}_l}
\left(\mathbf{y}_l - \widehat{\mathbf{y}}_l\right)'\mathbf{W}_l^{-1}
\left(\mathbf{y}_l - \widehat{\mathbf{y}}_l\right) \quad \mbox{ s.t. }
\mathbf{U}_l'\mathbf{y}_l = \mathbf{0}, \quad l=1, \ldots, L ,\cr
\Downarrow \cr
\widetilde{\mathbf{y}}_{l} = \left(\mathbf{I}_{n_b+n_l} -
\mathbf{W}_l\mathbf{U}_l\left(\mathbf{U}_l'\mathbf{W}_l
\mathbf{U}_l\right)^{-1}\mathbf{U}_l'\right)\widehat{\mathbf{y}}_{l},
\quad l=1,...,L,
\end{array}}
where \mjseqn{\mathbf{W}_l} is a (\mjseqn{n_l + n_b \times n_l + n_b})
p.d. matrix.
Combined Conditional Coherent (CCC) forecasts are obtained by taking
the simple average of the \mjseqn{L} level conditional, and of the bottom-up
reconciled forecasts, respectively (Di Fonzo and Girolimetto, 2021):
\mjsdeqn{\widetilde{\mathbf{y}}_{CCC}=\frac{1}{L+1}\sum_{l=1}^{L+1} \mathbf{S}\widetilde{\mathbf{b}}_l,}
with \mjsdeqn{\widetilde{\mathbf{b}}_{L+1} = \widehat{\mathbf{b}}.}

Non-negative reconciled forecasts may be obtained by setting \code{nn_type}
alternatively as:
\itemize{
  \item to apply non-negative constraints to each level:
  \itemize{
    \item \code{nn_type = "KAnn"} (only \code{const = "endo"})
    \item \code{nn_type = "osqp"}
  }
  \item to apply non-negative constraints only to the CCC forecasts:
  \itemize{
    \item \code{nn_type = "sntz"} ("set-negative-to-zero")
  }
}

\strong{Temporal hierarchies}

The extension to the case of \strong{temporal hierarchies} is quite simple.
Using the same notation as in \code{\link[FoReco]{thfrec}()}, the
following `equivalences' hold between the symbols used
for the level conditional cross-sectional reconciliation and the ones
used in temporal reconciliation:
\mjsdeqn{L \equiv p-1, \quad (n_a, n_b, n) \equiv (k^*, m, k^*+m),}
and
\mjsdeqn{\mathbf{C} \equiv \mathbf{K} , \;
\mathbf{C}_1 \equiv \mathbf{K}_1 = \mathbf{1}_m', \;
\mathbf{C}_2 \equiv \mathbf{K}_2 = \mathbf{I}_{\frac{m}{k_{p-1}}},\; \ldots, \;
\mathbf{C}_L \equiv \mathbf{K}_{p-1} = \mathbf{I}_{\frac{m}{k_{2}}} \otimes
\mathbf{1}_{k_{2}}',\; \mathbf{S} \equiv \mathbf{R}.}

The description of the \strong{cross-temporal extension} is currently under progress.
}
\examples{
data(FoReco_data)
### LCC and CCC CROSS-SECTIONAL FORECAST RECONCILIATION
# Cross sectional aggregation matrix
C <- rbind(FoReco_data$C, c(0,0,0,0,1))
# monthly base forecasts
id <- which(simplify2array(strsplit(colnames(FoReco_data$base), split = "_"))[1, ] == "k1")
mbase <- t(FoReco_data$base[, id])[,c("T", "A","B","C","AA","AB","BA","BB","C")]
# residuals
id <- which(simplify2array(strsplit(colnames(FoReco_data$res), split = "_"))[1, ] == "k1")
mres <- t(FoReco_data$res[, id])[,c("T", "A","B","C","AA","AB","BA","BB","C")]
# covariance matrix of all the base forecasts, computed using the in-sample residuals
Wres <- cov(mres)
# covariance matrix of the bts base forecasts, computed using the in-sample residuals
Wb <- Wres[c("AA","AB", "BA", "BB", "C"),
           c("AA","AB", "BA", "BB", "C")]
# bts seasonal averages
obs_1 <- FoReco_data$obs$k1
bts_sm <- apply(obs_1, 2, function(x) tapply(x[1:168],rep(1:12, 14), mean))
bts_sm <- bts_sm[,c("AA", "AB", "BA", "BB", "C")]

## EXOGENOUS CONSTRAINTS AND DIAGONAL COVARIANCE MATRIX (Hollyman et al., 2021)
# Forecast reconciliation for an elementary hierarchy:
# 1 top-level series + 5 bottom-level series (Level 2 base forecasts are not considered).
# The input is given by the base forecasts of the top and bottom levels series,
# along with a vector of positive weights for the bts base forecasts
exo_EHd <- lccrec(basef = mbase[, c("T","AA","AB", "BA", "BB", "C")],
                 weights = diag(Wb), bnaive = bts_sm)

# Level conditional reconciled forecasts
# recf/L1: Level 1 reconciled forecasts for the whole hierarchy
# L2: Middle-Out reconciled forecasts hinging on Level 2 conditional reconciled forecasts
# L3: Bottom-Up reconciled forecasts
exo_LCd <- lccrec(basef = mbase, C = C, nl = c(1,3), weights = diag(Wb),
                 CCC = FALSE, bnaive = bts_sm)

# Combined Conditional Coherent (CCC) reconciled forecasts
# recf: CCC reconciled forecasts matrix
# L1: Level 1 conditional reconciled forecasts for the whole hierarchy
# L2: Middle-Out reconciled forecasts hinging on Level 2 conditional reconciled forecasts
# L3: Bottom-Up reconciled forecasts
exo_CCCd <- lccrec(basef = mbase, C = C, nl = c(1,3), weights = diag(Wb))

## EXOGENOUS CONSTRAINTS AND FULL COVARIANCE MATRIX
# Simply substitute weights=diag(Wb) with weights=Wb
exo_EHf <- lccrec(basef = mbase[, c("T","AA","AB", "BA", "BB", "C")], weights = Wb, bnaive = bts_sm)
exo_LCf <- lccrec(basef = mbase, C = C, nl = c(1,3), weights = Wb, CCC = FALSE, bnaive = bts_sm)
exo_CCCf <- lccrec(basef = mbase, C = C, nl = c(1,3), weights = Wb, bnaive = bts_sm)

## ENDOGENOUS CONSTRAINTS AND DIAGONAL COVARIANCE MATRIX
# parameters of function htsrec(), like "type" and "nn_type" may be used

# Forecast reconciliation for an elementary hierarchy with endogenous constraints
W1 <- Wres[c("T","AA","AB", "BA", "BB", "C"),
           c("T","AA","AB", "BA", "BB", "C")]
endo_EHd <- lccrec(basef = mbase[, c("T","AA","AB", "BA", "BB", "C")],
                 weights = diag(W1), const = "endo", CCC = FALSE)

# Level conditional reconciled forecasts with endogenous constraints
endo_LCd <- lccrec(basef = mbase, C = C, nl = c(1,3), weights = diag(Wres),
                  const = "endo", CCC = FALSE)

# Combined Conditional Coherent (CCC) reconciled forecasts with endogenous constraints
endo_CCCd <- lccrec(basef = mbase, C = C, nl = c(1,3),
                    weights = diag(Wres), const = "endo")

## ENDOGENOUS CONSTRAINTS AND FULL COVARIANCE MATRIX
# Simply substitute weights=diag(Wres) with weights=Wres
endo_EHf <- lccrec(basef = mbase[, c("T","AA","AB", "BA", "BB", "C")],
                   weights = W1,
                   const = "endo")
endo_LCf <- lccrec(basef = mbase, C = C, nl = c(1,3),
                   weights = Wres, const = "endo", CCC = FALSE)
endo_CCCf <- lccrec(basef = mbase-40, C = C, nl = c(1,3),
                   weights = Wres, const = "endo")

### LCC and CCC TEMPORAL FORECAST RECONCILIATION
# top ts base forecasts ([lowest_freq' ...  highest_freq']')
topbase <- FoReco_data$base[1, ]
# top ts residuals ([lowest_freq' ...  highest_freq']')
topres <- FoReco_data$res[1, ]
Om_bt <- cov(matrix(topres[which(simplify2array(strsplit(names(topres),
            "_"))[1,]=="k1")], ncol = 12, byrow = TRUE))
t_exo_LCd <- lccrec(basef = topbase, m = 12, weights = diag(Om_bt), CCC = FALSE)
t_exo_CCCd <- lccrec(basef = topbase, m = 12, weights = diag(Om_bt))

### LCC and CCC CROSS-TEMPORAL FORECAST RECONCILIATION
idr <- which(simplify2array(strsplit(colnames(FoReco_data$res), "_"))[1,]=="k1")
bres <- FoReco_data$res[-c(1:3), idr]
bres <- lapply(1:5, function(x) matrix(bres[x,], nrow=14, byrow = TRUE))
bres <- do.call(cbind, bres)
ctbase <- FoReco_data$base[c("T", "A","B","C","AA","AB","BA","BB","C"),]
ct_exo_LCf <- lccrec(basef = ctbase, m = 12, C = C, nl = c(1,3),
                    weights = diag(cov(bres)), CCC = FALSE)
ct_exo_CCCf <- lccrec(basef = ctbase, m = 12, C = C, nl = c(1,3),
                     weights = diag(cov(bres)), CCC = TRUE)

}
\references{
Di Fonzo, T., and Girolimetto, D. (2021), Cross-temporal forecast reconciliation:
Optimal combination method and heuristic alternatives, \emph{International Journal
of Forecasting}, in press.

Di Fonzo, T., Girolimetto, D. (2021), Forecast combination based forecast reconciliation:
insights and extensions, \href{https://arxiv.org/abs/2106.05653}{arXiv:2006.08570}.

Hollyman, R., Petropoulos, F., Tipping, M.E. (2021), Understanding Forecast Reconciliation,
\emph{European Journal of Operational Research} (in press).
}
\seealso{
Other reconciliation procedures: 
\code{\link{cstrec}()},
\code{\link{ctbu}()},
\code{\link{htsrec}()},
\code{\link{iterec}()},
\code{\link{octrec}()},
\code{\link{tcsrec}()},
\code{\link{tdrec}()},
\code{\link{thfrec}()}
}
\concept{reconciliation procedures}
