.onLoad <- function(lib, pkg)
{
    ## Most of the holiday date series here have New Years Day for a few more
    ## years beyond the end of known data - having one day holiday in the
    ## data for a year helps catch lack of holiday data in other programs.
    ## These New Years Days holidays are calculated as the nearest weekday
    ## to Jan 1st and may not be NOT correct for the exchange, because
    ## different exchanges can use different rules for when to take
    ## the New Years Day holiday (if at all.)
    ##
    ## It is OK to have weekend dates in the holiday data -- will have no
    ## effect.

    ## Sources of trading holiday data:
    ##   http://www.jquantlib.org/index.php/Exchange_Holidays
    ##   http://www.asiaetrading.com/resource/holiday-calendar/

    ###########################################################################
    ##
    ## AUSEX
    ##
    ## 2004-5 cross checked against the exchange calendar:
    ## http://www.asx.com.au/about/Calendar_AA2.shtm
    ##
    registerHolidays('AUSEX', dateParse(
    c(
        "1990/01/01", "1990/01/29", "1990/04/13", "1990/04/16", "1990/04/25",
        "1990/06/11", "1990/12/25", "1990/12/26", "1990/12/31", "1991/01/01",
        "1991/01/28", "1991/03/29", "1991/04/01", "1991/04/25", "1991/06/10",
        "1991/12/25", "1991/12/26", "1992/01/01", "1992/01/27", "1992/04/17",
        "1992/04/20", "1992/06/08", "1992/12/25", "1992/12/26", "1992/12/28",
        "1993/01/01", "1993/04/09", "1993/04/12", "1993/06/14", "1993/12/27",
        "1993/12/28", "1994/01/03", "1994/01/26", "1994/04/01", "1994/04/04",
        "1994/04/25", "1994/06/13", "1994/10/03", "1994/12/26", "1994/12/27",
        "1995/01/02", "1995/01/26", "1995/04/14", "1995/04/17", "1995/04/25",
        "1995/06/12", "1995/10/02", "1995/12/25", "1995/12/26", "1996/01/01",
        "1996/01/02", "1996/01/26", "1996/04/05", "1996/04/08", "1996/04/25",
        "1996/06/10", "1996/10/07", "1996/12/25", "1996/12/26", "1997/01/01",
        "1997/01/27", "1997/03/28", "1997/03/31", "1997/04/25", "1997/06/09",
        "1997/08/04", "1997/10/06", "1997/12/25", "1997/12/26",

        "1998/01/01", "1998/01/26", "1998/04/10", "1998/04/13", "1998/06/08",
        "1998/12/25", "1998/12/28",

        "1999/01/01", "1999/01/26", "1999/04/02", "1999/04/05", "1999/06/14",
        "1999/12/27", "1999/12/28", "1999/12/31",

        "2000/01/03", "2000/01/26", "2000/04/21", "2000/04/24", "2000/04/25",
        "2000/06/12", "2000/12/25", "2000/12/26",

        "2001/01/01", "2001/01/26", "2001/04/13", "2001/04/16", "2001/04/25",
        "2001/06/11", "2001/12/25", "2001/12/26",

        "2002/01/01", "2002/01/28", "2002/03/29", "2002/04/01", "2002/04/25",
        "2002/06/10", "2002/12/25", "2002/12/26",

        "2003/01/01", "2003/01/27", "2003/04/18", "2003/04/21", "2003/04/25",
        "2003/06/09", "2003/12/25", "2003/12/26",

        "2004/01/01", "2004/01/26", "2004/04/09", "2004/04/12", "2004/06/14",
        "2004/12/25", "2004/12/27", "2004/12/28",

        "2005/01/03", "2005/01/26", "2005/03/25", "2005/03/28", "2005/04/25",
        "2005/06/13", "2005/12/26", "2005/12/27",

        ## from http://www.asx.com.au/supervision/operational/trading_calendar/2006.htm
        ## deleted quite a few days that were here -- those days were listed
        ## as "no settlement", but they are trading days.
        "2006/01/02", "2006/01/26", "2006/04/14", "2006/04/17",
        "2006/04/25", "2006/06/12", "2006/12/25", "2006/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/2007.htm
        ## deleted quite a few days that were here -- those days were listed
        ## as "no settlement", but they are trading days.
        "2007/01/01", "2007/01/26", "2007/04/06", "2007/04/09",
        "2007/04/25", "2007/06/11", "2007/12/25", "2007/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/asx/2008.htm
        "2008/01/01", "2008/01/28", "2008/03/21", "2008/03/24",
        "2008/04/25", "2008/06/09", "2008/12/25", "2008/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/asx/2009.htm
        "2009/01/01", "2009/01/26", "2009/04/10", "2009/04/13",
        "2009/06/08", "2009/12/25", "2009/12/28",

        ## from http://www.asx.com.au/trading_services/sycom-trading-calendar-2010.htm
        "2010/01/01", "2010/01/26", "2010/04/02", "2010/04/05", "2010/04/26",
        "2010/06/13", "2010/12/26", "2010/12/27",

        ## from http://www.asx.com.au/trading_services/asx-24-trading-calendar-2011.htm
        "2011/01/03", "2011/01/26", "2011/04/22", "2011/04/25", "2011/04/26",
        "2011/06/14", "2011/12/27", "2011/12/28",

        ## from http://www.asx.com.au/trading_services/asx-24-trading-calendar-2011.htm
        "2012/01/02", "2012/01/26", "2012/04/06", "2012/04/09", "2012/04/25",
        "2012/06/11", "2012/12/25", "2012/12/26",

        ## Check holidays on Bloomberg with "CDR AU 2009", though that shows up lots
        ## of extras that are holidays in Australia, but for which the market is open

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2013/01/01", "2014/01/01", "2015/01/01"
    )))

    ###########################################################################
    ##
    ## GBREX
    ##
    registerHolidays('GBREX', dateParse(
    c(
        "1994/01/03", "1994/04/01", "1994/04/04", "1994/05/02", "1994/05/30", "1994/08/29",
        "1994/12/26", "1994/12/27", "1995/01/02", "1995/04/14", "1995/04/17", "1995/05/01",
        "1995/05/29", "1995/08/28", "1995/12/25", "1995/12/26", "1996/01/01", "1996/04/05",
        "1996/04/08", "1996/05/06", "1996/05/27", "1996/08/26", "1996/12/25", "1996/12/26",
        "1997/01/01", "1997/03/28", "1997/03/31", "1997/05/05", "1997/05/26", "1997/08/25",
        "1997/12/25", "1997/12/26", "1998/01/01", "1998/04/10", "1998/04/13", "1998/05/04",
        "1998/05/25", "1998/08/31", "1998/12/25", "1998/12/28", "1998/12/31", "1999/01/01",
        "1999/04/02", "1999/04/05", "1999/05/03", "1999/05/31", "1999/08/30", "1999/12/27",
        "1999/12/28", "1999/12/31", "2000/01/03", "2000/04/21", "2000/04/24", "2000/05/01",
        "2000/05/29", "2000/08/28", "2000/12/25", "2000/12/26", "2001/01/01", "2001/04/13",
        "2001/04/16", "2001/05/07", "2001/05/28", "2001/08/27", "2001/12/25", "2001/12/26",
        "2002/01/01", "2002/03/29", "2002/04/01", "2002/05/06", "2002/06/03",
        "2002/06/04", "2002/08/26", "2002/12/25", "2002/12/26", "2003/01/01", "2003/04/18",
        "2003/04/21", "2003/05/05", "2003/05/26", "2003/08/25", "2003/12/25", "2003/12/26",
        "2004/01/01", "2004/04/09", "2004/04/12", "2004/05/03", "2004/05/31", "2004/08/30",
        "2004/12/27", "2004/12/28",

        ## 2005 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays.htm
        "2005/01/03", "2005/03/25", "2005/03/28", "2005/05/02", "2005/05/30", "2005/08/29",
        "2005/12/26", "2005/12/27",

        ## 2006 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2006/01/02", "2006/04/14", "2006/04/17", "2006/05/01", "2006/05/29", "2006/08/28",
        "2006/12/25", "2006/12/26",

        ## 2007 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2007/01/01", "2007/04/06", "2007/04/09", "2007/05/07", "2007/05/28", "2007/08/27",
        "2007/12/25", "2007/12/26",

        ## 2008 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2008/01/01", "2008/03/21", "2008/03/24", "2008/05/05", "2008/05/26", "2008/08/25",
        "2008/12/25", "2008/12/26",

        ## 2009 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2009/01/01", "2009/04/10", "2009/04/13", "2009/05/04", "2009/05/25", "2009/08/31",
        "2009/12/25", "2009/12/28",

        ## Check holidays on Bloomberg with "CDR GB 2009"

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2010/01/01", "2010/12/31", "2012/01/02", "2013/01/01", "2014/01/01"
    )))

    ###########################################################################
    ##
    ## HKGEX
    ##
    registerHolidays('HKGEX', dateParse(
    c(
        "1996/01/01", "1996/02/19", "1996/02/20", "1996/02/21", "1996/04/04",
        "1996/04/05", "1996/04/08", "1996/06/17", "1996/06/20", "1996/08/26",
        "1996/10/21", "1996/12/25", "1996/12/26",

        "1997/01/01", "1997/02/06", "1997/02/07", "1997/03/28", "1997/03/31",
        "1997/06/09", "1997/06/30", "1997/07/01", "1997/07/02", "1997/08/18",
        "1997/09/17", "1997/10/01", "1997/10/02", "1997/10/10", "1997/12/25",
        "1997/12/26",

        "1998/01/01", "1998/01/28", "1998/01/29", "1998/01/30", "1998/04/06",
        "1998/04/10", "1998/04/13", "1998/07/01", "1998/08/17", "1998/10/01",
        "1998/10/02", "1998/10/06", "1998/10/28", "1998/12/25",

        "1999/01/01", "1999/02/16", "1999/02/17", "1999/02/18", "1999/04/02",
        "1999/04/05", "1999/04/06", "1999/06/18", "1999/07/01", "1999/09/16",
        "1999/10/01", "1999/10/18", "1999/12/27", "1999/12/31",

        "2000/02/04", "2000/02/07", "2000/04/04", "2000/04/21", "2000/04/24",
        "2000/05/01", "2000/05/11", "2000/06/06", "2000/09/13", "2000/10/02",
        "2000/10/06", "2000/12/25", "2000/12/26",

        "2001/01/01", "2001/01/24", "2001/01/25", "2001/01/26", "2001/04/05",
        "2001/04/13", "2001/04/16", "2001/04/30", "2001/05/01", "2001/06/25",
        "2001/07/02", "2001/07/06", "2001/07/25", "2001/10/01", "2001/10/02",
        "2001/10/25", "2001/12/25", "2001/12/26",

        "2002/01/01", "2002/02/12", "2002/02/13", "2002/02/14", "2002/03/29",
        "2002/04/01", "2002/04/05", "2002/05/01", "2002/05/20", "2002/07/01",
        "2002/10/01", "2002/10/14", "2002/12/25", "2002/12/26",

        "2003/01/01", "2003/01/31", "2003/02/03", "2003/04/18", "2003/04/21",
        "2003/05/01", "2003/05/08", "2003/06/04", "2003/07/01", "2003/09/12",
        "2003/10/01", "2003/12/25", "2003/12/26",

        "2004/01/01", "2004/01/22", "2004/01/23", "2004/04/05", "2004/04/09",
        "2004/04/12", "2004/05/26", "2004/06/22", "2004/07/01", "2004/09/29",
        "2004/10/01", "2004/10/22", "2004/12/27",

        "2005/02/09", "2005/02/10", "2005/02/11", "2005/03/25", "2005/03/28",
        "2005/04/05", "2005/05/02", "2005/05/16", "2005/07/01", "2005/09/19",
        "2005/10/11", "2005/12/26", "2005/12/27",

        ## from http://www.hkex.com.hk/tradinfo/tradcal/caln06e.htm
        "2006/01/02",
        "2006/01/30", "2006/01/31", "2006/04/05", "2006/04/14", "2006/04/17",
        "2006/05/01", "2006/05/05", "2006/05/31", "2006/10/02", "2006/10/30",
        "2006/12/25", "2006/12/26",

        ## New Years for 2007.  Not published yet, so this is a guess.
        "2007/01/01"
    )))

    ###########################################################################
    ##
    ## JPNEX
    ##
    registerHolidays('JPNEX', dateParse(
    c(
        "1994/01/03", "1994/02/11", "1994/03/21", "1994/04/29", "1994/05/04",
        "1994/05/05", "1994/09/15", "1994/09/23", "1994/10/10", "1994/11/03", "1994/11/23",
        "1994/12/23", "1995/01/02", "1995/01/03", "1995/01/16", "1995/03/21", "1995/05/03",
        "1995/05/04", "1995/05/05", "1995/09/15", "1995/10/10", "1995/11/03", "1995/11/23",
        "1996/01/01", "1996/01/02", "1996/01/03", "1996/01/15", "1996/02/12", "1996/03/20",
        "1996/04/29", "1996/05/03", "1996/05/06", "1996/09/16", "1996/09/23", "1996/10/10",
        "1996/11/04", "1996/12/23", "1996/12/31", "1997/01/01", "1997/01/02", "1997/01/03",
        "1997/01/15", "1997/02/11", "1997/03/20", "1997/04/29", "1997/05/05", "1997/07/21",
        "1997/09/15", "1997/09/23", "1997/10/10", "1997/11/03", "1997/11/24", "1997/12/23",
        "1997/12/31", "1998/01/01", "1998/01/02", "1998/01/15", "1998/02/11", "1998/04/29",
        "1998/05/04", "1998/05/05", "1998/07/20", "1998/09/15",
        "1998/09/23", "1998/11/03", "1998/11/23", "1998/12/23", "1998/12/31", "1999/01/01",
        "1999/01/15", "1999/02/11", "1999/03/22", "1999/04/29", "1999/05/03", "1999/05/04",
        "1999/05/05", "1999/07/20", "1999/09/15", "1999/09/23", "1999/10/11", "1999/11/03",
        "1999/11/23", "1999/12/23", "1999/12/31", "2000/01/03", "2000/01/10", "2000/02/11",
        "2000/03/20", "2000/05/03", "2000/05/04", "2000/05/05", "2000/07/20", "2000/09/15",
        "2000/10/09", "2000/11/03", "2000/11/23", "2001/01/01", "2001/01/02", "2001/01/03",
        "2001/01/08", "2001/02/12", "2001/03/20", "2001/04/30", "2001/05/03", "2001/05/04",
        "2001/07/20", "2001/09/24", "2001/10/08", "2001/11/23", "2001/12/24", "2001/12/31",
        "2002/01/01", "2002/01/02", "2002/01/03", "2002/01/14", "2002/02/11", "2002/03/21",
        "2002/04/29", "2002/05/03", "2002/05/06", "2002/09/16", "2002/09/23", "2002/10/14",
        "2002/11/04", "2002/12/23", "2002/12/31", "2003/01/01", "2003/01/02",
        "2003/01/03", "2003/01/13", "2003/02/11", "2003/03/21", "2003/04/29",
        "2003/05/05", "2003/07/21", "2003/09/15", "2003/09/23", "2003/10/13", "2003/11/03",
        "2003/11/24", "2003/12/23", "2003/12/31",
        "2004/01/01", "2004/01/02", "2004/01/12", "2004/02/11", "2004/04/29", "2004/05/03",
        "2004/05/04", "2004/05/05", "2004/07/19", "2004/09/20", "2004/09/23", "2004/10/11",
        "2004/11/03", "2004/11/23", "2004/12/23", "2004/12/31",
        "2005/01/03", "2005/01/10", "2005/02/11", "2005/03/21", "2005/04/29", "2005/05/03",
        "2005/05/04", "2005/05/05", "2005/07/18", "2005/09/19", "2005/09/23", "2005/10/10",
        "2005/11/03", "2005/11/23", "2005/12/23",

        ## Recent data is from http://www.tse.or.jp/english/guide/calendar.html
        ## Supposedly 2003/05/03 was a holiday, but I think we had close prices for that day

        ## 2006/2007 data is from http://www.tse.or.jp/english/guide/calendar.html
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd(!!!)

        "2006/01/02", "2006/01/03", "2006/01/09", "2006/03/21", "2006/05/03",
        "2006/05/04", "2006/05/05", "2006/07/17", "2006/09/18", "2006/09/23",
        "2006/10/09", "2006/11/03", "2006/11/23",

        "2007/01/01", "2007/01/02", "2007/01/03", "2007/01/08", "2007/02/12", "2007/03/21", "2007/04/30",
        "2007/05/03", "2007/05/04", "2007/05/05", "2007/07/16", "2007/09/17",
        "2007/09/24", "2007/10/08", "2007/11/03", "2007/11/23", "2007/12/24", "2007/12/31",

        ## 2008 data is from http://www.tse.or.jp/english/about/calendar.html
        ## (visit http://www.tse.or.jp/english, then "About TSE"->Calendar)
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd & Dec 31 (!!!)

        "2008/01/01", "2008/01/02", "2008/01/03", "2008/01/14", "2008/02/11", "2008/03/20", "2008/04/29",
        "2008/05/05", "2008/05/06", "2008/07/21", "2008/09/15",
        "2008/09/23", "2008/10/13", "2008/11/03", "2008/11/24", "2008/12/23", "2008/12/31",

        ## 2009 data is from http://www.tse.or.jp/english/about/calendar.html
        ## (visit http://www.tse.or.jp/english, then "About TSE"->Calendar)
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd & Dec 31 (!!!)
        ## Not sure about the status of 05/03 and 05/04 -- these were also listed
        ## as holidays in 2008, but we didn't have them in our database.

        "2009/01/01", "2009/01/02", "2009/01/03", "2009/01/12", "2009/02/11", "2009/03/20", "2009/04/29",
        "2009/05/03", "2009/05/04", "2009/05/05", "2009/05/06", "2009/07/20", "2009/09/21",
        "2009/09/22", "2009/09/23", "2009/10/12", "2009/11/03", "2009/11/23", "2009/12/23", "2009/12/31",

        ## 2010 data is from http://www.timeanddate.com/calendar/?year=2010&country=26
        ## agrees with http://www.asiaetrading.com/resource/holiday-calendar/
        "2010/01/01", "2010/01/02", "2010/01/03", "2010/01/11", "2010/02/11", "2010/03/22", "2010/04/29",
        "2010/05/03", "2010/05/04", "2010/05/05", "2010/07/19", "2010/09/20",
        "2010/09/23", "2010/10/11", "2010/11/03", "2010/11/23", "2010/12/23", "2010/12/31",

        ## 2011 data is from http://www.tse.or.jp/english/about/calendar.html
        "2011/01/01", "2011/01/02", "2011/01/03", "2011/01/10", "2011/02/11", "2011/03/21", "2011/04/29",
        "2011/05/03", "2011/05/04", "2011/05/05", "2011/07/18", "2011/09/21",
        "2011/09/23", "2011/10/10", "2011/11/03", "2011/11/23", "2011/12/23", "2011/12/31",

        ## 2012 data is from http://www.tse.or.jp/english/about/calendar.html
        "2012/01/01", "2012/01/02", "2012/01/03", "2012/01/09", "2012/02/11", "2012/03/20", "2012/04/29",
        "2012/05/03", "2012/05/04", "2012/05/05", "2012/07/16", "2012/09/17",
        "2012/09/22", "2012/10/08", "2012/11/03", "2012/11/23", "2012/12/23", "2012/12/24", "2012/12/31",

        ## Check holidays on Bloomberg with "CDR JN 2009"

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2013/01/01", "2014/01/01", "2015/01/01"
    )))

    ###########################################################################
    ##
    ## JPNEX.halfday (not up to date)
    ##
    registerHolidays('JPNEX.halfday', dateParse(
    c(
         "2003/01/06", "2003/12/30",
         "2004/01/05", "2004/12/30",
         "2005/01/04", "2005/12/30",
         "2006/01/04", "2006/12/29",
         "2006/01/04", "2006/12/28"
    )))


    ###########################################################################
    ##
    ## KOREX (not up to date)
    ##
    ## Data cross checked against the Bloomberg "CDR SK 2004"
    ##
    registerHolidays('KOREX', dateParse(
    c(
        "1996/01/01", "1996/01/02", "1996/02/19", "1996/02/20", "1996/03/01",
        "1996/04/05", "1996/04/11", "1996/05/01", "1996/05/05", "1996/05/24",
        "1996/06/04", "1996/07/17", "1996/08/15", "1996/09/26", "1996/09/27",
        "1996/09/28", "1996/10/03", "1996/12/25", "1996/12/28", "1996/12/30",
        "1996/12/31",

        "1997/01/02", "1997/02/07", "1997/02/08", "1997/03/01", "1997/04/05",
        "1997/05/01", "1997/05/05", "1997/05/24", "1997/07/17", "1997/08/15",
        "1997/09/15", "1997/09/16", "1997/09/17", "1997/10/03", "1997/12/18",
        "1997/12/25", "1997/12/29", "1997/12/30", "1997/12/31",

        "1998/01/01", "1998/01/02", "1998/01/27", "1998/01/28", "1998/01/29",
        "1998/05/01", "1998/05/05", "1998/06/04", "1998/06/06", "1998/07/17",
        "1998/08/15", "1998/10/03", "1998/10/05", "1998/10/06", "1998/12/12",
        "1998/12/19", "1998/12/25", "1998/12/26", "1998/12/29", "1998/12/30",
        "1998/12/31",

        "1999/01/01", "1999/01/02", "1999/02/15", "1999/02/16", "1999/02/17",
        "1999/03/01", "1999/04/05", "1999/05/01", "1999/05/05", "1999/05/22",
        "1999/06/06", "1999/07/17", "1999/08/15", "1999/09/23", "1999/09/24",
        "1999/09/25", "1999/10/03", "1999/12/25", "1999/12/29", "1999/12/30",
        "1999/12/31",

        "2000/01/01", "2000/01/02", "2000/01/03", "2000/02/04", "2000/02/05",
        "2000/02/06", "2000/03/01", "2000/04/05", "2000/04/13", "2000/05/01",
        "2000/05/05", "2000/05/11", "2000/06/06", "2000/07/17", "2000/08/15",
        "2000/09/11", "2000/09/12", "2000/09/13", "2000/10/03", "2000/12/25",
        "2000/12/27", "2000/12/28", "2000/12/29", "2000/12/30",

        "2001/01/01", "2001/01/23", "2001/01/24", "2001/01/25", "2001/03/01",
        "2001/04/05", "2001/05/01", "2001/05/05", "2001/06/06", "2001/07/17",
        "2001/08/15", "2001/10/01", "2001/10/02", "2001/10/03", "2001/12/25",
        "2001/12/31",

        "2002/01/01", "2002/02/11", "2002/02/12", "2002/02/13", "2002/03/01",
        "2002/04/05", "2002/05/01", "2002/05/05", "2002/05/19", "2002/06/06",
        "2002/06/13", "2002/07/01", "2002/07/17", "2002/08/15", "2002/09/20",
        "2002/09/21", "2002/10/03", "2002/12/19", "2002/12/25", "2002/12/31",

        "2003/01/01", "2003/01/31", "2003/02/01", "2003/03/01", "2003/04/05",
        "2003/05/01", "2003/05/05", "2003/05/08", "2003/06/06", "2003/07/17",
        "2003/08/15", "2003/09/10", "2003/09/11", "2003/09/12", "2003/10/03",
        "2003/12/25","2003/12/31",

        "2004/01/01", "2004/01/21", "2004/01/22", "2004/01/23", "2004/03/01",
        "2004/04/05", "2004/04/15", "2004/05/01", "2004/05/05", "2004/05/26",
        "2004/06/06", "2004/07/17", "2004/08/15", "2004/09/27", "2004/09/28",
        "2004/09/29", "2004/10/03", "2004/12/25", "2004/12/31",

        "2005/02/08", "2005/02/09", "2005/02/10", "2005/03/01", "2005/04/05",
        "2005/05/05", "2005/06/06", "2005/08/15", "2005/09/19", "2005/10/03",
        "2005/12/30",

        ## from http://sm.krx.co.kr/webeng/new_index.jsp (click on "Trading
        ## Hours & Calendar" under "One click service" -- had to view in
        ## internet explorer -- didn't see any 2006 holidays there)
        ## From the Bloomberg "CDR SK 2006"

        "2006/01/30", "2006/03/01", "2006/05/01", "2006/05/05", "2006/06/06",
        "2006/07/17", "2006/08/15", "2006/10/03", "2006/10/05", "2006/10/06",
        "2006/12/25",

        ## New Years for 2007.  Not published yet, so this is a guess.
        "2007/01/01"
    )))


    ###########################################################################
    ##
    ## NewYears
    ##
    registerHolidays('NewYears', as.Date(ISOdate(1980:2020, 1, 1)))

    ###########################################################################
    ##
    ## NEWYEAR
    ##
    ## The weekday day closest to new year's day, which seems to be the only day of
    ## the year on which there are no prices for international stocks.
    ## Note that this doesn't seem to be the holiday rule for GBR, at least: they
    ## have 01/03 as New Year's day for 2005 (2005/01/01 is a Saturday)
    ## Neither is holiday rule for NYSE: they don't have any day off
    ## for 2005/01/01.
    ##
    ##
    registerHolidays('NEWYEAR', dateParse(
    c(
        '1980/01/01','1981/01/01','1982/01/01','1982/12/31','1984/01/02','1985/01/01',
        '1986/01/01','1987/01/01','1988/01/01','1989/01/02','1990/01/01','1991/01/01',
        '1992/01/01','1993/01/01','1993/12/31','1995/01/02','1996/01/01','1997/01/01',
        '1998/01/01','1999/01/01','1999/12/31','2001/01/01','2002/01/01','2003/01/01',
        '2004/01/01','2004/12/31','2006/01/02','2007/01/01','2008/01/01','2009/01/01',
        '2010/01/01','2010/12/31','2012/01/02','2013/01/01','2014/01/01','2015/01/01',
        '2016/01/01','2017/01/02','2018/01/01','2019/01/01','2020/01/01'
    )))

    ###########################################################################
    ##
    ## NYSE
    ## This list includes holidays but not unsheduled closings, e.g., for funerals of
    ## presidents and for events that force closings.
    ##
    ##
    registerHolidays('NYSE', dateParse(
    c(
        '1980/01/01','1980/02/18','1980/04/04','1980/05/26','1980/07/04','1980/09/01',
        '1980/11/04','1980/11/27','1980/12/25','1981/01/01','1981/02/16','1981/04/17',
        '1981/05/25','1981/07/03','1981/09/07','1981/11/26','1981/12/25','1982/01/01',
        '1982/02/15','1982/04/09','1982/05/31','1982/07/05','1982/09/06','1982/11/25',
        '1982/12/24','1983/02/21','1983/04/01','1983/05/30','1983/07/04','1983/09/05',
        '1983/11/24','1983/12/26','1984/01/02','1984/02/20','1984/04/20','1984/05/28',
        '1984/07/04','1984/09/03','1984/11/22','1984/12/25','1985/01/01','1985/02/18',
        '1985/04/05','1985/05/27','1985/07/04','1985/09/02','1985/11/28','1985/12/25',
        '1986/01/01','1986/02/17','1986/03/28','1986/05/26','1986/07/04','1986/09/01',
        '1986/11/27','1986/12/25','1987/01/01','1987/02/16','1987/04/17','1987/05/25',
        '1987/07/03','1987/09/07','1987/11/26','1987/12/25','1988/01/01','1988/02/15',
        '1988/04/01','1988/05/30','1988/07/04','1988/09/05','1988/11/24','1988/12/26',
        '1989/01/02','1989/02/20','1989/03/24','1989/05/29','1989/07/04','1989/09/04',
        '1989/11/23','1989/12/25','1990/01/01','1990/02/19','1990/04/13','1990/05/28',
        '1990/07/04','1990/09/03','1990/11/22','1990/12/25','1991/01/01','1991/02/18',
        '1991/03/29','1991/05/27','1991/07/04','1991/09/02','1991/11/28','1991/12/25',
        '1992/01/01','1992/02/17','1992/04/17','1992/05/25','1992/07/03','1992/09/07',
        '1992/11/26','1992/12/25','1993/01/01','1993/02/15','1993/04/09','1993/05/31',
        '1993/07/05','1993/09/06','1993/11/25','1993/12/24','1994/02/21','1994/04/01',
        '1994/05/30','1994/07/04','1994/09/05','1994/11/24','1994/12/26','1995/01/02',
        '1995/02/20','1995/04/14','1995/05/29','1995/07/04','1995/09/04','1995/11/23',
        '1995/12/25','1996/01/01','1996/02/19','1996/04/05','1996/05/27','1996/07/04',
        '1996/09/02','1996/11/28','1996/12/25','1997/01/01','1997/02/17','1997/03/28',
        '1997/05/26','1997/07/04','1997/09/01','1997/11/27','1997/12/25','1998/01/01',
        '1998/01/19','1998/02/16','1998/04/10','1998/05/25','1998/07/03','1998/09/07',
        '1998/11/26','1998/12/25','1999/01/01','1999/01/18','1999/02/15','1999/04/02',
        '1999/05/31','1999/07/05','1999/09/06','1999/11/25','1999/12/24','2000/01/17',
        '2000/02/21','2000/04/21','2000/05/29','2000/07/04','2000/09/04','2000/11/23',
        '2000/12/25','2001/01/01','2001/01/15','2001/02/19','2001/04/13','2001/05/28',
        '2001/07/04','2001/09/03','2001/11/22','2001/12/25','2002/01/01','2002/01/21',
        '2002/02/18','2002/03/29','2002/05/27','2002/07/04','2002/09/02','2002/11/28',
        '2002/12/25','2003/01/01','2003/01/20','2003/02/17','2003/04/18','2003/05/26',
        '2003/07/04','2003/09/01','2003/11/27','2003/12/25','2004/01/01','2004/01/19',
        '2004/02/16','2004/04/09','2004/05/31','2004/07/05','2004/09/06','2004/11/25',
        '2004/12/24','2005/01/17','2005/02/21','2005/03/25','2005/05/30','2005/07/04',
        '2005/09/05','2005/11/24','2005/12/26','2006/01/02','2006/01/16','2006/02/20',
        '2006/04/14','2006/05/29','2006/07/04','2006/09/04','2006/11/23','2006/12/25',
        '2007/01/01','2007/01/15','2007/02/19','2007/04/06','2007/05/28','2007/07/04',
        '2007/09/03','2007/11/22','2007/12/25','2008/01/01','2008/01/21','2008/02/18',
        '2008/03/21','2008/05/26','2008/07/04','2008/09/01','2008/11/27','2008/12/25',
        '2009/01/01','2009/01/19','2009/02/16','2009/04/10','2009/05/25','2009/07/03',
        '2009/09/07','2009/11/26','2009/12/25','2010/01/01','2010/01/18','2010/02/15',
        '2010/04/02','2010/05/31','2010/07/05','2010/09/06','2010/11/25','2010/12/24',
        '2011/01/17','2011/02/21','2011/04/22','2011/05/30','2011/07/04','2011/09/05',
        '2011/11/24','2011/12/26','2012/01/02','2012/01/16','2012/02/20','2012/04/06',
        '2012/05/28','2012/07/04','2012/09/03','2012/11/22','2012/12/25','2013/01/01',
        '2013/01/21','2013/02/18','2013/03/29','2013/05/27','2013/07/04','2013/09/02',
        '2013/11/28','2013/12/25','2014/01/01','2014/01/20','2014/02/17','2014/04/18',
        '2014/05/26','2014/07/04','2014/09/01','2014/11/27','2014/12/25','2015/01/01',
        '2015/01/19','2015/02/16','2015/04/03','2015/05/25','2015/07/03','2015/09/07',
        '2015/11/26','2015/12/25','2016/01/01','2016/01/18','2016/02/15','2016/03/25',
        '2016/05/30','2016/07/04','2016/09/05','2016/11/24','2016/12/26','2017/01/02',
        '2017/01/16','2017/02/20','2017/04/14','2017/05/29','2017/07/04','2017/09/04',
        '2017/11/23','2017/12/25','2018/01/01','2018/01/15','2018/02/19','2018/03/30',
        '2018/05/28','2018/07/04','2018/09/03','2018/11/22','2018/12/25','2019/01/01',
        '2019/01/21','2019/02/18','2019/04/19','2019/05/27','2019/07/04','2019/09/02',
        '2019/11/28','2019/12/25','2020/01/01','2020/01/20','2020/02/17','2020/04/10',
        '2020/05/25','2020/07/03','2020/09/07','2020/11/26','2020/12/25'
    )))

    ###########################################################################
    ##
    ## NYSE.Closings
    ## These need to be updated when there is a closing.
    ## NYSE traditionally closes for the funeral of a President of the US
    ##
    registerHolidays('NYSE.Closings', dateParse(
    c(
         "1973/01/25",  # funeral of former President Lyndon B. Johnson
         "1977/07/14",  # Closed due to blackout in New York
         "1994/04/27",  # funeral of Richard M. Nixon
         "2001/09/11",  # Attack on World Trade Center
         "2001/09/12",  # Attack on World Trade Center
         "2001/09/13",  # Attack on World Trade Center
         "2001/09/14",  # Attack on World Trade Center
         "2004/06/11",  # funeral of former President Ronald Reagan
         "2007/01/02"   # National Day of mouring for Gerald Ford
    )))

    ###########################################################################
    ##
    ## NYSEC
    ##
    ## Combine holidays and historical closings (e.g., funeral of Richard Nixon,
    ## World Trade Center attack).
    ##
    registerHolidays("NYSEC", c(holidays(1980:2020, "NYSE"),
                                holidays(1980:2020, "NYSE.Closings",
                                         silent = TRUE)))

    ###########################################################################
    ##
    ## NYSEC.halfday (not up to date)
    ##
    registerHolidays('NYSE.halfday', dateParse(
    c(
         "2003/07/03", "2003/11/28", "2003/12/24", "2003/12/26"
    )))
}
