#' Rcpp apply Generator
#' 
#' Use this function as a generator for your own \code{apply} functions;
#' that is, functions you would like to apply over rows or columns of a
#' matrix.
#' 
#' @param fun A character string defining the C++ function. It must be
#' in terms of a variable \code{x}, and it must return a \code{double}.
#' \code{x} is a reference to the current row/column being iterated over.
#' @param name An internal name for the function.
#' @param file A location to output the file. Defaults to a temporary
#' file as generated by \code{\link{tempfile}}().
#' @export
rcpp_apply_generator <- function( fun, name=NULL, file=NULL ) {
  
  require("Rcpp")
  
  if( getRversion() < "2.15.0" ||
        packageVersion("Rcpp") < "0.10.0"
      ) {
    message("Error: This function requires Rcpp > 0.10.1 and R > 2.15.1")
    return( invisible(NULL) )
  }
  
  ## generate a C++ source file based on 'fun'
  if( is.null(file) ) {
    
    cpp_source <- paste( sep="", tempfile(), ".cpp" )
    
  } else {
    
    if( length( grep( "\\.cpp", file ) ) == 0 ) {
      file <- paste( sep="", file, ".cpp" )
    }
    
    cpp_source <- file
    
  }
  
  cat("C++ source code will be written to", cpp_source, ".\n")
  
  ## an environment for the cppSource-exported functions to live in
  cpp_env <- new.env()
  
  ## internal name for the function
  if( is.null(name) ) {  
    name <- paste( collapse="", 
                   sample( c(letters, LETTERS), size=20, replace=TRUE)
    )
  } 
  
  ## open a connection to the file, and ensure we close it after
  conn <- file( cpp_source, 'w' )
  on.exit( close(conn) )
  
  ## 'cat' our source code out to a file, to later be sourced
  cat( file=conn, sep="", paste( sep="", collapse="",
                                 
                                 "#include <Rcpp.h>
                                 using namespace Rcpp;
                                 
                                 inline double do_", name, "( const NumericMatrix::Column& x ) {\n", fun, "\n }
                                 
                                 // [[Rcpp::export]]
                                 NumericVector ", name, "( NumericMatrix X ) {
                                 
                                 int nCols = X.ncol();
                                 NumericVector out = no_init(nCols);
                                 
                                 for( int j=0; j < nCols; j++ ) {
                                 NumericMatrix::Column tmp = X(_, j);
                                 out[j] = do_", name, "( tmp );
                                 }
                                 
                                 return out;
                                 
                                 }") )
  
  ## source the file into the 'cpp_env' environment
  cat("Compiling...\n")
  sourceCpp( cpp_source, env=cpp_env )
  cat("Done!")
  
  ## return a function that exposes the compiled code to the user
  return( function(X, dim) {
    
    force( cpp_source )
    
    if( dim == 1 ) {
      X <- t(X)
    }
    
    call <- call(name, X)
    return( eval( call, envir=cpp_env) )
    
  })
  
}