#' Simulated Annealing for LHD
#'
#' \code{SA} returns an LHD matrix generated by simulated annealing algorithm (SA)
#'
#' @param n A positive integer.
#' @param k A positive integer.
#' @param N A positive integer.
#' @param T0 A positive number.
#' @param rate A positive percentage.
#' @param Tmin A positive number.
#' @param Imax A positive integer.
#' @param OC An optimality criterion.
#' @param p A positive integer.
#' @param q The default is set to be 1, and it could be either 1 or 2.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} LHD.
#' @details \itemize{
#' \item \code{n} stands for the number of rows (or run size).
#' \item \code{k} stands for the number of columns (or the number of factors).
#' \item \code{N} stands for the number of iterations. The default is set to be 10.
#' \item \code{T0} stands for the user-defined initial temperature. The default is set to be 10.
#' \item \code{rate} stands for temperature decrease rate, and it should be in (0,1). For example, rate=0.25 means the temperature decreases by 25\% each time. The default is set to be 10\%.
#' \item \code{Tmin} stands for the minimium temperature allowed. When current temperature becomes smaller or equal to \code{Tmin}, the stopping criterion for current loop is met. The default is set to be 1.
#' \item \code{Imax} stands for the maximum perturbations the algorithm will try without improvements before temperature is reduced. For the computation complexity consideration, \code{Imax} is recommended to be smaller or equal to 5, which is the default setting.
#' \item \code{OC} stands for the optimality criterion, the default setting is "phi_p", and it could be one of the following: "phi_p", "AvgAbsCor", "MaxAbsCor", "MaxProCriterion".
#' \item \code{p} is the parameter in the phi_p formula, and \code{p} is prefered to be large. The default is set to be 15.
#' \item If \code{q} is 1 (the default setting), \code{dij} is the rectangular distance. If \code{q} is 2, \code{dij} is the Euclidean distance.
#' }
#'
#' @references Morris, M.D., and Mitchell, T.J. (1995) Exploratory designs for computer experiments. \emph{Journal of Statistical Planning and Inference}, \strong{43}, 381-402.
#'
#' @examples
#' #generate a 5 by 3 maximin distance LHD with the default setting
#' try=SA(n=5,k=3)
#' try
#' phi_p(try)   #calculate the phi_p of "try".
#'
#' #Another example
#' #generate a 8 by 4 nearly orthogonal LHD
#' try2=SA(n=8,k=4,OC="AvgAbsCor")
#' try2
#' AvgAbsCor(try2)  #calculate the average absolute correlation.
#' @export

SA=function(n,k,N=10,T0=10,rate=0.1,Tmin=1,Imax=5,OC="phi_p",p=15,q=1){
  #n and k are the rs and fa.
  #N: maximum number of iterations.
  #T0: initial temperature
  #rate: temperature decrease rate. 0<rate<1
  #Tmin: minumum temperature for each itertaion,TPmin > 0
  #Imax:# of perturbations the algorithm will try without improvements before Temperature is reduced
  #OC: optimality criterion, the default is "phi_p", along with default p and q

  C=1  #step 1: counter index

  X=rLHD(n=n,k=k)   #step 2

  Xbest=X;TP=T0;Flag=1

  if(OC=="phi_p"){
    while (C<=N) {

      while(Flag==1 & TP>Tmin){
        Flag=0;I=1

        while (I<=Imax) {
          rcol=sample(1:k,1)   #step 3:Randomly choose a column
          Xnew=exchange(X=X,j=rcol)  #step 4:Exchange two randomly elements within the column "rcol"

          a=phi_p(X=Xnew,p=p,q=q)       #step 5 begins here
          b=phi_p(X=X,p=p,q=q)
          if (a<b){X=Xnew;Flag=1}
          if (a>=b){
            prob=exp((b-a)/TP)
            draw=sample(c(0,1),1,prob=c(1-prob,prob))    #draw==1 means replace
            if(draw==1){X=Xnew;Flag=1}
          }                         #step 5 ends here

          c=phi_p(X=Xbest,p=p,q=q)
          if (a<c){Xbest=Xnew;I=1}
          if (a>=c){I=I+1}

        }

        TP=TP*(1-rate)
      }

      C=C+1;TP=T0;Flag=1
    }


  }

  if(OC=="AvgAbsCor"){
    while (C<=N) {

      while(Flag==1 & TP>Tmin){
        Flag=0;I=1

        while (I<=Imax) {
          rcol=sample(1:k,1)   #step 3:Randomly choose a column
          Xnew=exchange(X=X,j=rcol)  #step 4:Exchange two randomly elements within the column "rcol"

          a=AvgAbsCor(X=Xnew)       #step 5 begins here
          b=AvgAbsCor(X=X)
          if (a<b){X=Xnew;Flag=1}
          if (a>=b){
            prob=exp((b-a)/TP)
            draw=sample(c(0,1),1,prob=c(1-prob,prob))    #draw==1 means replace
            if(draw==1){X=Xnew;Flag=1}
          }                         #step 5 ends here

          c=AvgAbsCor(X=Xbest)
          if (a<c){Xbest=Xnew;I=1}
          if (a>=c){I=I+1}

        }

        TP=TP*(1-rate)
      }

      C=C+1;TP=T0;Flag=1
    }


  }

  if(OC=="MaxAbsCor"){
    while (C<=N) {

      while(Flag==1 & TP>Tmin){
        Flag=0;I=1

        while (I<=Imax) {
          rcol=sample(1:k,1)   #step 3:Randomly choose a column
          Xnew=exchange(X=X,j=rcol)  #step 4:Exchange two randomly elements within the column "rcol"

          a=MaxAbsCor(X=Xnew)       #step 5 begins here
          b=MaxAbsCor(X=X)
          if (a<b){X=Xnew;Flag=1}
          if (a>=b){
            prob=exp((b-a)/TP)
            draw=sample(c(0,1),1,prob=c(1-prob,prob))    #draw==1 means replace
            if(draw==1){X=Xnew;Flag=1}
          }                         #step 5 ends here

          c=MaxAbsCor(X=Xbest)
          if (a<c){Xbest=Xnew;I=1}
          if (a>=c){I=I+1}

        }

        TP=TP*(1-rate)
      }

      C=C+1;TP=T0;Flag=1
    }


  }

  if(OC=="MaxProCriterion"){
    while (C<=N) {

      while(Flag==1 & TP>Tmin){
        Flag=0;I=1

        while (I<=Imax) {
          rcol=sample(1:k,1)   #step 3:Randomly choose a column
          Xnew=exchange(X=X,j=rcol)  #step 4:Exchange two randomly elements within the column "rcol"

          a=MaxProCriterion(X=Xnew)       #step 5 begins here
          b=MaxProCriterion(X=X)
          if (a<b){X=Xnew;Flag=1}
          if (a>=b){
            prob=exp((b-a)/TP)
            draw=sample(c(0,1),1,prob=c(1-prob,prob))    #draw==1 means replace
            if(draw==1){X=Xnew;Flag=1}
          }                         #step 5 ends here

          c=MaxProCriterion(X=Xbest)
          if (a<c){Xbest=Xnew;I=1}
          if (a>=c){I=I+1}

        }

        TP=TP*(1-rate)
      }

      C=C+1;TP=T0;Flag=1
    }


  }

  Xbest
}
