#' Fast Maximin Distance LHD
#'
#' \code{FastMmLHD} returns a maximin distance LHD matrix generated by the construction method of Wang, L., Xiao, Q., and Xu, H. (2018)
#'
#' @param n A positive integer, which stands for the number of rows (or run size).
#' @param k A positive integer, which stands for the number of columns (or factor size).
#' @param method A distance measure method. The default setting is "manhattan", and it could be one of the following: "euclidean", "maximum", "manhattan", "canberra", "binary" or "minkowski". Any unambiguous substring can be given.
#' @param t1 A tunning parameter, which determines how many repeats will be implemented to search for the optimal design. The default is set to be 10.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} maximin distance LHD.
#'
#' @references Wang, L., Xiao, Q., and Xu, H. (2018)  Optimal maximin $L_{1}$-distance Latin hypercube designs based on good lattice point designs. \emph{The Annals of Statistics}, \strong{46}(6B), 3741-3766.
#'
#' @examples
#' #n by n design when 2n+1 is prime
#' try=FastMmLHD(8,8)
#' try
#' phi_p(try)   #calculate the phi_p of "try".
#'
#' #n by n design when n+1 is prime
#' try2=FastMmLHD(12,12)
#' try2
#' phi_p(try2)   #calculate the phi_p of "try2".
#'
#' #n by n-1 design when n is prime
#' try3=FastMmLHD(7,6)
#' try3
#' phi_p(try3)   #calculate the phi_p of "try3".
#'
#' #General cases
#' try4=FastMmLHD(24,8)
#' try4
#' phi_p(try4)   #calculate the phi_p of "try4".
#'
#' @export

FastMmLHD=function(n,k,method="manhattan",t1=10){
  #n: number of runs
  #k: number of columns
  #method: the distance method, with default setting is "manhattan".
  #t1: tunning parameter

  if (!(class(n) == "numeric") || !(n%%1 == 0) || (n < 3))
  {
    stop("'n' must be a postive integer no less than 3")
  }
  if (!(class(k) == "numeric") || !(k%%1 == 0) || (k < 2))
  {
    stop("'k' must be a postive integer no less than 2")
  }
  if (n < k) {stop("n must be no less than k")}


  if (!is.na(pmatch(method, "euclidian")) || (method == "l2") || (method == "L2"))
  {method = "euclidean"}
  if (!is.na(pmatch(method, "manhattan"))|| (method == "l1") || (method == "L1"))
  {method = "manhattan"}

  METHODS = c("euclidean", "manhattan")
  count = pmatch(method, METHODS)

  if (is.na(count)==TRUE)
    stop("invalid distance method")
  if (count == -1)
    stop("ambiguous distance method")
  ###

  #the first special case of n \times n design when m = 2n+1 is prime. (2.3)
  if ((n == k) && numbers::isPrime(2*n + 1)==TRUE)
  {
    m = 2*n + 1
    resultdesign = MWT(GLP(n=m,k=m-1,h=1:(m-1)))/2
    return(resultdesign[1:n, 1:n]-1)
  }

  #the second special case of n \times (k<=n) design when n+1 is prime. (2.2)
  else if ((n >= k) && numbers::isPrime(n + 1)==TRUE)
  {
    resultdesign = NULL
    minvalue = 0
    for(i in 1:t1)
    {
      result = LOO(n=n,k=k,method=method)
      newresult = result[[1]]
      newminvalue = result[[2]]
      if (newminvalue > minvalue)
      {
        resultdesign = newresult
        minvalue = newminvalue
      }
    }

    return(resultdesign)
  }

  #the third specical case of n \times (k<n) design when n is prime. (2.1)
  else if ((n > k) && numbers::isPrime(n)==TRUE)
  {
    resultdesign = NULL
    minvalue = 0
    for(i in 1:t1)
    {
      result = LPWT(n=n,k=k,method=method)
      newresult = result[[1]]
      newminvalue = result[[2]]
      if (newminvalue > minvalue)
      {
        resultdesign = newresult
        minvalue = newminvalue
      }
    }
    return(resultdesign)
  }

  else {
    #General case, when above special cases are not true.
    #coprime vector to n
    copn=NULL
    for(i in 1:(n-1)){

      if (numbers::coprime(i,n)==TRUE)
      {
        copn=c(copn,i)
      }
    }

    #Euler function value of n
    eul=length(copn)

    if (k>eul){
      stop(paste0("the number of column for this run size must be no greater than ", eul))
    }

    else {

      basedesign = GLP(n=n,k=eul,h=sample(copn,eul))

      resultdesign = WT(basedesign,baseline=0)

      minvalue = min(stats::dist(resultdesign, method = method))

      for (c in 1:(n-1))
      {
        newdesign = (basedesign + c) %% n
        newresult = WT(newdesign,baseline=0)
        newminvalue = min(stats::dist(newresult, method = method))

        if (newminvalue > minvalue)
        {
          resultdesign = newresult
          minvalue = newminvalue
        }
      }


      #This is the searching part: search for the best subset if k!=eul.
      if(k==eul){
        return(resultdesign)
      }

      else {
        Temp=NULL
        Temp.value=Inf

        for (i in 1:t1) {
          rcol=sample(1:eul,k)   #random columns
          X=resultdesign[,rcol]
          X.value=phi_p(X)

          if(X.value<=Temp.value){
            Temp=X
            Temp.value=X.value
          }
        }

        print("Design has been generated, but it may not be the best one for this size.")
        return(Temp)
      }

    }

  }

}
