% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/corrected_distances.R
\name{get_effective_gap}
\alias{get_effective_gap}
\title{Effective Distances between fuel layers}
\usage{
get_effective_gap(effective_depth, number_steps = 1, min_height= 1.5, verbose=TRUE)
}
\arguments{
\item{effective_depth}{Tree metrics with the recalculated depth values after considering distances greater than the actual height bin step (output of [get_real_depths()]
function). An object of the class data frame.}

\item{number_steps}{Numeric value for the number of height bin steps that can be jumped to reshape fuels layers.}

\item{min_height}{Numeric value for the actual minimum base height (in meters).}

\item{verbose}{Logical, indicating whether to display informational messages (default is TRUE).}
}
\value{
A data frame giving the effective distances (> any number of steps) between consecutive fuel layers.
}
\description{
This function recalculates the distance between fuel layers after considering distances greater than any number of height bin steps.
}
\details{
List of tree metrics:
\itemize{
  \item treeID: tree ID with strings and numeric values
  \item treeID1: tree ID with only numeric values
  \item dist: Distance between consecutive fuel layers (m)
  \item dptf: Depth of fuel layers (m) after considering distances greater than the actual height bin step
  \item effdist: Effective distance between consecutive fuel layers (m) after considering distances greater than any number of steps
  \item Hcbh: Base height of each fuel separated by a distance greater than the certain number of steps
  \item Hdist: Height of the distance (> any number of steps) between consecutive fuel layers (m)
  \item Hdptf: Height of the depth of fuel layers (m) after considering distances greater than the actual step
  \item max_height: Maximum height of the tree
}
}
\examples{
library(magrittr)
library(stringr)
library(dplyr)

# Before running this example, make sure to run get_real_depths().
if (interactive()) {
effective_depth <- get_real_depths()
LadderFuelsR::effective_depth$treeID <- factor(LadderFuelsR::effective_depth$treeID)

trees_name1 <- as.character(effective_depth$treeID)
trees_name2 <- factor(unique(trees_name1))

corr_distance_metrics_list <- list()

for (i in levels(trees_name2)) {
tree1 <- effective_depth |> dplyr::filter(treeID == i)
corr_distance_metrics <- get_effective_gap(tree1, number_steps = 1, min_height= 1.5, verbose=TRUE)
corr_distance_metrics_list[[i]] <- corr_distance_metrics
}

# Combine the individual data frames
effective_distances <- dplyr::bind_rows(corr_distance_metrics_list)

# Get original column names
original_column_names <- colnames(effective_distances)

# Specify prefixes
desired_order <- c("treeID", "Hcbh", "dptf","effdist","dist", "Hdist", "Hdptf", "max_")

# Identify unique prefixes
prefixes <- unique(sub("^([a-zA-Z]+).*", "\\\\1", original_column_names))
# Initialize vector to store new order
new_order <- c()

# Loop over desired order of prefixes
for (prefix in desired_order) {
 # Find column names matching the current prefix
matching_columns <- grep(paste0("^", prefix), original_column_names, value = TRUE)
# Append to the new order
new_order <- c(new_order, matching_columns)
}
effective_distances <- effective_distances[, new_order]
}
}
\seealso{
\code{\link{get_real_depths}}
}
\author{
Olga Viedma, Carlos Silva, JM Moreno and A.T. Hudak
}
