% Generated by roxygen2 (4.0.1): do not edit by hand
\name{metab}
\alias{metab}
\title{Calculate metabolism}
\usage{
metab(data, method, wtr.name="wtr", irr.name="irr", do.obs.name="do.obs", ...)
}
\arguments{
\item{data}{a data.frame whose columns are

 \code{"datetime"} = class POSIXct vector

 \code{"do.obs"} = numeric vector of oxygen concentration in mg/L

 \code{"do.sat"} = numeric vector of saturated oxygen concentration in mg/L

 \code{"k.gas"} = numeric vector of gas exchange coefficient values in m/day, should be 0 when depth of do.obs is shallower than z.mix

 \code{"z.mix"} = numeric vector of mixing depth values in meters

 \code{"irr"} = numeric vector of PAR values, arbitrary units

 \code{"wtr"} = numeric vector of water temperature values, arbitrary units

Columns that are not used by a particular statistical method do not need to be supplied.}

\item{method}{a character string specifying one of the 5 statistical methods
(bayesian, bookkeep, kalman, ols, mle)}

\item{wtr.name}{the name of the column containing temperature at the depth of do.obs (predictor variable for R)}

\item{irr.name}{the name of the column containing irradiance (predictor variable for GPP)}

\item{do.obs.name}{the name of the column in data containing the DO observations (in mg/L) to be used as the response variable}

\item{...}{arguments to be passed on to the metabolism model specified by \code{method}}
}
\value{
A data.frame containing columns for year, doy (day of year, julian day plus fraction of day), GPP, R, and NEP
\item{year}{integer year}
\item{doy}{numeric, day of year + fraction of day, where the day is the julian day, and a fraction of 0.5 corresponds to noon}
\item{GPP}{numeric, gross primary production, in units of mg O2 per liter per day. By convention, this value is positive.}
\item{R}{numeric, respiration, in units of mg O2 per liter per day. By convention, this value is negative}
\item{NEP}{numeric, net ecosystem production, in units of mg O2 per liter per day. For most methods this equal GPP+R, but this is not necessarily the case for \code{"method"="bookkeep"}}
Note that different models will have different \link{attributes} attached to them.
}
\description{
Returns daily time series of gross primary production (GPP), respiration (R), and net ecosystem production (NEP). Depending on the method used, other information may be returned as well. Calculations are made using one of 5 statistical methods.
}
\examples{
# fake data
	datetime <- seq(as.POSIXct("2014-06-16 00:00:00", tz="GMT"),
     as.POSIXct("2014-06-17 23:55:00", tz="GMT"), length.out=288*2)
	do.obs <- 2*sin(2*pi*(1/288)*(1:(288*2))+0.75*pi) + 8 + rnorm(288*2, 0, 0.5)
	wtr <- 3*sin(2*pi*(1/288)*(1:(288*2))+pi) + 17 + rnorm(288*2, 0, 0.15)
	do.sat <- LakeMetabolizer:::o2.at.sat.base(wtr, 960)
	irr <- (1500*sin(2*pi*(1/288)*(1:(288*2))+1.5*pi) +650 + rnorm(288*2, 0, 0.25)) *
     ifelse(is.day(datetime, 42.3), 1, 0)
	k.gas <- 0.4
	z.mix <- 1

# plot time series
	plot(wtr, type="l", xaxt="n", yaxt="n", xlab="", ylab="")
	par(new=TRUE); plot(do.obs, type="l", col="blue", xaxt="n", yaxt="n", xlab="", ylab="")
	par(new=TRUE); plot(irr, type="l", col="orange", xaxt="n", yaxt="n", xlab="", ylab="")
	abline(v=144, lty="dotted")
	abline(v=288)
	legend("topleft", legend=c("wtr", "do.obs", "irr"), lty=1,
     col=c("black", "blue", "orange"), inset=c(0.08, 0.01))

# put data in a data.frame
	data <- data.frame(datetime=datetime, do.obs=do.obs, do.sat=do.sat, k.gas=k.gas,
     z.mix=z.mix, irr=irr, wtr=wtr)

# run each metabolism model
	m.bk <- metab(data, "bookkeep", lake.lat=42.6)
	m.ols <- metab(data, "ols", lake.lat=42.6)
	m.mle <- metab(data, "mle", lake.lat=42.6)
	m.kal <- metab(data, "kalman", lake.lat=42.6)
	\dontrun{m.bay <- metab(data, "bayesian", lake.lat=42.6)}

# example attributes
names(attributes(m.ols))
attributes(m.ols)$mod
}
\author{
Ryan D. Batt
}
\seealso{
Metabolism models: \link{metab.bookkeep}, \link{metab.ols}, \link{metab.mle}, \link{metab.kalman}, \link{metab.bayesian}

For smoothing noisy temperature: \link{temp.kalman}

To calculate do.sat: \link{o2.at.sat}

To calculate k.gas: \link{k600.2.kGAS}

To calculate k600 values for k.gas: \link{k.cole}, \link{k.crusius}, \link{k.macIntyre}, \link{k.read}
}
\keyword{metabolism}

