% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/lb.R
\name{lb}
\alias{lb}
\title{Linearized Bregman solver for linear, binomial, multinomial models
 with lasso, group lasso or column lasso penalty.}
\usage{
lb(X, y, kappa, alpha, family = c("gaussian", "binomial", "multinomial"),
  group.type = c("ungrouped", "grouped", "columned"), index = NA,
  intercept = TRUE, normalize = TRUE, iter = 1000)
}
\arguments{
\item{X}{An n-by-p matrix of predictors}

\item{y}{Response Variable}

\item{kappa}{The damping factor of the Linearized Bregman Algorithm that is
defined in the reference paper. See details.}

\item{alpha}{Parameter in Linearized Bregman algorithm which controls the
step-length of the discretized solver for the Bregman Inverse Scale Space.
See details.}

\item{family}{Response type}

\item{group.type}{There are three kinds of group type. "Column" is only
available for multinomial model.}

\item{index}{For group models, the index is a vector that determines the
group of the parameters. Parameters of the same group should have equal
value in index. Be careful that multinomial group model default assumes
the variables in same column are in the same group.}

\item{intercept}{if TRUE, an intercept is included in the model (and not
penalized), otherwise no intercept is included. Default is TRUE.}

\item{normalize}{if TRUE, each variable is scaled to have L2 norm
square-root n. Default is TRUE.}

\item{iter}{Number of iterations.}
}
\value{
A "lb" class object is returned. The list contains the call,
the type, the path, the intercept term a0 and value for alpha, kappa,
iter, and meanvalue, scale factor of X, meanx and normx.
}
\description{
Solver for the entire solution path of coefficients for Linear Bregman iteration.
}
\details{
The Linearized Bregman solver computes the whole regularization path
 for different types of lasso-penalty for gaussian, binomial and
 multinomial models through iterations. It is the Euler forward
 discretized form of the continuous Bregman Inverse Scale Space
 Differential Inclusion. For binomial models, the response variable y
 is assumed to be a vector of two classes which is transformed in to \{1,-1\}.
 For the multinomial models, the response variable y can be a vector of k classes
 or a n-by-k matrix that each entry is in \{0,1\} with $1$ indicates
 the class. Under all circumstances, two parameters, kappa
 and alpha need to be specified beforehand. The definitions of kappa
 and alpha are the same as that defined in the reference paper.
 Parameter alpha is defined as stepsize and kappa is the damping factor
 of the Linearized Bregman Algorithm that is defined in the reference paper.
}
\examples{
#Examples in the reference paper
library(MASS)
n = 200;p = 100;k = 30;sigma = 1
Sigma = 1/(3*p)*matrix(rep(1,p^2),p,p)
diag(Sigma) = 1
A = mvrnorm(n, rep(0, p), Sigma)
u_ref = rep(0,p)
supp_ref = 1:k
u_ref[supp_ref] = rnorm(k)
u_ref[supp_ref] = u_ref[supp_ref]+sign(u_ref[supp_ref])
b = as.vector(A\%*\%u_ref + sigma*rnorm(n))
kappa = 16
alpha = 1/32
max_step = 160
object <- lb(A,b,kappa,alpha,family="gaussian",group="ungrouped",
             intercept=FALSE,normalize=FALSE,iter=max_step)
attach(object, warn.conflicts= FALSE)
plot((0:max_step)*alpha,c(0,path[,1]),type="l",xlim=c(0,3),
     ylim=c(min(path),max(path)),xlab="t",ylab=bquote(beta),
     main=bquote(paste("LB ",kappa,"=",.(kappa))))
for (j in 2:100){
  points((0:max_step)*alpha,c(0,path[,j]),type="l")
}
detach(object)


#Diabetes, linear case
library(Libra)
data(diabetes)
attach(diabetes)
object <- lb(x,y,100,1e-3,family="gaussian",group="ungrouped",iter=1000)
plot(object)

#Simulated data, binomial case
X <- matrix(rnorm(1000*256), nrow=1000, ncol=256)
alpha <- c(rep(1,50), rep(0,206))
y <- 2*as.numeric(runif(1000)<1/(1+exp(-X \%*\% alpha)))-1
group = c(rep(1:5,10),rep(6:108,2))
result <- lb(X,y,kappa=5,alpha=1,family="binomial",
             intercept=FALSE,normalize = FALSE,iter=500)
plot(result)

#Simulated data, multinomial case
X <- matrix(rnorm(1000*256), nrow=1000, ncol=256)
alpha <- matrix(c(rnorm(50*3), rep(0,206*3)),nrow=3)
P <- exp(alpha\%*\%t(X))
P <- scale(P,FALSE,apply(P,2,sum))
y <- rep(0,1000)
rd <- runif(1000)
y[rd<P[1,]] <- 1
y[rd>1-P[3,]] <- -1
result <- lb(X,y,kappa=5,alpha=0.1,family="multinomial",
 group.type="columned",intercept=FALSE,normalize = FALSE,iter=500)
plot(result)
}
\author{
Feng Ruan, Jiechao Xiong and Yuan Yao
}
\references{
Ohser, Ruan, Xiong, Yao and Yin, Sparse Recovery via Differential
 Inclusions, \url{http://arxiv.org/abs/1406.7728}
}
\keyword{regression}

