calc_SourceDoseRate <- structure(function(#Calculation of the source dose rate via the date of measurement 
  ### Calculating the dose rate of the irradiation source via the date of measurement
  ### based on: source calibration date, source dose rate, dose rate error.
  ### The function returns a data.frame that provides the input argument dose_rate 
  ### for the function \code{\link{Second2Gray}}. 
  
  # ===========================================================================
  ##author<<
  ## Margret C. Fuchs, AWI Potsdam (Germany), \cr
  ## Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne (France), \cr
  
  ##section<<
  ## version 0.1
  # ===========================================================================
  
  measurement.date,
  ### \code{\link{character}} (\bold{required}): date of measurement in "YYYY-MM-DD"
  
  calib.date,
  ### \code{\link{character}} (\bold{required}): date of source calibration in "YYYY-MM-DD"
  
  calib.dose.rate,
  ### \code{\link{numeric}} (\bold{required}): dose rate at date of calibration in Gy/s or Gy/min
  
  calib.error,
  ### \code{\link{numeric}} (\bold{required}): error of dose rate at date of calibration Gy/s or Gy/min
  
  source.type = "Sr-90",
  ### \code{\link{character}} (with default): specify irrdiation source 
  ### (\code{Sr-90} or \code{Co-60} or \code{Am-214}), see details for further information
 
  dose.rate.unit = "Gy/s"
  ### \code{\link{character}} (with default): specify dose rate unit for input 
  ### (\code{Gy/min} or \code{Gy/s}), the output is given in Gy/s
  ### as valid for the function \code{\link{Second2Gray}}
  
){
  

  # -- calc days since source calibration
  decay.time <- as.Date(c(measurement.date, calib.date))
  decay.days <- decay.time[1] - decay.time[2]

  
  # -- calc dose rate of source at date of measurement, considering the chosen source-type
  
  ##set halflife
  halflife.years  <- switch(
    source.type,
    "Sr-90" = 28.90,  
    "Am-241" = 432.6,
    "Co-60" = 5.274)
         
  if(is.null(halflife.years)){
  
    stop("[calc_SourceDoseRate()] Source type unknown or currently not supported!")
    
  }
  

    halflife.days  <- halflife.years * 365
    
    # N(t) = N(0)*e^((lambda * t) with lambda = log(2)/T1.2)
    measurement.dose.rate <- (calib.dose.rate) * 
                              exp((-log(2) / halflife.days) * as.numeric(decay.days))
    measurement.dose.rate.error <- (calib.error) * 
                              exp((-log(2) / halflife.days) * as.numeric(decay.days))

  
  
  # -- convert to input unit to [Gy/s]
  if(dose.rate.unit == "Gy/min"){
        source.dose.rate <- measurement.dose.rate / 60
        source.dose.rate.error <- source.dose.rate * 
    (measurement.dose.rate.error / measurement.dose.rate)

  }else if(dose.rate.unit == "Gy/s"){
        source.dose.rate <- measurement.dose.rate
        source.dose.rate.error <- measurement.dose.rate.error
     
  }
  
  
  # Output --------------------------------------------------------------------------------------
  
  dose_rate <- data.frame(dose.rate = source.dose.rate, dose.rate.error = source.dose.rate.error)
  
  temp.return <- set_RLum.Results(
    data = list(
      dose.rate = dose_rate,
      parameters = list(source.type = source.type, 
                        halflife = halflife.years, 
                        dose.rate.unit = dose.rate.unit)
    ))
  
  return(temp.return)

  # DOCUMENTATION - INLINEDOC LINES -----------------------------------------
  
  ##details<<
  ## Calculation of the source dose rate based on the time elapsed since the last 
  ## calibration of the irradiation source. Decay parameters assume a Sr-90 
  ## beta source.
  ## \deqn{dose.rate = D0 * exp(-log(2) / T.1/2 * t)} \cr
  ## with:
  ## D0   <- calibration dose rate
  ## T.1/2 <- half-life of the source nuclide (here in days)
  ## t    <- time since source calibration (in days)
  ## log(2) / T.1/2 equals the decay constant lambda
  ##
  ## Information on the date of measurements may be taken from  
  ## the data's original .BIN file 
  ## (using e.g., BINfile <- readBIN2R() and the slot BINfile@METADATA$DATE) 
  ##
  ## \bold{Allowed source types and related values}
  ##
  ## \tabular{rllll}{
  ## \bold{#}   \tab \bold{Source type} \tab \bold{T.1/2} \tab \bold{Reference} \cr
  ## [1]        \tab Sr-90              \tab 28.90 y      \tab NNDC, Brookhaven National Laboratory \cr
  ## [2]        \tab Am-214             \tab 432.6 y      \tab NNDC, Brookhaven National Laboratory \cr
  ## [3]        \tab Co-60              \tab 5.274 y      \tab NNDC, Brookhaven National Laboratory 
  ## }  
  
  ##value<<
  ## Returns an S4 object of type \code{\linkS4class{RLum.Results}}. 
  ## Slot \code{data} contains a \code{\link{list}} with the following structure:\cr 
  ## $ dose.rate (data.frame)\cr
  ## .. $ dose.rate  \cr        
  ## .. $ dose.rate.error   \cr     
  ## $ parameters (list) \cr
  ## .. $ source.type\cr
  ## .. $ halflife\cr
  ## .. $ dose.rate.unit
  
  ##references<<
  ## NNDC, Brookhaven National Laboratory (\code{http://www.nndc.bnl.gov/})
  
  ##note<<
  ## # 
  
  ##seealso<<
  ## \code{\link{Second2Gray}}
  
  ##keyword<<
  ## manip
  
}, ex=function(){
  
  ##(1) Simple function usage
  ##Basic calculation of the dose rate for a specific date
  dose.rate <-  calc_SourceDoseRate(measurement.date = "2012-01-27",
                                    calib.date = "2014-12-19",
                                    calib.dose.rate = 0.0438,
                                    calib.error = 0.0019)
  
  ##show results
  get_RLum.Results(dose.rate)

  ##(2) Usage in combination with another function (e.g., Second2Gray() )
  ## load example data 
  data(ExampleData.DeValues, envir = environment())
  
  ## use the calculated variable dose.rate as input argument
  ## to convert De(s) to De(Gy)
  Second2Gray(ExampleData.DeValues, dose.rate)
  
  
})   