\name{plot_KDE}
\alias{plot_KDE}
\title{Plot kernel density estimate with statistics}
\description{Plot a kernel density estimate of measurement values in combination with
the actual values and associated error bars in ascending order.
Optionally, statistical measures such as mean, median, standard
deviation, standard error and quartile range can be provided visually
and numerically.}
\usage{plot_KDE(data, na.exclude = TRUE, weights = FALSE, values.cumulative = TRUE, 
    centrality, dispersion, summary, summary.pos, polygon.col, 
    order = TRUE, bw = "nrd0", output = FALSE, ...)}
\arguments{
  \item{data}{\code{\link{data.frame}} or \code{\linkS4class{RLum.Results}} object
(required): for \code{data.frame}: two columns: De (\code{values[,1]})
and De error (\code{values[,2]}). For plotting multiple data sets, these
must be provided as \code{list} (e.g. \code{list(dataset1, dataset2)}).}
  \item{na.exclude}{\code{\link{logical}} (with default): exclude NA values from the data
set prior to any further operations.}
  \item{weights}{\code{\link{logical}} (with default): calculate the KDE with De-errors
as weights. Attention, using errors as weights will result in a plot
similar to a a probability density plot, with all ambiguities related
to this plot type!}
  \item{values.cumulative}{\code{\link{logical}} (with default): show cumulative individual data.}
  \item{centrality}{\code{\link{character}}: measure(s) of centrality, used for
plotting vertical lines of the respective measure. Can be
one out of \code{"mean"}, \code{"median"}, \code{"mean.weighted"},
\code{"median.weighted"} and \code{"kdemax"}.}
  \item{dispersion}{\code{\link{character}}: measure of dispersion, used for
drawing the polygon that depicts the dose distribution. One out of
\code{"sd"} (standard deviation),\code{"2sd"} (2 standard deviations)
\code{"qr"} (quartile range).}
  \item{summary}{\code{\link{character}} (optional): adds numerical output to the plot.
Can be one or more out of: \code{"n"} (number of samples), \code{"mean"} (mean De
value), \code{"mean.weighted"} (error-weighted mean), \code{"median"} (median of
the De values), \code{"sdrel"} (relative standard deviation in
percent), \code{"sdabs"} (absolute standard deviation), \code{"serel"} (relative
standard error), \code{"seabs"} (absolute standard error), \code{"kdemax"} (maximum
of the KDE), \code{"skewness"} (skewness) and \code{"kurtosis"} (kurtosis).}
  \item{summary.pos}{\code{\link{numeric}} or \code{\link{character}} (with default): optional
position coordinates or keyword (e.g. \code{"topright"}) for the
statistical summary. Alternatively, the keyword \code{"sub"} may be
specified to place the summary below the plot header. However, this
latter option in only possible if \code{mtext} is not used. In case
of coordinate specification, y-coordinate refers to the right y-axis.}
  \item{polygon.col}{\code{\link{character}} or \code{\link{numeric}} (with default): colour
of the polygon showing the dose dispersion around the central value.
Only relevant if \code{dispersion} is specified.}
  \item{order}{\code{\link{logical}}: Order data in ascending order.}
  \item{bw}{\code{\link{character}} (with default): bin-width, chose a numeric
value for manual setting.}
  \item{output}{\code{\link{logical}}: Optional output of numerical plot parameters.
These can be useful to reproduce similar plots. Default is \code{FALSE}.}
  \item{\dots}{further arguments and graphical parameters passed to \code{\link{plot}}.}
}
\details{The function allow passing several plot arguments, such as \code{main},
\code{xlab}, \code{cex}. However, as the figure is an overlay of two
separate plots, \code{ylim} must be specified in the order: c(ymin_axis1,
ymax_axis1, ymin_axis2, ymax_axis2) when using the cumulative values plot
option. Similarly, if other than the default
colours are desired, the argument col must be provided with colours in
the following order: probability density function, De values, De error
bars, sd or qr polygon. The line type (\code{lty}) for additional
measures of centrality will cycle through the default values (1, 2, ...)
by default, i.e. KDE line solid, further vertical lines dashed, dotted,
dash-dotted and so on. To change this behaviour specify the desired
order of line types (e.g. \code{lty = c(1, 3, 2, 5)}).
See examples for some further explanations. For
details on the calculation of the bin-width (parameter \code{bw}) see
\code{\link{density}}.}


\author{Michael Dietze, GFZ Potsdam (Germany),\cr
Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne, \cr
R Luminescence Package Team}
\note{The plot output is no 'PD' plot (cf. the discussion of Berger and
Galbraith in Ancient TL; see references)!}


\seealso{\code{\link{density}}, \code{\link{plot}}}
\examples{
## read example data set
data(ExampleData.DeValues, envir = environment())
ExampleData.DeValues <-
  Second2Gray(ExampleData.DeValues$BT998, c(0.0438,0.0019))

## create plot straightforward
plot_KDE(data = ExampleData.DeValues)

## create plot with logarithmic x-axis
plot_KDE(data = ExampleData.DeValues,
         log = "x")

## create plot with user-defined labels and axes limits
plot_KDE(data = ExampleData.DeValues,
         main = "Dose distribution",
         xlab = "Dose [s]",
         ylab = c("KDE estimate", "Cumulative dose value"),
         xlim = c(100, 250),
         ylim = c(0, 0.08, 0, 30))

## create plot with centrality lines and distribution polygons
plot_KDE(data = ExampleData.DeValues,
         ylim = c(0, 0.08, 0, 35),
         centrality = c("median", "mean"),
         dispersion = "sd",
         polygon.col = "lightblue")

## create plot with statistical summary below header
plot_KDE(data = ExampleData.DeValues,
         summary = c("n", "median", "skewness", "qr"))

## create plot with statistical summary as legend
plot_KDE(data = ExampleData.DeValues,
         summary = c("n", "mean", "sdrel", "seabs"),
         summary.pos = "topleft")

## split data set into sub-groups, one is manipulated, and merge again
data.1 <- ExampleData.DeValues[1:15,]
data.2 <- ExampleData.DeValues[16:25,] * 1.3
data.3 <- list(data.1, data.2)

## create plot with two subsets straightforward
plot_KDE(data = data.3)

## create plot with two subsets and summary legend at user coordinates
plot_KDE(data = data.3,
         summary = c("n", "median", "skewness"),
         summary.pos = c(110, 0.07),
         col = c("blue", "orange"))

## example of how to use the numerical output of the function
## return plot output to draw a thicker KDE line
KDE <- plot_KDE(data = ExampleData.DeValues,
                output = TRUE)

## read out coordinates of KDE graph
KDE.x <- KDE$De.density[[1]]$x
KDE.y <- KDE$De.density[[1]]$y

## transform y-values to right y-axis dimensions
KDE.y <- KDE.y / max(KDE.y) * (nrow(ExampleData.DeValues) - 1) + 1

## draw the KDE line
lines(x = KDE.x,
      y = KDE.y,
      lwd = 3)
}
\section{Function version}{3.5 (2015-03-04 13:25:51)}
