% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.functions.R
\name{mb.run}
\alias{mb.run}
\title{Run MBNMA time-course models}
\usage{
mb.run(network, parameters.to.save = NULL, fun = "linear",
  user.fun = NULL, alpha = "study", beta.1 = list(pool = "rel",
  method = "common"), beta.2 = NULL, beta.3 = NULL, beta.4 = NULL,
  positive.scale = FALSE, intercept = TRUE, rho = NULL,
  covar = NULL, var.scale = NULL, class.effect = list(),
  UME = FALSE, pd = "pv", parallel = TRUE, priors = NULL,
  n.iter = 10000, n.chains = 3, n.burnin = floor(n.iter/2),
  n.thin = max(1, floor((n.iter - n.burnin)/1000)), model.file = NULL,
  arg.params = NULL, ...)
}
\arguments{
\item{network}{An object of class \code{mb.network}.}

\item{parameters.to.save}{A character vector containing names of parameters
to monitor in JAGS}

\item{fun}{is a character specifying a functional form to be assigned to the
time-course. Options are given in \code{details}.}

\item{user.fun}{A character specifying any relationship including \code{time} and
one/several of: \code{alpha}, \code{beta.1}, \code{beta.2}, \code{beta.3}, \code{beta.4}.}

\item{alpha}{Refers to the baseline mean response and is a character object
that can take either:
\itemize{
\item \code{"study"} to constrain baseline to be equal for all
arms within a study (\code{i} index is added))
\item \code{"arm"} to allow baseline to
vary between arms within a study (\code{i,k} index is added)).
}}

\item{beta.1}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{beta.2}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{beta.3}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{beta.4}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{positive.scale}{A boolean object that indicates whether all continuous
mean responses (y) are positive and therefore whether the baseline response
should be given a prior that constrains it to be positive.}

\item{intercept}{A boolean object that indicates whether an intercept is to
be included in the model. Can be used to imply whether mean responses in
data are change from baseline (\code{FALSE}) or not (setting it to \code{FALSE}
removes the intercept, \code{alpha}, from the model).}

\item{rho}{The correlation coefficient when modelling correlation between time points. If left
as \code{NULL} (the default) then this implies modelling no correlation between time points.
Can either be assigned the string \code{"estimate"} to indicate that rho should be estimated
from the data, or assigned a numeric value, which fixes \code{rho} in the model to the assigned
value, either for when \code{rho} is calculated externally or for use in deterministic sensitivity
analyses.}

\item{covar}{A character specifying the covariance structure to use for the
multivariate normal likelihood. Can currently take either \code{"CS"} (compound
symmetry) or \code{"AR1"} (autoregressive AR1).}

\item{var.scale}{A numeric vector indicating the relative scale of variances between
correlated time-course parameters when relative effects are modelled on more than
one time-course parameter(see Details LINK). Each element of
the vector refers to the relative scale of each of the time-course parameters that is
modelled using relative effects.}

\item{class.effect}{A list of named strings that determines which time-course
parameters to model with a class effect and what that effect should be
(\code{"common"} or \code{"random"}). For example: \code{list("beta.2"="common", "beta.3"="random")}.}

\item{UME}{Can take either \code{TRUE} or \code{FALSE} (for an unrelated mean effects
model on all or no time-course parameters respectively) or can be a vector
of parameter name strings to model as UME. For example: \code{c("beta.1", "beta.2")}.}

\item{pd}{Can take either:
\itemize{
\item \code{pv} only pV will be reported (as automatically outputted by R2jags).
\item \code{plugin} calculates pD by the plug-in
method \insertCite{spiegelhalter2002}{MBNMAtime}. It is faster, but may output negative
non-sensical values, due to skewed deviances that can arise with non-linear models.
\item \code{pd.kl} calculates pD by the Kullback–Leibler divergence \insertCite{plummer2008}{MBNMAtime}. This
will require running the model for additional iterations but
will always produce a positive result.
\item \code{popt} calculates pD using an optimism adjustment which allows for calculation
of the penalized expected deviance \insertCite{plummer2008}{MBNMAtime}
}}

\item{parallel}{A boolean value that indicates whether JAGS should be run in
parallel (\code{TRUE}) or not (\code{FALSE}). If \code{TRUE} then the number of cores to
use is automatically calculated.}

\item{priors}{A named list of parameter values (without indices) and
replacement prior distribution values given as strings
\strong{using distributions as specified in JAGS syntax}.}

\item{n.iter}{number of total iterations per chain (including burn in; default: 15000)}

\item{n.chains}{number of Markov chains (default: 3)}

\item{n.burnin}{length of burn in, i.e. number of iterations to discard at the
beginning. Default is \code{n.iter/2``, that is, discarding the first half of the simulations. If }n.burnin` is 0, jags() will run 100 iterations for adaption.}

\item{n.thin}{thinning rate. Must be a positive integer. Set \code{n.thin > 1`` to save memory and computation time if }n.iter\code{is large. Default is}max(1, floor(n.chains * (n.iter-n.burnin) / 1000))`` which will only thin if there are at least 2000
simulations.}

\item{model.file}{A JAGS model written as a character object that can be used
to overwrite the JAGS model that is automatically written based on the
specified options. Useful when amending priors using replace.prior()}

\item{arg.params}{Contains a list of arguments sent to \code{mb.run()} by time-course
specific wrapper functions}

\item{...}{Arguments to be sent to R2jags.}
}
\value{
An object of S3 class \code{c("mbnma", "rjags")`` containing parameter results from the model. Can be summarized by }print()\code{and can check traceplots using}R2jags::traceplot()\code{or various functions from the package}mcmcplots`.

Nodes that are automatically monitored (if present in the model) have the
following interpretation. They will have an additional suffix that relates
to the name/number of the time-course parameter to which they correspond
(e.g. \code{d.et50} or \code{d.1}):
\itemize{
\item \code{d} The pooled relative effect for a given
treatment compared to the network reference treatment for a particular
time-course parameter, reported if \code{pool="rel"}
\item \code{sd.d} The between-study SD (heterogeneity) for relative effects,
reported if \code{pool="rel"} and \code{method="random"}
\item \code{D} The class effect for a given class compared to the
network reference class for a particular time-course parameter
\item \code{sd.D} The standard deviation for the pooled relative effects of treatments within a
given class from a model with a random class effect.
\item \code{beta} If \code{pool="const"} then only a single node will be present in the
output, which corresponds to the absolute value of a particular time-course
parameter across the network, If \code{pool="arm"}
then for the relevant time-course parameter there will be one node for
each treatment, which represents the absolute value of the time-course
parameter for each treatment
\item \code{sd.beta} Reported if \code{method="random"} and \code{pool} is either \code{"const"} or \code{"arm"}.
If \code{pool="const"} this represents the between-study SD for the absolute value of a particular
time-course parameter exchangeable across the network. If \code{pool="arm"}
this represents the between-study SD for the absolute value of a
particular time-course parameter exchangeable by treatment
\item \code{rho} The correlation coefficient for correlation between time-points. Its
interpretation will differ depending on the covariance structure used
\item \code{totresdev} The residual deviance of the model
\item \code{deviance} The deviance of the model
}

If there are errors in the JAGS model code then the object will be a list
consisting of two elements - an error message from JAGS that can help with
debugging and \code{model.arg}, a list of arguments provided to \code{mb.run()}
which includes \code{jagscode}, the JAGS code for the model that can help
users identify the source of the error.
}
\description{
Fits a Bayesian time-course model for model-based network meta-analysis
(MBNMA) that can account for repeated measures over time within studies by
applying a desired time-course function. Follows the methods of \insertCite{pedder2019;textual}{MBNMAtime}.
}
\section{Time-course parameters}{

Time-course parameters in the model must be provided as a list with named elements
\code{pool} and \code{method}.

\code{pool} is used to define the approach used for pooling of a given time-course parameter and
can take any of the following values:
\itemize{
\item \code{"rel"} indicates that relative effects should be pooled for this time-course parameter.
This preserves randomisation
within included studies, are likely to vary less between studies (only due to effect modification),
and allow for testing of consistency between direct and indirect evidence. Pooling follows the
general approach for Network Meta-Analysis proposed by \insertCite{lu2004;textual}{MBNMAtime}.
\item \code{"arm"} indicates that study arms should be pooled within each treatment in the network
for this time-course parameter.
This allows estimation of absolute time-course parameter values, but makes stronger assumptions
regarding similarity of studies.
\item \code{"const"} indicates that study arms should be pooled across the whole network for this
time-course parameter  \emph{independently of assigned treatment}.
This implies using a single value across the network for this time-course parameter,
and may therefore be making very strong assumptions of similarity.
}

\code{method} is used to define the model used for meta-analysis for a given time-course parameter
and can take any of the following values:
\itemize{
\item \code{"common"} implies that all studies estimate the same true effect
(akin to a "fixed effects" meta-analysis)
\item \code{"random"} implies that all studies estimate a separate true effect, but that each
of these true effects vary randomly around a true mean effect. This approach allows
for modelling of between-study heterogeneity.
\item \code{numeric()} Assigned a numeric value - this can only be used if \code{pool="const"}. It indicates that
this time-course parameter should not be estimated from the data but should be assigned
the numeric value determined by the user. This can be useful for fixing specific time-course
parameters (e.g. Hill parameters in Emax functions or knot location in piecewise functions).
}

When relative effects are modelled on more than one time-course parameter,
correlation between the time-course parameters is automatically
estimated using a vague Wishart prior. This prior can be made slightly more informative
by specifying the relative scale of variances between the time-course parameters using
\code{var.scale}.
}

\section{Time-course function}{

Several general time-course functions are provided, but a
user-defined time-course relationship can instead be used.

Built-in time-course functions are:
\itemize{
\item \code{"linear"}: \code{beta.1} refers to the gradient
\item \code{"quadratic"}: \code{beta.1} refers to the gradient, \code{beta.2} refers
to the change in gradient
\item \code{"exponential"}: \code{beta.1} refers to the rate of gain/decay
\item \code{"emax"} (emax without a Hill parameter): \code{beta.1} refers to
Emax parameter, \code{beta.2} refers to ET50 parameter
\item \code{"emax.hill"} (emax with a Hill parameter): \code{beta.1} refers to Emax parameter, \code{beta.2} refers
to ET50 parameter, \code{beta.3} refers to Hill parameter
\item \code{"fract.poly.first"} (first-order fractional polynomial): \code{beta.1} refers to the slope
parameter, \code{beta.3} refers to the power parameter
\item \code{"fract.poly.second"} (second-order fractional polynomial): \code{beta.1} refers to the first slope
parameter, \code{beta.2} refers to the first power parameter, \code{beta.3} refers to
the first power parameter, \code{beta.4} refers to the second power parameter
\item \code{"piecelinear"} piecewise linear: \code{beta.1} refers to the gradient of the
first linear piece, \code{beta.2} refers to the gradient of the second linear
piece, \code{beta.3} refers to the knot location (the time at which the two
pieces are joined)
\item \code{"user"} (user-defined function: \code{user.fun} must be specified in arguments)
}
}

\section{Correlation between observations}{

When modelling correlation between observations using \code{rho}, values for \code{rho} must imply a
positive semidefinite covariance matrix. If estimating \code{rho} from the data (by assigning it
\code{"estimate"}), the default prior distribution (\code{dunif(-1,1)}) may include values that exclude
a positive semidefinite covariance matrix. This prior can be restricted (e.g. to \code{dunif(0,1)})
using the \code{priors} argument (see \code{\link{get.prior}})
}

\examples{
\donttest{
# Create mb.network object
network <- mb.network(osteopain)

# Fit a linear time-course MBNMA with random consistency treatment effects on beta.1 (slope)
#and equal baselines
#in study arms
mb.run(network, fun="linear",
  alpha="study", beta.1=list(pool="rel", method="random"))

# Fit an emax time-course MBNMA with fixed consistency treatment effects on beta.1 (emax),
#a common parameter estimated across the network for beta.2 (et50) and a Hill parameter of 0.5
#across the network on data reported as change from baseline
result <- mb.run(network, fun="emax.hill",
  beta.1=list(pool="rel", method="common"),
  beta.2=list(pool="const", method="common"),
  beta.3=list(pool="const", method=0.5),
  intercept=TRUE)


####### Examine MCMC diagnostics (using mcmcplots package) #######

# Density plots
mcmcplots::denplot(result, c("beta.2", "deviance", "totresdev"))

# Traceplots
mcmcplots::traplot(result)

# Caterpillar plots
mcmcplots::caterplot(result, "d.1")


########## Output ###########

# Print R2jags output and summary
print(result)
summary(result)

# Plot forest plot of results
plot(result)


###### Additional model arguments ######

# Fit a linear time-course MBNMA that accounts for correlation between time points
mb.run(network, fun="linear",
  beta.1=list(pool="rel", method="common"),
  rho="estimate", covar="CS")

# Define a user-defined time-course relationship for use in mb.run
time.fun <- "alpha + exp(beta.1 * time) + (time^beta.2)"

# Run model using Kullback-Liebler divergence to calculate pD
mb.run(network, fun="user", user.fun=time.fun,
  beta.1=list(pool="rel", method="random"),
  beta.2=list(pool="rel", method="common"),
  pd="pd.kl")
}
}
\references{
\insertAllCited
}
