#' Compute the Density-based Silhouette
#' 
#' Computes the Density-based Silhouette for a 'soft' clustering assignment matrix.
#' @param z A numeric matrix such that rows correspond to observations, columns correspond to clusters, and rows sum to \code{1}.
#' @param ztol A small (single, numeric, non-negative) tolerance parameter governing whether small assignment probabilities are treated instead as crisp assignments. Defaults to \code{1E-100}.
#' @param weights An optional numeric vector giving observation-specific weights for computing the (weighted) mean/median DBS (see \code{summ}).
#' @param summ A single character string indicating whether the (possibly weighted) "\code{mean}" (the default) or "\code{median}" DBS should be computed.
#' @param ... Catches unused arguments.
#'
#' @return A list with the following elements:
#' \describe{
#' \item{\code{silvals}}{A matrix where each row contains the cluster to which each observation belongs in the first column and the observation-specific DBS width in the second column.}
#' \item{\code{msw}}{Depending on the value of \code{summ}, either the mean or median DBS width.}
#' \item{\code{wmsw}}{Depending on the value of \code{summ}, either the weighted mean or weighted median DBS width.}}
#' @note When calling \code{\link{MEDseq_fit}}, the \code{summ} argument can be passed via the \code{...} construct, in which case it governs both the DBS and ASW criteria.
#' @importFrom matrixStats "rowSums2" "weightedMedian" "weightedMean"
#' @importFrom TraMineR "seqdef"
#' @references Menardi, G. (2011). Density-based Silhouette diagnostics for clustering methods. \emph{Statistics and Computing} 21(3): 295-308.
#' @export 
#' @seealso \code{\link{MEDseq_fit}}
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords utility
#' @usage
#' dbs(z,
#'     ztol = 1E-100,
#'     weights = NULL,
#'     summ = c("mean", "median"),
#'     ...)
#' @examples
#' # Generate a toy z matrix
#' z <- abs(matrix(rnorm(50), ncol=2))
#' z <- z/rowSums(z)
#' 
#' # Return the mean DBS width
#' dbs(z)$msw
#' 
#' # For real sequence data
#' data(mvad)
#' \donttest{
#' mod <- MEDseq_fit(seqdef(mvad[,15:86]), G=10, modtype="UCN", weights=mvad$weight)
#' 
#' dbs(mod$z, weights=mvad$weight)}
dbs               <- function(z, ztol = 1E-100, weights = NULL, summ = c("mean", "median"), ...) {
  if(any(!is.matrix(z), !is.numeric(z)) ||
     ncol(z)      <= 1     ||
     nrow(z)      <= 1)          stop("'z' must be a numeric matrix with 2 or more columns & 2 or more rows", call.=FALSE)
  z               <- .renorm_z(z)
  if(length(ztol)  > 1     ||
     !is.numeric(ztol)     ||
     ztol          < 0)          stop("Invalid 'ztol'", call.=FALSE)
  if(!missing(weights))     {
    N             <- nrow(z)
   if(!is.numeric(weights) ||
      length(weights)      != N) stop(paste0("'weights' must be a numeric vector of length N=", N), call.=FALSE)
   if(any(weights < 0)     || 
      any(!is.finite(weights)))  stop("'weights' must be positive and finite", call.=FALSE)
  }
  if(!missing(summ)        && 
    (length(summ)  > 1     ||
     !is.character(summ)))       stop("'summ' must be a single character string", call.=FALSE)
  summ            <- match.arg(summ)
  MAP             <- if(any(names(list(...)) == "MAP")) list(...)$MAP else max.col(z)
  z               <- matrix(z[order(row(z), -z)], nrow(z), byrow=TRUE)
  l2              <- log(z[,2L])
  zz              <- log(z[,1L])     - l2
  zz.inf          <- is.infinite(zz) | l2 < log(ztol)
  ds              <- zz/max(abs(zz[!zz.inf]))
  ds[zz.inf]      <- 1L
  ds[is.nan(ds)]  <- 0L
  DS              <- cbind(cluster=MAP, dbs_width=ds)
  class(DS)       <- "MEDsil"
  msw             <- switch(EXPR=summ, median=stats::median(ds), mean=mean(ds))
  dbs_res         <- list(silvals = DS, msw = msw, wmsw = ifelse(is.null(weights), msw, 
                          switch(EXPR=summ, median=weightedMedian(ds, weights), mean=weightedMean(ds, weights))))
  attr(dbs_res, "summ")    <- summ
    return(dbs_res)
}

#' Extract results from a MEDseq model
#'
#' Utility function for extracting results of submodels from "\code{MEDseq}" object when a range of models were run via \code{\link{MEDseq_fit}}.
#' @param x An object of class \code{"MEDseq"} generated by \code{\link{MEDseq_fit}} or an object of class \code{"MEDseqCompare"} generated by \code{\link{MEDseq_compare}}.
#' @param what A character string indicating the desired results to extract.
#' @param rank A number indicating what \code{rank} model results should be extracted from, where the \code{rank} is determined by \code{criterion}. Defaults to \code{1}, i.e. the best model.
#' @param criterion The \code{criterion} used to determine the ranking. Defaults to "\code{dbs}".
#' @param G Optional argument giving the number of components in the model for which results are desired.
#' @param modtype Optional argument the desired model type for which results are desired.
#' @param noise A logical indicating whether models with a noise component should be considered. Defaults to \code{TRUE}.
#' @param ... Catches unused arguments.
#'
#' @return The desired results extracted from the \code{MEDseq} model.
#' @details The arguments \code{rank} and \code{criterion} are invoked when one or more of the arguments \code{G} and \code{modtype} are missing. Thus, supplying \code{G} and \code{modtype} allows \code{rank} and \code{criterion} to be bypassed entirely.
#' @note Arguments to this function can be supplied to \code{\link{plot.MEDseq}} via the \code{...} construct.
#' @export
#' @importFrom TraMineR "seqdef"
#' @seealso \code{\link{MEDseq_fit}}, \code{\link{plot.MEDseq}}
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords utility
#' @usage
#' get_MEDseq_results(x,
#'                    what = c("z", "MAP", "DBS", "ASW"), 
#'                    rank = 1L, 
#'                    criterion = c("dbs", "asw", "bic", "icl", 
#'                                  "aic", "cv", "nec", "loglik"), 
#'                    G = NULL, 
#'                    modtype = NULL, 
#'                    noise = TRUE, 
#'                    ...)
#' @examples
#' \donttest{data(biofam)
#' 
#' mod <- MEDseq_fit(seqdef(biofam[10:25] + 1L), G=9:10)
#' 
#' # Extract the MAP clustering of the best 9-cluster model according to the asw criterion
#' get_MEDseq_results(mod, what="MAP", G=9, criterion="asw")}
get_MEDseq_results            <- function(x, what = c("z", "MAP", "DBS", "ASW"), rank = 1L, criterion = c("dbs", "asw", "bic", "icl", "aic", "cv", "nec", "loglik"), G = NULL, modtype = NULL, noise = TRUE, ...) {
    UseMethod("get_MEDseq_results")
}

#' @method get_MEDseq_results MEDseq
#' @export
get_MEDseq_results.MEDseq     <- function(x, what = c("z", "MAP", "DBS", "ASW"), rank = 1L, criterion = c("dbs", "asw", "bic", "icl", "aic", "cv", "nec", "loglik"), G = NULL, modtype = NULL, noise = TRUE, ...) {
  x               <- if(inherits(x, "MEDseqCompare")) x$optimal else x
  if(!missing(what)           && 
     (length(what) > 1        ||
     !is.character(what)))       stop("'what' must be a single character string",    call.=FALSE)
  what            <- match.arg(what)
  minG            <- 1L  + is.element(what, c("DBS", "ASW"))
  if(!(missing(G) -> m.G)     &&
    (length(G)    != 1        ||
     !is.numeric(G)           ||
     (G < minG    || floor(G) != G))) {
    if(is.element(what, 
       c("dbs", "asw"))) {       stop(paste0("'G' must be a single integer > 1 when 'what'=", what), call.=FALSE)
    } else                       stop("'G' must be a single integer >= 1",           call.=FALSE)
  }  
  if(!(missing(modtype) ->
       m.M) &&
    (length(modtype) > 1      ||
     !is.character(modtype)))    stop("'modtype' must be a single character string", call.=FALSE)
  if(length(noise) > 1        ||
     !is.logical(noise))         stop("'noise' must be a single logical indicator",  call.=FALSE)
  if(!missing(criterion)      ||
     !missing(rank)           ||
     any(m.G, m.M))            {
    if(!missing(criterion)    && 
      (length(criterion)  > 1 ||
      !is.character(criterion))) stop("'criterion' must be a single character string", call.=FALSE)
    criterion     <- match.arg(criterion)
    if((criterion == "nec"    ||
       criterion  == "dbs"    ||
       criterion  == "asw")   &&
      !m.G  &&  G == 1)          stop(paste0("Can't select based on the ", toupper(criterion), " criterion when G=1"), call.=FALSE)
    if(criterion  == "dbs"    &&
       attr(x, "Algo") == "CEM") stop(paste0("Can't select based on the ", toupper(criterion), " criterion as the CEM algorithm was used to fit the model"), call.=FALSE)
    if(criterion  == "cv"     &&
       !attr(x, "CV"))           stop("Can't select based on the CV criterion as cross-validated likelihood wasn't performed", call.=FALSE)
    tmp           <- switch(EXPR=criterion, bic=x$BIC, icl=x$ICL, aic=x$AIC, cv=x$CV, nec=x$NEC, dbs=x$DBS, asw=x$ASW, loglik=x$LOGLIK)
    if(!noise)     {
      tmp         <- tmp[,colnames(tmp) %in% c("CC", "UC", "CU", "UU"), drop=FALSE]
    }
    if(!m.G)       {
      Gallow      <- as.numeric(rownames(tmp))
      if(!(G %in% Gallow))       stop("Invalid 'G'",       call.=FALSE)
      tmp         <- tmp[as.character(G),, drop=FALSE]
    }
    if(!m.M)       {
      Mallow      <- colnames(tmp)
      if(!(modtype %in% Mallow)) stop("Invalid 'modtype'", call.=FALSE)
      tmp         <- tmp[,modtype, drop=FALSE]
    }
    class(tmp)    <- "MEDcriterion"
    attr(tmp, "Criterion")  <- toupper(criterion)
    if(length(rank) > 1     ||
       !is.numeric(rank)    ||
       rank != floor(rank)  ||
       rank <= 0  ||
       rank  > sum(!is.na(tmp))) stop("Invalid 'rank'",    call.=FALSE)
    best          <- strsplit(names(.pick_MEDCrit(tmp, pick=rank)$crits[rank]), ",")[[1L]]
    modtype       <- best[1L]
    G             <- as.numeric(best[2L])
  }
  switch(EXPR=what, 
         DBS=, ASW=         {
    SILS          <- switch(EXPR=what, DBS=x$DBSvals, ASW=x$ASWvals)
    summ          <- attr(SILS, "Summ")
    if(!(G  %in%
       as.numeric(names(SILS)))) stop("Invalid 'G' value", call.=FALSE)
    S             <- SILS[[as.character(G)]]
    s.ind         <- if(noise) names(S) else names(S)[names(S) %in% c("CC", "UC", "CU", "UU")]
    if(!(modtype %in% s.ind))    stop("Invalid 'modtype'", call.=FALSE)
    res           <- S[[modtype]]
    if(anyNA(res))               message("Selected model didn't converge: no silhouettes available\n")
    attr(res, "Summ")      <- summ
  }, {
    ZS            <- x$ZS
    if(!(G  %in%
       as.numeric(names(ZS))))   stop("Invalid 'G' value", call.=FALSE)
    Z             <- ZS[[as.character(G)]]
    z.ind         <- if(noise) names(Z) else names(Z)[names(Z) %in% c("CC", "UC", "CU", "UU")]
    if(!(modtype %in% z.ind))    stop("Invalid 'modtype'", call.=FALSE)
    res           <- Z[[modtype]]
    if(anyNA(res))               message("Selected model didn't converge: no partition available\n")
    if(what == "MAP")       {
      MAP         <- max.col(res)
      res         <- if(noise) replace(MAP, MAP == G, 0L) else MAP
    }
  })
  attr(res, "G")           <- G
  attr(res, "ModelType")   <- modtype
  attr(res, "Noise")       <- is.element(modtype, c("CCN", "UCN", "CUN", "UUN"))
    return(res)
}

#' Choose the best MEDseq model
#'
#' Takes one or more sets of "\code{MEDseq}" models fitted by \code{\link{MEDseq_fit}} and ranks them according to a specified model selection criterion. It's possible to respect the internal ranking within each set of models, or to discard models within each set which were already deemed sub-optimal. This function can help with model selection via exhaustive or stepwise searches.
#' @param ... One or more objects of class \code{"MEDseq"} outputted by \code{\link{MEDseq_fit}}. All models must have been fit to the same data set. A single \emph{named} list of such objects can also be supplied. Additionally, objects of class \code{"MEDseqCompare"} outputted by this very function can also be supplied here.
#' 
#' This argument is only relevant for the \code{\link{MEDseq_compare}} function and will be ignored for the associated \code{print} function.
#' @param criterion The criterion used to determine the ranking. Defaults to \code{"dbs"}, the density-based silhouette.
#' @param pick The (integer) number of models to be ranked and compared. Defaults to \code{10L}. Will be constrained by the number of models within the \code{"MEDseq"} objects supplied via \code{...} if \code{optimal.only} is \code{FALSE}, otherwise constrained simply by the number of \code{"MEDseq"} objects supplied. Setting \code{pick=Inf} is a valid way to select all models.
#' @param optimal.only Logical indicating whether to only rank models already deemed optimal within each \code{"MEDeq"} object (\code{TRUE}), or to allow models which were deemed suboptimal enter the final ranking (\code{FALSE}, the default). See \code{details}.
#' @param x,index,digits, Arguments required for the associated \code{print} function:
#' \describe{
#' \item{\code{x}}{An object of class \code{"MEDseqCompare"} resulting from a call to \code{\link{MEDseq_compare}}.}
#' \item{\code{index}}{A logical or numeric vector giving the indices of the rows of the table of ranked models to print. This defaults to the full set of ranked models. It can be useful when the table of ranked models is large to examine a subset via this \code{index} argument, for display purposes.}
#' \item{\code{digits}}{The number of decimal places to round model selection criteria to (defaults to 3).}}
#' @note The \code{criterion} argument here need not comply with the criterion used for model selection within each \code{"MEDseq"} object, but be aware that a mismatch in terms of \code{criterion} \emph{may} require the optimal model to be re-fit in order to be extracted, thereby slowing down \code{\link{MEDseq_compare}}.
#' 
#' If random starts had been used via \code{init.z="random"} the \code{optimal} model may not necessarily correspond to the highest-ranking model in the presence of a criterion mismatch, due to the randomness of the initialisation. 
#'
#' A dedicated \code{print} function exists for objects of class \code{"MEDseqCompare"} and \code{\link{plot.MEDseq}} can also be called on objects of class \code{"MEDseqCompare"}.
#' @return A list of class \code{"MEDseqCompare"}, for which a dedicated print function exists, containing the following elements (each of length \code{pick}, and ranked according to \code{criterion}, where appropriate):
#' \item{\code{data}}{The name of the data set to which the models were fitted.}
#' \item{\code{optimal}}{The single optimal model (an object of class \code{"MEDseq"}) among those supplied, according to the chosen \code{criterion}.}
#' \item{\code{pick}}{The final number of ranked models. May be different (i.e. less than) the supplied \code{pick} value.}
#' \item{\code{MEDNames}}{The names of the supplied \code{"MEDseq"} objects.}
#' \item{\code{modelNames}}{The MEDseq model names (denoting the constraints or lack thereof on the precision parameters).}
#' \item{\code{G}}{The optimal numbers of components.}
#' \item{\code{df}}{The numbers of estimated parameters.}
#' \item{\code{iters}}{The numbers of EM/CEM iterations.}
#' \item{\code{bic}}{BIC values, ranked according to \code{criterion}.}
#' \item{\code{icl}}{TCL values, ranked according to \code{criterion}.}
#' \item{\code{aic}}{AIC values, ranked according to \code{criterion}.}
#' \item{\code{cv}}{Cross-validated log-likelihood values, ranked according to \code{criterion}.}
#' \item{\code{nec}}{NEC values, ranked according to \code{criterion}.}
#' \item{\code{dbs}}{(Weighted) mean/median DBS values, ranked according to \code{criterion}.}
#' \item{\code{asw}}{(Weighted) mean/median ASW values, ranked according to \code{criterion}.}
#' \item{\code{loglik}}{Maximal log-likelihood values, ranked according to \code{criterion}.}
#' \item{\code{gating}}{The gating formulas.}
#' \item{\code{algo}}{The algorithm used for fitting the model - either \code{"EM"}, \code{"CEM"}, \code{"cemEM"}.}
#' \item{\code{equalPro}}{Logical indicating whether mixing proportions were constrained to be equal across components.}
#' \item{\code{weights}}{Logical indicating whether the given model was fitted with sampling weights.}
#' \item{\code{noise}}{Logical indicating the presence/absence of a noise component. Only displayed if at least one of the compared models has a noise component.}
#' \item{\code{noise.gate}}{Logical indicating whether gating covariates were allowed to influence the noise component's mixing proportion. Only printed for models with a noise component, when at least one of the compared models has gating covariates.}
#' \item{\code{equalNoise}}{Logical indicating whether the mixing proportion of the noise component for \code{equalPro} models is also equal (\code{TRUE}) or estimated (\code{FALSE}).}
#' @details The purpose of this function is to conduct model selection on \code{"MEDseq"} objects, fit to the same data set, with different combinations of gating network covariates or different initialisation settings.
#'
#' Model selection will have already been performed in terms of choosing the optimal number of components and MEDseq model type within each supplied set of results, but \code{\link{MEDseq_compare}} will respect the internal ranking of models when producing the final ranking if \code{optimal.only} is \code{FALSE}: otherwise only those models already deemed optimal within each \code{"MEDseq"} object will be ranked.
#'
#' As such if two sets of results are supplied when \code{optimal.only} is \code{FALSE}, the 1st, 2nd and 3rd best models could all belong to the first set of results, meaning a model deemed suboptimal according to one set of covariates could be superior to one deemed optimal under another set of covariates.
#' @export
#' @keywords clustering main
#' @importFrom TraMineR "seqdef"
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @references Keefe Murphy, T. Brendan Murphy, Raffaella Piccarreta, and I. Claire Gormley (2019). Clustering longitudinal life-course sequences using mixtures of exponential-distance models. \emph{To appear}. <\href{https://arxiv.org/abs/1908.07963}{arXiv:1908.07963}>.
#' @seealso \code{\link{MEDseq_fit}}, \code{\link{plot.MEDseq}}
#' @usage
#' MEDseq_compare(...,
#'                criterion = c("dbs", "asw", "bic", 
#'                              "icl", "aic", "cv", "nec"),
#'                pick = 10L,
#'                optimal.only = FALSE)
#' @examples
#' \dontshow{library(TraMineR)}
#' data(biofam)
#' seqs <- seqdef(biofam[10:25] + 1L)
#' covs <- biofam[2:3]
#' \donttest{ 
#' # Fit a range of models
#' m1   <- MEDseq_fit(seqs, G=9:10)
#' m2   <- MEDseq_fit(seqs, G=9:10, gating=~sex, covars=covs)
#' m3   <- MEDseq_fit(seqs, G=9:10, gating=~birthyr, covars=covs)
#' m4   <- MEDseq_fit(seqs, G=9:10, gating=~sex + birthyr, covars=covs)
#' 
#' # Rank only the optimal models and examine the best model
#' (comp <- MEDseq_compare(m1, m2, m3, m4, optimal.only=TRUE))
#' (best <- comp$optimal)
#' (summ <- summary(best))
#' 
#' # Examine all models visited, including those already deemed suboptimal
#' # Only print models with gating covariates & 10 components
#' comp2 <- MEDseq_compare(m1, m2, m3, m4, pick=Inf)
#' print(comp2, comp2$gating != "None" & comp2$G == 10)}
MEDseq_compare    <- function(..., criterion = c("dbs", "asw", "bic", "icl", "aic", "cv", "nec"), pick = 10L, optimal.only = FALSE) {
  crit.miss       <- missing(criterion)
  if(!missing(criterion)   && (length(criterion) > 1 ||
     !is.character(criterion)))  stop("'criterion' must be a single character string", call.=FALSE)
  criterion       <- match.arg(criterion)
  num.miss        <- missing(pick)
  opt.miss        <- missing(optimal.only)
  if(length(pick) != 1     ||
     !is.numeric(pick))          stop("'pick' must be a single number", call.=FALSE)
  if(floor(pick)  != pick  ||
     pick          < 1)          stop("'pick' must be a strictly positive integer", call.=FALSE)
  if(length(optimal.only)   > 1 ||
     !is.logical(optimal.only))  stop("'optimal.only' must be a single logical indicator", call.=FALSE)
  call            <- match.call(expand.dots=TRUE)[-1L]
  call            <- if(crit.miss) call else call[-which(names(call) == "criterion")]
  call            <- if(num.miss)  call else call[-which(names(call) == "pick")]
  call            <- if(opt.miss)  call else call[-which(names(call) == "optimal.only")]
  len.call        <- length(as.list(call))
  if(len.call     == 1     && inherits(..., "list") && !inherits(..., "MEDseq")) {
    dots          <- as.list(...)
    mod.names     <- unique(names(dots))
    comparison    <- sapply(dots, inherits, "MEDseqCompare", logical(1L))
    dat.name      <- if(any(comparison)) dots[[1L]]$data
    dots[comparison]       <- sapply(dots[comparison], "[", "optimal")
    MEDs          <- dots[mod.names]
    if(is.null(mod.names))       stop("When supplying models as a list, every element of the list must be named", call.=FALSE)
  } else           {
    dots          <- list(...)
    mod.names     <- vapply(call, deparse, character(1L))
    comparison    <- sapply(dots, inherits, "MEDseqCompare", logical(1L))
    dat.name      <- if(any(comparison)) dots[[1L]]$data
    dots[comparison]       <- sapply(dots[comparison], "[", "optimal")
    MEDs          <- stats::setNames(dots, mod.names)
    mod.names     <- unique(mod.names)
    MEDs          <- MEDs[mod.names]
  }
  Mclass          <- vapply(MEDs, class,          character(1L))
  if(any(Mclass   != "MEDseq"))  stop("All models must be of class 'MEDseq'!", call.=FALSE)
  data            <- lapply(MEDs, "[[", "data")
  data            <- lapply(data, unname)
  if(length(data)  > 1     && 
     !.unique_list(data))        stop("All models being compared must have been fit to the same data set!", call.=FALSE)
  dat.name        <- if(is.null(dat.name)) deparse(MEDs[[1L]]$call$seqs) else dat.name
  gate.x          <- lapply(MEDs,   "[[", "gating")
  algo            <- sapply(MEDs,   attr, "Algo")
  equalNoise      <- sapply(MEDs,   attr, "EqualNoise")
  equalPro        <- sapply(MEDs,   attr, "EqualPro")
  noise.gate      <- sapply(MEDs,   attr, "NoiseGate")
  weights         <- sapply(MEDs,   attr, "Weighted")
  gating          <- lapply(gate.x, attr, "Formula")
  BICs            <- lapply(MEDs, "[[", "BIC")
  ICLs            <- lapply(MEDs, "[[", "ICL")
  AICs            <- lapply(MEDs, "[[", "AIC")
  LLxs            <- lapply(MEDs, "[[", "LOGLIK")
  DFxs            <- lapply(MEDs, "[[", "DF")
  ITxs            <- lapply(MEDs, "[[", "ITERS")
  CVs             <- lapply(MEDs, "[[", "CV")
  NECs            <- lapply(MEDs, "[[", "NEC")
  DBSs            <- lapply(MEDs, "[[", "DBS")
  ASWs            <- lapply(MEDs, "[[", "ASW")
  cvnull          <- vapply(CVs,  is.null, logical(1L))
  necnull         <- vapply(NECs, is.null, logical(1L))
  dbsnull         <- vapply(DBSs, is.null, logical(1L))
  aswnull         <- vapply(ASWs, is.null, logical(1L))
  if(all(cvnull)  && 
     criterion    == "cv")       stop("'criterion' cannot be 'cv' when cross-validation was not performed for any of the supplied models", call.=FALSE)
  if(all(necnull) &&             
     criterion    == "nec")      stop("'criterion' cannot be 'nec' when all models being compared contain only 1 component", call.=FALSE)
  if(all(dbsnull) &&             
     criterion    == "dbs")      stop("'criterion' cannot be 'dbs' when all models being compared contain only 1 component", call.=FALSE)
  if(all(aswnull) &&             
     criterion    == "asw")      stop("'criterion' cannot be 'asw' when all models being compared contain only 1 component", call.=FALSE)
  choice          <- max(lengths(BICs))
  bics            <- lapply(BICs, function(x) .pick_MEDCrit(x, choice)$crits)
  icls            <- lapply(ICLs, function(x) .pick_MEDCrit(x, choice)$crits)
  aics            <- lapply(AICs, function(x) .pick_MEDCrit(x, choice)$crits)
  llxs            <- lapply(LLxs, function(x) .pick_MEDCrit(x, choice)$crits)
  dfxs            <- lapply(DFxs, function(x) .pick_MEDCrit(x, choice)$crits)
  itxs            <- lapply(ITxs, function(x) .pick_MEDCrit(x, choice)$crits)
  cvs             <- lapply(CVs,  function(x) if(!is.null(x)) .pick_MEDCrit(x, choice)$crits)[!cvnull]
  necs            <- lapply(NECs, function(x) if(!is.null(x)) .pick_MEDCrit(x, choice)$crits)[!necnull]
  dbss            <- lapply(DBSs, function(x) if(!is.null(x)) .pick_MEDCrit(x, choice)$crits)[!dbsnull]
  asws            <- lapply(ASWs, function(x) if(!is.null(x)) .pick_MEDCrit(x, choice)$crits)[!aswnull]
  if(optimal.only) {
    opt.names     <- names(.crits_names(lapply(switch(EXPR=criterion, bic=bics, icl=icls, aic=aics, cv=cvs, nec=necs, dbs=dbss, asw=asws), "[", 1L)))
  }
  bics            <- .crits_names(bics)
  icls            <- .crits_names(icls)
  aics            <- .crits_names(aics)
  llxs            <- .crits_names(llxs)
  dfxs            <- .crits_names(dfxs)
  itxs            <- .crits_names(itxs)
  cvs             <- .crits_names(cvs)
  necs            <- .crits_names(necs)
  dbss            <- .crits_names(dbss)
  asws            <- .crits_names(asws)
  if(criterion    == "cv"  &&
    (length(cvs)  != 
     length(bics)))              warning("Discarding models for which the CV criterion was not computed\n",  call.=FALSE, immediate.=TRUE)
  if(criterion    == "nec" &&
    (length(necs) != 
     length(bics)))              warning("Discarding models for which the NEC criterion was not computed\n", call.=FALSE, immediate.=TRUE)
  if(criterion    == "dbs" &&
    (length(dbss) != 
     length(bics)))              warning("Discarding models for which the DBS criterion was not computed\n", call.=FALSE, immediate.=TRUE)
  if(criterion    == "asw" &&
    (length(asws) != 
     length(bics)))              warning("Discarding models for which the ASW criterion was not computed\n", call.=FALSE, immediate.=TRUE)
  if(optimal.only) {
    bics          <- bics[names(bics) %in% opt.names]
    icls          <- icls[names(icls) %in% opt.names]
    aics          <- aics[names(aics) %in% opt.names]
    llxs          <- llxs[names(llxs) %in% opt.names]
    dfxs          <- dfxs[names(dfxs) %in% opt.names]
    itxs          <- itxs[names(itxs) %in% opt.names]
    cvs           <- cvs[names(cvs)   %in% opt.names]
    necs          <- necs[names(necs) %in% opt.names]
    dbss          <- dbss[names(dbss) %in% opt.names]
    asws          <- asws[names(asws) %in% opt.names]
  }
  crits           <- switch(EXPR=criterion, bic=bics, icl=icls, aic=aics, cv=cvs, nec=necs, dbs=dbss, asw=asws)
  pick            <- min(pick, length(crits))
  max.crits       <- sort(crits, decreasing=criterion != "nec")[seq_len(pick)]
  if(length(unique(max.crits))  < pick) {
    ties          <- max.crits == max.crits[1L]
    if(any(ties[-1L]))      {     warning(paste0("Ties for the optimal model exist according to the '", criterion, "' criterion: choosing the most parsimonious model\n"), call.=FALSE, immediate.=TRUE)
      df.ties     <- dfxs[names(max.crits)][which(ties)]
      max.crits[ties]      <- max.crits[order(df.ties)]
      if(any((df.ties      == df.ties[1L])[-1L])) {
        max.crits[ties]    <- max.crits[order(as.numeric(gsub(".*,", "", names(max.crits[ties]))))]
      }
    } else                       warning(paste0("Ties exist according to the '", criterion, "' criterion\n"), call.=FALSE, immediate.=TRUE)
  }
  max.names       <- names(max.crits)
  crit.names      <- gsub("\\|.*", "",          max.names)
  G               <- as.numeric(gsub(".*,", "", max.names))
  gating          <- unname(unlist(gating[crit.names]))
  modelNames      <- gsub(",.*", "", gsub(".*\\|", "", max.names))
  best.model      <- MEDs[[crit.names[1L]]]
  if(best.model$modtype != modelNames[1L] || best.model$G != G[1L]) {
    message("Re-fitting optimal model due to mismatched 'criterion'...\n\n")
    old.call    <- best.model$call
    old.call    <- c(as.list(old.call)[1L], list(criterion=criterion), as.list(old.call)[-1L])
    old.call    <- as.call(old.call[!duplicated(names(old.call))])
    if(!is.null(old.call$init.z)    &&
       old.call$init.z  == 
       "random")                 warning("Optimal model may differ slightly due to criterion mismatch and random starts used in the initialisation:\nPrinted output intended only as a guide", call.=FALSE, immediate.=TRUE)
    best.call   <- c(list(data=best.model$data, modtype=modelNames[1L], G=G[1L], criterion="bic", verbose=FALSE, do.cv=FALSE, do.nec=FALSE), as.list(old.call[-1L]))
    best.mod    <- try(do.call(MEDseq_fit, best.call[!duplicated(names(best.call))]), silent=TRUE)
    if(!inherits(best.model, "try-error")) {
      best.model$call               <- old.call
      best.model$modtype            <- best.mod$modtype
      best.model$G                  <- best.mod$G
      best.model$bic                <- best.mod$bic
      best.model$icl                <- best.mod$icl
      best.model$aic                <- best.mod$aic
      best.model$cv                 <- if(attr(best.model, "CV"))  best.model$CV[best.mod$G,best.mod$modtype]
      best.model$nec                <- if(attr(best.model, "NEC")) best.model$NEC[which(best.mod$G == as.numeric(rownames(best.model$NEC))),best.mod$modtype]
      best.model$dbs                <- if(attr(best.model, "DBS")) best.model$DBS[which(best.mod$G == as.numeric(rownames(best.model$DBS))),best.mod$modtype]
      best.model$asw                <- if(attr(best.model, "ASW")) best.model$ASW[which(best.mod$G == as.numeric(rownames(best.model$ASW))),best.mod$modtype]
      best.model$gating             <- best.mod$gating
      best.model$loglik             <- best.mod$loglik
      best.model$df                 <- best.mod$df
      best.model$iters              <- best.mod$iters
      best.model$params             <- best.mod$params
      best.model$z                  <- best.mod$z
      best.model$MAP                <- best.mod$MAP
      best.model$uncert             <- best.mod$uncert
      attributes(best.model)        <- attributes(best.mod)
    } else best.model               <- paste0("Failed to re-fit the optimal model: ", gsub("\"", "'", deparse(old.call, width.cutoff=500L), fixed=TRUE))
  }
  gating[gating == "~1" | G   == 1] <- "None"
  noise         <- modelNames %in% c("CCN", "UCN", "CUN", "UUN")
  noise.gate    <- ifelse(!noise, NA, noise.gate[crit.names])
  equalPro      <- replace(unname(equalPro[crit.names]), gating != "None" | G == 1, NA)
  equalNoise    <- ifelse(!noise | G == 1, NA, equalNoise[crit.names] & vapply(equalPro, isTRUE, logical(1L)))
  comp          <- list(data = dat.name, optimal = best.model, pick = pick, MEDNames = crit.names, modelNames = modelNames, G = as.integer(G), df = as.integer(unname(dfxs[max.names])), 
                        iters = as.integer(unname(itxs[max.names])), bic = unname(bics[max.names]), icl = unname(icls[max.names]), aic = unname(aics[max.names]), cv = unname(cvs[max.names]), 
                        nec = replace(unname(necs[max.names]), G == 1, NA), dbs = replace(unname(dbss[max.names]), G == 1, NA), asw = replace(unname(asws[max.names]), G == 1, NA), 
                        loglik = unname(llxs[max.names]), gating = gating, algo = unname(algo[crit.names]), weights = unname(weights[crit.names]), equalPro = equalPro, noise = unname(noise), 
                        noise.gate = unname(replace(noise.gate, gating == "None" | G <= 2, NA)), equalNoise = unname(replace(equalNoise, !equalPro | is.na(equalPro), NA)))
  class(comp)   <- c("MEDseqCompare", "MEDseq")
  bic.tmp       <- sapply(BICs, as.vector)
  attr(comp, "Crit")   <- criterion
  attr(comp, "Opt")    <- optimal.only
  attr(comp, "NMods")  <- c(tried = sum(vapply(bic.tmp, function(x) length(x[!is.na(x)]),    numeric(1L))),
                            ran   = sum(vapply(bic.tmp, function(x) length(x[is.finite(x)]), numeric(1L))))
    comp
}

#' Set control values for use with MEDseq_fit
#'
#' Supplies a list of arguments (with defaults) for use with \code{\link{MEDseq_fit}}.
#' @param algo Switch controlling whether models are fit using the \code{"EM"} (the default) or \code{"CEM"} algorithm. The option \code{"cemEM"} allows running the EM algorithm starting from convergence of the CEM algorithm.
#' @param init.z The method used to initialise the cluster labels. Defaults to "\code{kmedoids}". Other options include Ward hierarchical clustering ("\code{hc}"), "\code{random}" initialisation, and a user-supplied "\code{list}".
#' @param z.list A user supplied list of initial cluster allocation matrices, with number of rows given by the number of observations, and numbers of columns given by the range of component numbers being considered. Only relevant if \code{init.z == "z.list"}. These matrices are allowed correspond to both soft or hard clusterings, and will be internally normalised so that the rows sum to 1.
#' @param dist.mat An optional distance matrix to use for initialisation when \code{init.z} is one of "\code{kmedoids}" or "\code{hc}". Defaults to a Hamming distance matrix. This is an experimental feature and should only be tampered with by expert users.
#' @param unique A logical indicating whether the model is fit only to the unique observations (defaults to \code{TRUE}). When there are covariates, this means all unique combinations of covariate and sequence patterns, otherwise only the sequence patterns. When \code{weights} are supplied to \code{\link{MEDseq_fit}}, the weights are multiplied by the occurrence frequency of the corresponding sequence, otherwise weights are given by these occurrence frequencies. Thereafter, the weighted model is fit only to the unique observations if \code{TRUE}. Thus, significant computational gains can be made.
#' @param criterion When either \code{G} or \code{modtype} is a vector, \code{criterion} governs how the 'best' model is determined when gathering output. Note that all criteria will be returned in any case, if possible.
#' @param tau0 Prior mixing proportion for the noise component. If supplied, a noise component will be added to the model in the estimation, with \code{tau0} giving the prior probability of belonging to the noise component for \emph{all} observations. Typically supplied as a scalar in the interval (0, 1), e.g. \code{0.1}. Can be supplied as a vector when gating covariates are present and \code{noise.gate} is \code{TRUE}.
#' @param noise.gate A logical indicating whether gating network covariates influence the mixing proportion for the noise component, if any. Defaults to \code{TRUE}, but leads to greater parsimony if \code{FALSE}. Only relevant in the presence of a noise component; only effects estimation in the presence of gating covariates.
#' @param do.nec A logical indicating whether the normalised entropy criterion (NEC) should also be computed (for models with more than one component). Defaults to \code{FALSE}. When \code{TRUE}, models with \code{G=1} are fitted always.
#' @param do.cv A logical indicating whether cross-validated log-likelihood scores should also be computed (see \code{nfolds}). Defaults to \code{FALSE} due to significant computational burden incurred.
#' @param nfolds The number of folds to use when \code{isTRUE{do.cv}}.
#' @param nstarts The number of random initialisations to use when \code{init.z="random"}. Defaults to \code{1}. Results will be based on the random start yielding the highest estimated log-likelihood.
#' @param stopping The criterion used to assess convergence of the EM/CEM algorithm. The default (\code{"aitken"}) uses Aitken's acceleration method, otherwise the \code{"relative"} change in log-likelihood is monitored (which may be less strict).
#' @param equalPro Logical variable indicating whether or not the mixing proportions are to be constrained to be equal in the model. Default: \code{equalPro = FALSE}. Only relevant when \code{gating} covariates are \emph{not} supplied within \code{\link{MEDseq_fit}}, otherwise ignored. In the presence of a noise component, only the mixing proportions for the non-noise components are constrained to be equal (by default, see \code{equalNoise}), after accounting for the noise component.
#' @param equalNoise Logical which is only invoked when \code{isTRUE(equalPro)} and gating covariates are not supplied. Under the default setting (\code{FALSE}), the mixing proportion for the noise component is estimated, and remaining mixing proportions are equal; when \code{TRUE} all components, including the noise component, have equal mixing proportions.
#' @param tol A vector of length two giving relative convergence tolerances for 1) the log-likelihood of the EM/CEM algorithm, and 2) optimisation in the multinomial logistic regression in the gating network, respectively. The default is \code{c(1e-05, 1e-08)}. If only one number is supplied, it is used as the tolerance in both cases.
#' @param itmax A vector of length two giving integer limits on the number of iterations for 1) the EM/CEM algorithm, and 2) the multinomial logistic regression in the gating network, respectively. The default is \code{c(.Machine$integer.max, 100)}.
#' @param opti Charactering string indicating how central sequence parameters should be estimated. The default "\code{mode}" is exact and thus this experimental argument should only be tampered with by expert users. The option "\code{medoid}" fixes the central sequence(s) to be one of the observed sequences (like k-medoids). The other options \code{"first"} and \code{"GA"} use the first-improvement and genetic algorithms, respectively, to mutate the medoid. Pre-computation of the Hamming distance matrix for the observed sequences speeds-up computation of all options other than \code{"mode"}.
#' @param ordering Experimental feature that should only be tampered with by experienced users. Allows sequences to be reordered on the basis of the column-wise entropy when \code{opti} is "\code{first}" or "\code{GA}".
#' @param MaxNWts The maximum allowable number of weights in the call to \code{\link[nnet]{multinom}} for the multinomial logistic regression in the gating network. There is no instrinsic limit in the code, but increasing \code{MaxNWts} will probably allow fits that are very slow and time-consuming. It may be necessary to increase \code{MaxNWts} when categorical concomitant variables with many levels are included or the number of components is high.
#' @param verbose Logical indicating whether to print messages pertaining to progress to the screen during fitting. By default is \code{TRUE} if the session is interactive, and \code{FALSE} otherwise. If \code{FALSE}, warnings and error messages will still be printed to the screen, but everything else will be suppressed.
#' @param ... Catches unused arguments, and also allows the optional arguments \code{ztol} and \code{summ} to be passed to \code{\link{dbs}} (\code{ztol} and \code{summ}) and the ASW computation (\code{summ}).
#'
#' @return A named list in which the names are the names of the arguments and the values are the values supplied to the arguments.
#' @details \code{\link{MEDseq_control}} is provided for assigning values and defaults within \code{\link{MEDseq_fit}}. While the \code{criterion} argument controls the choice of the optimal number of components and MEDseq model type (in terms of the constraints or lack thereof on the precision parameters), \code{\link{MEDseq_compare}} is provided for choosing between fits with different combinations of covariates or different initialisation settings.
#' @importFrom nnet "multinom"
#' @keywords control
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @seealso \code{\link{MEDseq_fit}}, \code{\link{dbs}}, \code{\link[WeightedCluster]{wcKMedoids}}, \code{\link[cluster]{pam}}, \code{\link[cluster]{agnes}}, \code{\link[stats]{hclust}}, \code{\link[TraMineR]{seqdist}}, \code{\link[nnet]{multinom}}, \code{\link{MEDseq_compare}}
#' @references Keefe Murphy, T. Brendan Murphy, Raffaella Piccarreta, and I. Claire Gormley (2019). Clustering longitudinal life-course sequences using mixtures of exponential-distance models. \emph{To appear}. <\href{https://arxiv.org/abs/1908.07963}{arXiv:1908.07963}>.
#' 
#' Menardi, G. (2011). Density-based Silhouette diagnostics for clustering methods. \emph{Statistics and Computing} 21(3): 295-308.
#' @export
#' @usage 
#' MEDseq_control(algo = c("EM", "CEM", "cemEM"), 
#'                init.z = c("kmedoids", "hc", "random", "list"), 
#'                z.list = NULL, 
#'                dist.mat = NULL, 
#'                unique = TRUE, 
#'                criterion = c("dbs", "asw", "bic", "icl", "aic", "cv", "nec"), 
#'                tau0 = NULL, 
#'                noise.gate = TRUE, 
#'                do.nec = FALSE, 
#'                do.cv = FALSE, 
#'                nfolds = 10L, 
#'                nstarts = 1L, 
#'                stopping = c("aitken", "relative"), 
#'                equalPro = FALSE, 
#'                equalNoise = FALSE, 
#'                tol = c(1E-05, 1E-08), 
#'                itmax = c(.Machine$integer.max, 100L), 
#'                opti = c("mode", "medoid", "first", "GA"), 
#'                ordering = c("none", "decreasing", "increasing"), 
#'                MaxNWts = 1000L, 
#'                verbose = TRUE, 
#'                ...)
#' @examples
#' data(mvad)
#' 
#' # The CC MEDseq model is equivalent to k-medoids when the CEM 
#' # algorithm is employed, mixing proportions are constrained,
#' # and the central sequences are restricted to the observed sequences
#' ctrl   <- MEDseq_control(algo="CEM", equalPro=TRUE, opti="medoid", criterion="asw")
#' \donttest{
#' # Note that ctrl must be explicitly named 'ctrl'
#' mod   <- MEDseq_fit(seqdef(mvad[,15:86]), G=8, modtype="CC", weights=mvad$weight, ctrl=ctrl)
#' 
#' # Alternatively, specify the control arguments directly
#' mod   <- MEDseq_fit(seqdef(mvad[,15:86]), G=8, modtype="CC", weights=mvad$weight,
#'                     algo="CEM", equalPro=TRUE, opti="medoid", criterion="asw")
#' 
#' # Note that supplying control arguments via a mix of the ... construct and the named argument 
#' # 'control' or supplying MEDseq_control output without naming it 'control' can throw an error}
MEDseq_control    <- function(algo = c("EM", "CEM", "cemEM"), init.z = c("kmedoids", "hc", "random", "list"), z.list = NULL, dist.mat = NULL, unique = TRUE, 
                              criterion = c("dbs", "asw", "bic", "icl", "aic", "cv", "nec"), tau0 = NULL, noise.gate = TRUE, do.nec = FALSE, do.cv = FALSE, nfolds = 10L, 
                              nstarts = 1L, stopping = c("aitken", "relative"), equalPro = FALSE, equalNoise = FALSE, tol = c(1E-05, 1E-08), itmax = c(.Machine$integer.max, 100L), 
                              opti = c("mode", "medoid", "first", "GA"), ordering = c("none", "decreasing", "increasing"), MaxNWts = 1000L, verbose = TRUE, ...) {
  miss.args                <- list(tau0=missing(tau0), init.z = missing(init.z), z.list = missing(z.list))
  if(!missing(algo)        &&
    (length(algo)      > 1 ||
     !is.character(algo)))       stop("'algo' must be a character vector of length 1",      call.=FALSE)
  if(!missing(init.z)      &&
    (length(init.z)    > 1 ||
     !is.character(init.z)))     stop("'init.z' must be a character vector of length 1",    call.=FALSE)
  init.z                   <- match.arg(init.z)
  if(!missing(dist.mat))    {
    dist.mat               <- tryCatch(suppressWarnings(stats::as.dist(dist.mat)), error=function(e)     {
                                 stop("'dist.mat' must be coercible to the class 'dist'",   call.=FALSE) })
  }
  if(length(unique)    > 1 ||
     !is.logical(unique))        stop("'unique' must be a single logical indicator",        call.=FALSE)
  if(init.z == "random")    {
   if(length(nstarts) != 1 ||
      !is.numeric(nstarts) ||
      (nstarts         < 1 ||
       floor(nstarts) !=
       nstarts))                 stop(paste0("'nstarts' must be a single integer >= 1 if when 'init.z'=", init.z), call.=FALSE)
  }
  if(!missing(criterion)   &&
    (length(criterion) > 1 ||
     !is.character(criterion)))  stop("'criterion' must be a character vector of length 1", call.=FALSE)
  if(length(do.cv)     > 1 ||
     !is.logical(do.cv))         stop("'do.cv' must be a single logical indicator",         call.=FALSE)
  if(length(do.nec)    > 1 ||
     !is.logical(do.nec))        stop("'do.nec' must be a single logical indicator",        call.=FALSE)
  if(!missing(stopping)    &&
    (length(stopping)  > 1 ||
     !is.character(stopping)))   stop("'stopping' must be a character vector of length 1",  call.=FALSE)
  if(!miss.args$tau0       &&
    (!is.numeric(tau0)     ||
     any(tau0  < 0)        || 
     any(tau0 >= 1)))            stop("'tau0' must lie in the interval [0, 1)",             call.=FALSE)
  if(!missing(opti)        &&
    (length(opti)      > 1 ||
     !is.character(opti)))       stop("'opti' must be a character vector of length 1",      call.=FALSE)
  if(!missing(ordering)    &&
    (length(ordering)  > 1 ||
     !is.character(ordering)))   stop("'ordering' must be a character vector of length 1",  call.=FALSE)
  if(length(noise.gate)     > 1 ||
     !is.logical(noise.gate))    stop("'noise.gate' must be a single logical indicator",    call.=FALSE)
  if(length(MaxNWts)   > 1 ||
     !is.numeric(MaxNWts)  ||
     MaxNWts      <= 0)          stop("'MaxNWts' must a strictly positive scalar",          call.=FALSE)
  if(length(equalPro)  > 1 ||
     !is.logical(equalPro))      stop("'equalPro' must be a single logical indicator",      call.=FALSE)
  if(length(equalNoise)     > 1 ||
     !is.logical(equalNoise))    stop("'equalNoise' must be a single logical indicator",    call.=FALSE)
  if((len.tol     <- 
      length(tol)) > 2     ||
     !is.numeric(tol))           stop("'tol' must be a numeric vector of length at most 2", call.=FALSE)
  if(any(tol   < 0,
         tol  >= 1))             stop("'tol' must be in the interval [0, 1)",               call.=FALSE)
  if(len.tol  == 1)    tol <- rep(tol, 2L)
  if(length(itmax)         == 1) {
    itmax     <- c(itmax, 100L)
  } else if(length(itmax)  != 2) stop("'itmax' must be of length 2",                        call.=FALSE)
  if(!is.numeric(itmax)    ||
     any(floor(itmax) != itmax) ||
     any(itmax    <= 0))         stop("'itmax' must contain strictly positive integers",    call.=FALSE)
  inf         <- is.infinite(itmax)
  if(any(inf))   itmax[inf]     <- .Machine$integer.max
  itmax[1L]   <- ifelse(itmax[1L] == .Machine$integer.max, itmax[1L], itmax[1L] + 2L)
  if(length(verbose)   > 1 ||
     !is.logical(verbose))       stop("'verbose' must be a single logical indicator",       call.=FALSE)
  control                  <- list(algo = match.arg(algo), init.z = init.z, dist.mat = dist.mat, nstarts = nstarts, criterion = match.arg(criterion), nfolds = nfolds, do.cv = do.cv, do.nec = do.nec, 
                                   MaxNWts = MaxNWts, stopping = match.arg(stopping), tau0 = tau0, opti = match.arg(opti), ordering = match.arg(ordering), noise.gate = noise.gate, unique = unique,
                                   equalPro = equalPro, equalNoise = equalNoise, tol = tol[1L], g.tol = tol[2L], itmax = itmax[1L], g.itmax = itmax[2L], verbose = verbose, z.list = z.list)
  attr(control, "missing") <- miss.args
    return(control)
}

#' MEDseq: Mixtures of Exponential-Distance Models with Covariates
#'
#' Fits MEDseq models: mixtures of Exponential-Distance models with gating covariates and sampling weights. Typically used for clustering categorical/longitudinal life-course sequences. Additional arguments are available via the function \code{\link{MEDseq_control}}.
#' @param seqs A state-sequence object of class "\code{stslist}" as created by the \code{\link[TraMineR]{seqdef}} function in the \pkg{TraMineR} package.
#' @param G A positive integer vector specifying the numbers of mixture components (clusters) to fit. Defaults to \code{G=1:9}.
#' @param modtype A vector of character strings indicating the type of MEDseq models to be fitted, in terms of the constraints or lack thereof on the precision parameters. By default, all valid model types are fitted (except some only where \code{G > 1} or \code{G > 2}, see \code{note}). 
#' The models are named "\code{CC}", "\code{CU}", "\code{UC}", "\code{UU}", \code{CCN}", "\code{CUN}", "\code{UCN}", and "\code{UUN}". The first letter denotes whether the precision parameters are constrained/unconstrained across clusters. The second letter denotes whether the precision parameters are constrained/unconstrained across sequence positions (i.e. time points). The third letter denotes whether one of the components is constrained to have zero-precision/infinite variance. Such a noise component assumes sequences in that cluster follow a uniform distribution.
#' @param gating A \code{\link[stats]{formula}} for determining the model matrix for the multinomial logistic regression in the gating network when fixed covariates enter the mixing proportions. Defaults to \code{~1}, i.e. no covariates. This will be ignored where \code{G=1}. Continuous, categorical, and/or ordinal covariates are allowed. Logical covariates will be coerced to factors. Interactions, transformations, and higher order terms are permitted: the latter \strong{must} be specified explicitly using the \code{AsIs} operator (\code{\link{I}}). The specification of the LHS of the formula is ignored. Intercept terms are included by default.
#' @param covars An optional data frame (or a matrix with named columns) in which to look for the covariates in the \code{gating} network formula, if any. If not found in \code{covars}, any supplied \code{gating} covariates are taken from the environment from which \code{MEDseq_fit} is called. Try to ensure the names of variables in \code{covars} do not match any of those in \code{seqs}.
#' @param weights Optional numeric vector containing observation-specific sampling weights, which are accounted for in the model fitting and other functions where applicable. See the \code{unique} argument to \code{\link{MEDseq_control}} to see how incorporating weights also yields computational benefits.
#' @param ctrl A list of control parameters for the EM/CEM and other aspects of the algorithm. The defaults are set by a call to \code{\link{MEDseq_control}}.
#' @param ... Catches unused arguments (see \code{\link{MEDseq_control}}).
#' @param x,object,digits Arguments required for the \code{print} and \code{summary} functions: \code{x} and \code{object} are objects of class \code{"MEDseq"} resulting from a call to \code{\link{MEDseq_fit}}, while \code{digits} gives the number of decimal places to round to for printing purposes (defaults to 2).
#'
#' @return A list (of class \code{"MEDseq"}) with the following named entries (of which some may be missing, depending on the \code{criterion} employed), mostly corresponding to the chosen optimal model (as determined by the \code{criterion} within \code{\link{MEDseq_control}}):
#' \item{\code{call}}{The matched call.}
#' \item{\code{data}}{The input data, \code{seqs}.}
#' \item{\code{modtype}}{A character string denoting the MEDseq model type at which the optimal \code{criterion} occurs.}
#' \item{\code{G}}{The optimal number of mixture components according to \code{criterion}.}
#' \item{\code{params}}{A list with the following named components:
#' \describe{
#' \item{\code{theta}}{A matrix with \code{G} rows and P columns, where P is the number of sequence positions, giving the central sequences of each cluster. The mean of the noise component is not reported, as it does not contribute in any way to the likelihood.}
#' \item{\code{lambda}}{A matrix of precision parameters. Will contain \code{1} row if the 1st letter of \code{modtype} is "C" and \code{G} columns otherwise. Will contain \code{1} column if the 2nd letter of \code{modtype} is "C" and P columns otherwise, where P is the number of sequence positions. Precision parameter values of zero are reported for the noise component, if any. Note that values of \code{Inf} are also possible, corresponding to zero-variance, which is most likely under the "\code{UU}" or "\code{UUN}" models.}
#' \item{\code{tau}}{The mixing proportions: either a vector of length \code{G} or, if \code{gating} covariates were supplied, a matrix with an entry for each observation (rows) and component (columns).}}
#' }
#' \item{\code{gating}}{An object of class \code{"MEDgating"} and either \code{"multinom"} or \code{"glm"} (for single-component models) giving the \code{\link[nnet]{multinom}} regression coefficients of the \code{gating} network. If \code{gating} covariates were \emph{NOT} supplied (or the best model has just one component), this corresponds to a RHS of \code{~1}, otherwise the supplied \code{gating} formula. As such, a fitted \code{gating} network is always returned even in the absence of supplied covariates. If there is a noise component (and the option \code{noise.gate=TRUE} is invoked), its coefficients are those for the \emph{last} component. \strong{Users are cautioned against making inferences about statistical significance from summaries of the coefficients in the gating network}.}
#' \item{\code{z}}{The final responsibility matrix whose \code{[i,k]}-th entry is the probability that observation \emph{i} belonds to the \emph{k}-th component. If there is a noise component, its values are found in the \emph{last} column.}
#' \item{\code{MAP}}{The vector of cluster labels for the chosen model corresponding to \code{z}, i.e. \code{max.col(z)}. Observations belonging to the noise component, if any, will belong to component \code{0}.}
#' \item{\code{DBS}}{A matrix of \emph{all} (weighted) mean/median DBS values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note} and \code{\link{dbs}}.}
#' \item{\code{DBSvals}}{A list of lists giving the observation-specific DBS values for \emph{all} fitted models. The first level of the list corresponds to numbers of components, the second to the MEDseq model types.}
#' \item{\code{dbs}}{The (weighted) mean/median DBS value corresponding to the optimal model. May not necessarily be the optimal DBS.}
#' \item{\code{dbsvals}}{Observation-specific DBS values corresponding to the optimum model, which may not be optimal in terms of DBS.}
#' \item{\code{ASW}}{A matrix of \emph{all} (weighted) mean/median ASW values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{ASWvals}}{A list of lists giving the observation-specific ASW values for \emph{all} fitted models. The first level of the list corresponds to numbers of components, the second to the MEDseq model types.}
#' \item{\code{asw}}{The (weighted) mean/median ASW value corresponding to the optimal model. May not necessarily be the optimal ASW.}
#' \item{\code{aswvals}}{Observation-specific ASW values corresponding to the optimum model, which may not be optimal in terms of ASW.}
#' \item{\code{BIC}}{A matrix of \emph{all} BIC values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{ICL}}{A matrix of \emph{all} ICL values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{AIC}}{A matrix of \emph{all} AIC values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{LOGLIK}}{A matrix of \emph{all} maximal log-likelihood values with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{DF}}{A matrix giving the numbers of estimated parameters (i.e. the number of 'used' degrees of freedom) for \emph{all} visited models, with \code{length{G}} rows and \code{length(modtype)} columns. Subtract these numbers from the sample size to get the degrees of freedom. See \code{note}.}
#' \item{\code{ITERS}}{A matrix giving the total number of EM/CEM iterations for \emph{all} visited models, with \code{length{G}} rows and \code{length(modtype)} columns. See \code{note}.}
#' \item{\code{NEC}}{A matrix of \emph{all} NEC values with \code{length{G}} rows and \code{length(modtype)} columns, if available. See \code{note} and the argument \code{do.nec} to \code{\link{MEDseq_control}}.}
#' \item{\code{CV}}{A matrix of \emph{all} cross-validated log-likelihood values with \code{length{G}} rows and \code{length(modtype)} columns, if availabe. See \code{note} and the arguments \code{do.cv} and \code{nfolds} to \code{\link{MEDseq_control}}.}
#' \item{\code{bic}}{The BIC value corresponding to the optimal model. May not necessarily be the optimal BIC.}
#' \item{\code{icl}}{The ICL value corresponding to the optimal model. May not necessarily be the optimal ICL.}
#' \item{\code{aic}}{The AIC value corresponding to the optimal model. May not necessarily be the optimal AIC.}
#' \item{\code{loglik}}{The vector of increasing log-likelihood values for every EM/CEM iteration under the optimal model. The last element of this vector is the maximum log-likelihood achieved by the parameters returned at convergence.}
#' \item{\code{df}}{The number of estimated parameters in the optimal model (i.e. the number of 'used' degrees of freedom). Subtract this number from the sample size to get the degrees of freedom.}
#' \item{\code{iters}}{The total number of EM/CEM iterations for the optimal model.}
#' \item{\code{nec}}{The NEC value corresponding to the optimal model, if available. May not necessarily be the optimal NEC.}
#' \item{\code{cv}}{The cross-validated log-likelihood value corresponding to the optimal model, if available. May not necessarily be the optimal one.}
#' \item{\code{ZS}}{A list of lists giving the \code{z} matrices for \emph{all} fitted models. The first level of the list corresponds to numbers of components, the second to the MEDseq model types.}
#' \item{\code{uncert}}{The uncertainty associated with the \code{classification}.}
#' \item{\code{covars}}{A data frame gathering the set of covariates used in the \code{gating} network, if any. Will contain zero columns in the absence of gating covariates. Supplied gating covariates will be exluded if the optimal model has only one component. May have fewer columns than covariates supplied via the \code{covars} argument also, as only the included covariates are gathered here.}
#' @details The function effectively allows 8 different MEDseq precision parameter settings for models with or without gating network covariates. By constraining the mixing proportions to be equal (see \code{equalPro} in \code{\link{MEDseq_control}}) an extra special case is facilitated in the latter case. 
#' 
#' While model selection in terms of choosing the optimal number of components and the MEDseq model type is performed within \code{\link{MEDseq_fit}}, using one of the \code{criterion} options within \code{\link{MEDseq_control}}, choosing between multiple fits with different combinations of covariates or different initialisation settings can be done by supplying objects of class \code{"MEDseq"} to \code{\link{MEDseq_compare}}.
#' @note Where \code{DBS}, \code{ASW}, \code{BIC}, \code{ICL}, \code{AIC}, \code{LOGLIK}, \code{DF}, \code{ITERS}, \code{NEC}, and \code{CV} contain \code{NA} entries, this corresponds to a model which was not run; for instance a UU model is never run for single-component models as it is equivalent to CU, while a UCN model is never run for two-component models as it is equivalent to CCN. As such, one can consider the value as not really missing, but equivalent to the corresponding value. On the other hand, \code{-Inf} represents models which were terminated due to error, for which a log-likelihood could not be estimated. These objects all inherit the class \code{"MEDCriterion"} for which a dedicated printing functions exists.

#' @importFrom cluster "agnes" "pam"
#' @importFrom matrixStats "colSums2" "logSumExp" "rowLogSumExps" "rowMaxs" "rowMeans2" "rowSums2" "weightedMedian" "weightedMean"
#' @importFrom nnet "multinom"
#' @importFrom stringdist "stringdistmatrix"
#' @importFrom WeightedCluster "wcKMedoids" "wcSilhouetteObs"
#' @export
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @references Keefe Murphy, T. Brendan Murphy, Raffaella Piccarreta, and I. Claire Gormley (2019). Clustering longitudinal life-course sequences using mixtures of exponential-distance models. \emph{To appear}. <\href{https://arxiv.org/abs/1908.07963}{arXiv:1908.07963}>.
#' @keywords clustering main
#' @seealso \code{\link[TraMineR]{seqdef}}, \code{\link{MEDseq_control}}, \code{\link{MEDseq_compare}}, \code{\link{plot.MEDseq}}, \code{\link{I}}
#' @usage
#' MEDseq_fit(seqs, 
#'            G = 1L:9L, 
#'            modtype = c("CC", "UC", "CU", "UU", 
#'                        "CCN", "UCN", "CUN", "UUN"), 
#'            gating = NULL, 
#'            weights = NULL, 
#'            ctrl = MEDseq_control(...), 
#'            covars = NULL, 
#'            ...)
#' @examples
#' \dontshow{library(TraMineR)}
#' # Load the MVAD data
#' data(mvad)
#' mvad$Location <- factor(apply(mvad[,5:9], 1L, function(x) 
#'                  which(x == "yes")), labels = colnames(mvad[,5:9]))
#' mvad          <- list(covariates = mvad[c(3:4,10:14,87)],
#'                       sequences = mvad[,15L:86L], 
#'                       weights = mvad[,2])
#' mvad.cov      <- mvad$covariates
#' states        <- c("EM", "FE", "HE", "JL", "SC", "TR")
#' labels        <- c("Employment", "Further Education", "Higher Education", 
#'                    "Joblessness", "School", "Training")
#' mvad.seq      <- seqdef(mvad$sequences, states=states, labels=labels)
#' 
#' # Fit an exponential-distance model without clustering
#' mod0          <- MEDseq_fit(mvad.seq, G=1)
#' 
#' # Fit a range of unweighted mixture models without covariates
#' # Only consider models with a noise component
#' # Supply some MEDseq_control() arguments
#' \donttest{
#' mod1          <- MEDseq_fit(mvad.seq, G=9:10, modtype=c("CCN", "CUN", "UCN", "UUN"),
#'                             algo="CEM", init.z="hc", criterion="asw")
#' 
#' # Fit a model with weights and gating covariates
#' # Drop the 1st time point which was used to define the weights
#' mvad.seq2     <- seqdef(mvad$sequences[,-1], states=states, labels=labels)
#' mod2          <- MEDseq_fit(mvad.seq2, G=10, modtype="UCN", weights=mvad$weights, 
#'                             gating=~ fmpr + gcse5eq + livboth, covars=mvad.cov)
#'                             
#' # Examine this model in greater detail
#' summary(mod2)
#' summary(mod2$gating)
#' plot(mod2, "clusters")}
MEDseq_fit        <- function(seqs, G = 1L:9L, modtype = c("CC", "UC", "CU", "UU", "CCN", "UCN", "CUN", "UUN"), 
                              gating = NULL, weights = NULL, ctrl = MEDseq_control(...), covars = NULL, ...) {
  call            <- cX  <- match.call()
  if(!inherits(seqs, "stslist")) stop("'seqs' must be of class 'stslist'",        call.=FALSE)
  if(any(seqs     ==
         attr(seqs, "nr")))      stop("Missing values in 'seqs' are not allowed", call.=FALSE)
  SEQ             <- apply(.fac_to_num(seqs), 1L, .num_to_char)
  seqX            <- seqs
  levs            <- attr(seqs, "alphabet")
  attr(SEQ, "N")  <- N2  <- attr(SEQ, "W") <- N <- nrow(seqs)
  attr(SEQ, "T")  <- P   <- ncol(seqs)
  attr(SEQ, "V")  <- V   <- length(levs)
  attr(SEQ, "V1") <- V1  <- V - 1L
  attr(SEQ, "V1V")       <- V1/V
  attr(SEQ, "logV1")     <- log(V1)
  attr(SEQ, "lPV")       <- P * log(V)
  if(any(c(N, P, V)      <= 1))  stop("The number of sequences, the sequence length, and the sequence vocabulary must all be > 1", call.=FALSE)
  if(!is.null(modtype)   &&
     !is.character(modtype))     stop("'modtype' must be a character vector", call.=FALSE)
  modtype         <- if(is.null(modtype)) c("CC", "UC", "CU", "UU", "CCN", "UCN", "CUN", "UUN") else modtype
  l.meth          <- match.arg(modtype, several.ok=TRUE)
  dots            <- list(...)
  if(!missing(ctrl)      &&
     length(dots[names(dots) %in%
     names(ctrl)]) > 0)          stop("Arguments cannot be supplied via the '...' construct when the named argument 'ctrl' is supplied", call.=FALSE) 
  algo            <- ctrl$algo
  criterion       <- ctrl$criterion
  init.z          <- ctrl$init.z
  z.list          <- ctrl$z.list
  dist.mat        <- ctrl$dist.mat
  nstarts         <- switch(EXPR=init.z, random=ctrl$nstarts, 1L)
  startseq        <- seq_len(nstarts)
  equalPro        <- ctrl$equalPro
  equalNoise      <- ctrl$equalNoise
  noise.gate      <- ctrl$noise.gate
  verbose         <- ctrl$verbose
  ctrl$warn       <- TRUE
  x.ctrl          <- list(equalPro=equalPro, noise.gate=noise.gate, equalNoise=equalNoise)
  ctrl$ordering   <- ifelse(ctrl$opti == "first", ctrl$ordering, "none")
  miss.args       <- attr(ctrl, "missing")
  zin.miss        <- miss.args$init.z
  zli.miss        <- miss.args$z.list
  if(!zli.miss)    {
    if(!inherits(z.list,
       "list")    ||
       !all(sapply(z.list, 
       inherits, "matrix")))     stop("'z.list' must be a list of matrices if supplied", call.=FALSE)
    if(zin.miss   &&
       init.z     != "list")   { 
      init.z      <- "list"
      if(isTRUE(verbose))        message("'init.z' set to 'list' as 'z.list' was supplied\n")
    }
  }
  covmiss         <- missing(covars)
  mt1             <- unique(vapply(l.meth, function(lx) switch(EXPR=lx, CC=, UC="CC", CU=, UU="CU", "CCN"), character(1L)))
  mt2             <- unique(vapply(l.meth, function(lx) switch(EXPR=lx, UCN="CCN", UUN="CUN", lx),          character(1L)))
  mtg             <- unique(l.meth)
  n.meths         <- c("CCN", "UCN", "CUN", "UUN")
  n.meth          <- l.meth %in% n.meths
  if(!(tmiss      <- miss.args$tau0) && 
    all(ctrl$tau0 == 0))       {
    if(all(n.meth))              stop("'tau0' is zero: models with noise component cannot be fitted",        call.=FALSE)
    if(any(n.meth))              warning("'tau0' is zero: models with noise component will not be fitted\n", call.=FALSE, immediate.=TRUE)
    l.meths       <- c("CC", "UC", "CU", "UU")
  } else l.meths  <- c("CC", "UC", "CU", "UU", "CCN", "UCN", "CUN", "UUN")
  HAM.mat         <- suppressMessages(seqdist(seqs, "HAM", full.matrix=FALSE))
  if(is.null(dist.mat))        { 
    dist.mat      <- HAM.mat
  } else if((sqrt(length(dist.mat) * 2L + N) !=
             N))                 stop("Invalid 'dist.mat' dimensions", call.=FALSE)
  HAM.mat         <- as.matrix(HAM.mat)
  
  if((gate.x      <- !missing(gating))) {
    if(!inherits(gating, 
                 "formula"))     stop("'gating' must be a formula", call.=FALSE)
    if(!covmiss)   {   
     if((!is.data.frame(covars) &&
         !is.matrix(covars))    ||
      is.null(colnames(covars)))  stop("'covars' must be a data.frame or matrix with named columns if supplied", call.=FALSE)
      call$covars <- covars   <- as.data.frame(covars)
    }
    if(inherits(try(stats::terms(gating), silent=TRUE), "try-error")) {
      if(covmiss)                stop("Can't use '.' in 'gating' formula without supplying 'covars' argument", call.=FALSE)
      gating      <- attr(stats::terms(gating, data=covars), "term.labels")
      gating      <- stats::reformulate(if(length(gating) == 0) "1" else gating, response="z")
    }
    gating        <- tryCatch(stats::update.formula(stats::as.formula(gating), z ~ .), error=function(e) {
                                 stop("Invalid 'gating' network formula supplied", call.=FALSE) })
    if(gating[[3L]]      == 1) { 
      if(verbose)                message("Not including gating network covariates with only intercept on gating formula RHS\n")
      gate.x      <- FALSE
    }
    gate.names    <- stats::terms(gating)
    gate.names    <- labels(gate.names)[attr(gate.names, "order") <= 1]
  } 
  if(gate.x)       {
    covars        <- eval(bquote(stats::model.frame(.(stats::update.formula(gating, NULL ~ .)), data=.(call$covars), drop.unused.levels=TRUE)), envir=parent.frame(), enclos=environment())
    gate.names    <- colnames(covars)
    covars        <- cbind(covars, eval(bquote(stats::model.frame(.(as.formula(paste("~", paste(eval(bquote(all.vars(.(gating))), envir=parent.frame())[-1L], collapse="+")))), data=.(call$covars), drop.unused.levels=TRUE)), envir=parent.frame(), enclos=environment()))
    covars        <- covars[,unique(colnames(covars)), drop=FALSE]
  }
  gate.names      <- if(gate.x)  gate.names[!is.na(gate.names)]
  if(!covmiss)     {
    if(!all(gate.names  %in% 
            colnames(covars)))   stop("Supplied gating covariates not found in supplied 'covars'", call.=FALSE)
    covars        <- if(gate.x)  covars                                                   else as.data.frame(matrix(0L, nrow=N, ncol=0L))
  } else {
    if(any(grepl("\\$", 
                 gate.names)))   stop("Don't supply covariates to the gating network using the $ operator: use the 'covars' argument instead", call.=FALSE)
    covars        <- if(gate.x)  stats::model.frame(gating[-2L], drop.unused.levels=TRUE) else as.data.frame(matrix(0L, nrow=N, ncol=0L))
  }
  if(nrow(covars) != N)          stop("'gating' covariates must contain the same number of rows as 'seqs'", call.=FALSE)
  glogi           <- vapply(covars, is.logical, logical(1L))
  covars[,glogi]  <- sapply(covars[,glogi], as.factor)
  if(covmiss)      {
    covars        <- data.frame(covars, stringsAsFactors=TRUE)
  }
  
  if(ctrl$do.wts  <- do.wts   <- 
     !missing(weights))   {
    if(is.null(weights))         stop("Invalid 'weights' supplied", call.=FALSE)
    if(!is.numeric(weights)   ||
       length(weights)   != N)   stop(paste0("'weights' must be a numeric vector of length N=", N), call.=FALSE)
    if(any(weights < 0)  || 
       any(!is.finite(weights))) stop("'weights' must be positive and finite", call.=FALSE)
    if(ctrl$do.wts       <- 
       do.wts     <- (length(unique(weights)) > 1)) {
      weights     <- replace(weights, weights == 0, .Machine$double.eps)
      attr(SEQ, "Weights")    <- weights
      attr(SEQ, "W")          <- sum(weights)
    }
  } 
  do.uni          <- ctrl$unique
  DF              <- seqs   
  uni.sum         <- sum(!duplicated(DF))
  DF              <- if(gate.x)  cbind(DF, covars)  else DF
  DF              <- if(do.wts)  cbind(DF, weights) else DF
  uni.ind         <- !duplicated(DF)
  sum.uni         <- sum(uni.ind)
  if(sum.uni < N  && !do.uni  &&
     verbose)                   message(paste0("Number of unique observations (", sum.uni, ") is less than N (", N, ")", ifelse(uni.sum < sum.uni, paste0(" - \nNumber of unique sequences only, ignoring covariates, is ", uni.sum, ":\n"), ": "), "Consider setting 'unique'=TRUE\n"))
  if(do.uni       <- do.uni   &&
     sum.uni < N)  {            
    if(verbose)                 message(paste0("Proceeding with ", sum.uni, " unique observations, out of N=", N, "\n"))
    agg.DF        <- merge(data.frame(cbind(id=seq_len(N)), DF), data.frame(stats::aggregate(cbind(DF[0L], count=1L), DF, length)), by=colnames(DF), sort=FALSE)
    agg.id        <- order(agg.DF$id)
    c2            <- agg.DF$count[agg.id]
    counts        <- c2[uni.ind]
    weights       <- if(do.wts) counts * agg.DF$weights[agg.id][uni.ind] else counts
    if(ctrl$do.wts       <- (length(unique(weights)) > 1)) {
      dist.mat2   <- dist.mat
      HAM.mat2    <- HAM.mat[uni.ind,uni.ind]
      w2          <- rep(0L, N)
      w2[uni.ind]        <- weights
      dist.mat    <- stats::as.dist(as.matrix(dist.mat)[uni.ind,uni.ind])
      seqs        <- seqs[uni.ind,,   drop=FALSE]
      atts        <- attributes(SEQ)
      SEQ         <- apply(.fac_to_num(seqs), 1L, .num_to_char)
      attributes(SEQ)         <- atts
      attr(SEQ, "N")     <- N <- nrow(seqs)
      attr(SEQ, "Weights")    <- weights
      attr(SEQ, "W")          <- sum(weights)
      covars      <- covars[uni.ind,, drop=FALSE]
      dis.agg     <- rep(seq_len(N), counts)[agg.id]
    }
  } 
  if(!do.uni || !ctrl$do.wts)  {
    uni.ind       <- rep(TRUE, N)
    w2            <- weights
    dist.mat2     <- dist.mat
    HAM.mat2      <- HAM.mat
  }
  
  if(any(G        != floor(G))    &&
     any(G         < 1))         stop("'G' must be strictly positive", call.=FALSE)
  if(any(G        >= N))       {
    G             <- G[G <= N]
    if(length(G)   > 1)        { warning("Removing G values >= the number of observations\n",  call.=FALSE, immediate.=TRUE)
    } else                       stop("G values must be less than the number of observations", call.=FALSE)
  }
  G               <- rG  <- unique(sort(as.integer(G)))
  do.nec          <- ctrl$do.nec
  if(!do.nec      &&
     algo         != "CEM"    &&
     criterion    == "nec"    &&
     !all(G == 1))             { 
    if(verbose)                  message("Forcing 'do.nec' to TRUE as criterion='nec'\n")
    do.nec        <- TRUE
  }
  if(do.nec &&
     algo         != "CEM"    &&
     !any(G == 1))             { 
    if(verbose)                  message("Forcing G=1 models to be fitted for NEC criterion computation\n")
    G             <- rG  <- unique(c(1L, G))  
  }
  if(all(G  == 1)) { if(verbose) message("Silhouettes not computed as only single component models are being fitted\n")
    do.dbs        <- 
    do.asw        <- FALSE
    if(criterion  == "dbs")    { message("DBS criterion cannot be used to select among only single-component models: defaulting to 'criterion'=\"bic\"\n")
      criterion   <- "bic"
    }
    if(criterion  == "asw")    { message("ASW criterion cannot be used to select among only single-component models: defaulting to 'criterion'=\"bic\"\n")
      criterion   <- "bic"
    }
    if(do.nec     && 
       !(do.nec   <- FALSE))   {
      if(verbose)                message("Forcing 'do.nec' to FALSE as only single component models are being fit\n")
    }   
    if(criterion  == "nec")      stop("NEC criterion cannot be used to select among only single-component models", call.=FALSE)
  } else if(algo  == "CEM")    { 
    if(verbose)                  message("Density-based silhouettes not computed as the CEM algorithm is employed\n")
    do.asw        <- TRUE
    do.dbs        <- FALSE
    if(criterion  == "dbs")      stop("DBS criterion cannot be used to select among models fitted via CEM", call.=FALSE)
    if(do.nec     &&
       !(do.nec   <- FALSE))   {
      if(verbose)                message("Forcing 'do.nec' to FALSE as models are being fit via CEM\n")
    }  
    if(criterion  == "nec")      stop("NEC criterion cannot be used to select among models fitted via CEM", call.=FALSE)
  } else do.dbs   <- do.asw   <- TRUE
  len.G           <- length(G)
  if(any(G > 1L))  {
    G1            <- any(G == 1L)
    G2            <- any(G == 2L)
    GG            <- any(G  > 2L)
    all.mod       <- if(all(G1, G2, GG)) unique(c(mtg, mt2, mt1)) else if(all(G1, G2)) unique(c(mt2, mt1)) else if(all(G1, GG)) unique(c(mtg, mt1)) else if(all(G2, GG)) unique(c(mtg, mt2)) else if(G2) mt2 else mtg
    all.mod       <- l.meths[l.meths %in% all.mod]
    if(init.z     == "hc")     {
      hcZ         <- if(do.wts) stats::hclust(dist.mat2, method="ward.D2", members=w2) else agnes(dist.mat2, diss=TRUE, method="ward")
    }
    if(!zli.miss)  {
     if(length(z.list) != len.G) stop(paste0("'z.list' must be a list of length ", len.G), call.=FALSE)  
     if(!all(vapply(z.list, ncol, numeric(1L)) == 
             G))                 stop("Each element of 'z.list' must have 'G' columns", call.=FALSE)
     if(!all(vapply(z.list, nrow, numeric(1L)) == 
             N2))                stop(paste0("Each element of 'z.list' must have N=", N2, " rows"), call.=FALSE) 
      z.list      <- lapply(seq_along(G), function(g) z.list[[g]][uni.ind,, drop=FALSE])
    }
    if(all(zli.miss, init.z   == 
           "list"))              stop(paste0("'z.list' must be supplied if 'init.z' is set to 'list'"), call.=FALSE)
  } else all.mod  <- l.meths[l.meths %in% mt1]
  multi           <- length(all.mod) > 1
  cvsel           <- ctrl$do.cv
  ctrl$do.cv      <- FALSE
  if(!cvsel       && 
     criterion    == "cv")     { 
    if(verbose)                  message("Forcing 'do.cv' to TRUE as criterion='cv'\n")
    cvsel         <- TRUE
  }
  if(ctrl$numseq  <- any(c("CU", "UU", "CUN", "UUN") %in% all.mod, ctrl$opti == "mode", ctrl$ordering != "none")) {
    numseq        <- sapply(SEQ, .char_to_num)
    attr(numseq, "T")    <- P
  } else numseq   <- NULL
  
  BICs            <-
  ICLs            <-
  AICs            <-
  NECs            <- 
  DBSs            <- 
  ASWs            <- 
  LL.x            <- 
  DF.x            <-
  IT.x            <-
  Nzero.x         <-
  Ninfty.x        <- provideDimnames(matrix(NA, nrow=len.G, ncol=length(all.mod)), base=list(as.character(G), all.mod))
  ZS              <- 
  DBSvals         <- 
  ASWvals         <- replicate(len.G, list())
  if(isTRUE(cvsel))         {
    nfolds        <- pmin(ctrl$nfolds, N)
    if(length(nfolds) != 1 ||
       !is.numeric(nfolds) ||
       (nfolds    <= 1     ||
        floor(nfolds) !=
        nfolds))                 stop("'nfolds' must be a single integer > 1 if cross-validated likelihood is invoked", call.=FALSE)
    CV.x          <- BICs
    cv.ind        <- sample(N)
    cv.SEQ        <- SEQ[cv.ind]
    gCV           <- covars[cv.ind,,        drop=FALSE]
    hmCV          <- HAM.mat[cv.ind,cv.ind, drop=FALSE]
    if(ctrl$numseq)         {
      cv.numseq   <- numseq[,cv.ind, drop=FALSE]
    }
    if(ctrl$do.wts)         {
      cv.wts      <- weights[cv.ind]
    }
    cv.folds      <- cut(seq_len(N), breaks=nfolds, labels=FALSE)
    fcount        <- tabulate(cv.folds, nfolds)
    Nfcount       <- N - fcount
    foldseq       <- seq_len(nfolds)
  }
  crit.tx         <-
  crit.gx         <- -sqrt(.Machine$double.xmax)
  noise           <- all.mod %in% c("CCN", "UCN", "CUN", "UUN")
  nonoise         <- any(!noise)
  noise           <- any(noise)

  gate.G          <- matrix(ifelse(rG > 1, gate.x, FALSE), nrow=2L, ncol=length(rG), byrow=TRUE)
  if(gate.x)       {
    Gn            <- G - !noise.gate
    if(gate.x     &&
      ((any(Gn    <= 1)  && noise) ||
       any(G      <= 1))) {      
      if(verbose)                message(paste0("Can't include gating network covariates ", ifelse(noise.gate, "in a single component mixture", "where G is less than 3 when 'noise.gate' is FALSE\n")))
     gate.G[2L,Gn <= 1]  <- FALSE
    }
  } else           {
    Gn            <- G
    gating        <- stats::as.formula(z ~ 1)
    environment(gating)  <- environment()
  }
  noise.gate      <- ifelse(gate.G, noise.gate, TRUE)
  if(all(equalPro, gate.x)) { 
    if(verbose)                  message("Can't constrain mixing proportions to be equal when gating covariates are supplied\n")
    equalPro      <- FALSE
  }
  equal.tau       <- rbind(ifelse(G == 1, TRUE, equalPro), ifelse(Gn < 1, TRUE, equalPro)) & !gate.G
  equal.n0        <- (rbind(G == 1, Gn == 1) | equalNoise) & equal.tau
  attr(covars, "Gating") <- gate.names
  if(!identical(gating, 
                .drop_constants(covars, 
                gating)))        stop("Constant columns exist in gating formula; remove offending gating covariate(s) and try again", call.=FALSE)
  G.last          <- G[len.G]
  if(isTRUE(verbose))            message("\n################\n")
  
  for(g  in G)     {
    if(isTRUE(verbose))     {    message(paste0("\n", g, " cluster model", ifelse(multi, "s", ""), " -\n"))
      last.G      <- g   == G.last
    }
    if(ctrl$numseq)         {
      attr(numseq,   "G")  <- g
    } 
    attr(SEQ, "G")         <- g
    h             <- which(G   == g)
    g0 <- attr(SEQ, "G0")  <- g - noise
    if(isTRUE(noise))   {
      tau0        <- if(tmiss) 1/g else ctrl$tau0
      if(length(tau0)   > 1)   {
        tau0      <- tau0[uni.ind]
        if(anyNA(tau0))          stop("Invalid 'tau0' supplied", call.=FALSE)
        if(all(gate.G[2L,h]   && 
           noise.gate[2L,h]))  {
          if(N != length(tau0))  stop(paste0("'tau0' must be a scalar or a vector of length N=", N), call.=FALSE)
        } else                   stop("'tau0' must be a scalar in the interval (0, 1)", call.=FALSE)
      }
    }

    if(g  > 1)     {
      algog       <- algo
      if(init.z   == "random" &&
         nstarts   > 1)     {
        if(isTRUE(nonoise)) {
          zg      <- replicate(nstarts, list(.unMAP(sample(seq_len(g),  size=N, replace=TRUE), groups=seq_len(g))))
        }
        if(isTRUE(noise))   {
          if(g0    > 1)     {
            zg0   <- replicate(nstarts, list(.unMAP(sample(seq_len(g0), size=N, replace=TRUE), groups=seq_len(g0))))
            zg0   <- lapply(zg0, function(x) cbind(x * (1 - tau0), tau0))
          } else   {
            zg0   <- matrix(tau0, nrow=N, ncol=2L)
          }
        }
      } else       {
        if(isTRUE(nonoise)) {
          switch(EXPR=init.z, 
                 list=      {
          zg      <- .renorm_z(z.list[[h]])
          },       {
          zg      <- .unMAP(switch(EXPR=init.z, 
                                   random=sample(seq_len(g),  size=N, replace=TRUE),
                                   kmedoids= if(do.wts) {
                                     zz <- wcKMedoids(dist.mat, k=g,  weights=weights, cluster.only=TRUE)
                                       as.numeric(factor(zz, labels=seq_along(unique(zz))))
                                     } else pam(dist.mat2, k=g, cluster.only=TRUE)[uni.ind], 
                                   hc=stats::cutree(hcZ, k=g)[uni.ind]), groups=seq_len(g))
          })
        }
        if(isTRUE(noise))   {
          switch(EXPR=init.z, 
                 list=      {
          zg0     <- .renorm_z(z.list[[h]])
          },       {
          if(g0    > 1)     {
            zg0   <- .unMAP(switch(EXPR=init.z, 
                                   random=sample(seq_len(g0), size=N, replace=TRUE),
                                   kmedoids= if(do.wts) {
                                     zz <- wcKMedoids(dist.mat, k=g0, weights=weights, cluster.only=TRUE)
                                       as.numeric(factor(zz, labels=seq_along(unique(zz))))
                                     } else pam(dist.mat2, k=g0, cluster.only=TRUE)[uni.ind],  
                                   hc=stats::cutree(hcZ, k=g0)[uni.ind]), groups=seq_len(g0))
            zg0   <- cbind(zg0 * (1 - tau0), tau0)
          } else   {
            zg0   <- matrix(tau0, nrow=N, ncol=2L)
          }
          })
        }
      }
      modtypes    <- if(g  == 2L) mt2 else mtg
    } else         {
      algog       <- "EM"
      zg  <- zg0  <- matrix(1L, nrow=N)
      modtypes    <- mt1
    }
      
    T.last        <- modtypes[length(modtypes)]
    for(modtype   in modtypes) {
      if(isTRUE(verbose))      { message(paste0("\n\tModel: ", modtype, "\n"))
        last.T    <- modtype  == T.last
      }
      ctrl$nmeth  <- is.element(modtype, n.meths)
      ctrl$equalNoise    <- equal.n0[ctrl$nmeth    + 1L,h]
      ctrl$equalPro      <- equal.tau[ctrl$nmeth   + 1L,h]
      ctrl$gate.g        <- gate.G[ctrl$nmeth      + 1L,h]
      ctrl$noise.gate    <- ifelse(ctrl$nmeth, noise.gate[2L,h], TRUE)
      zm          <- if(attr(SEQ, "Noise") <- gN0 <- ctrl$nmeth) zg0 else zg
      if(gN0      &&   !ctrl$noise.gate    && algog  != "EM") {
        if(init.z == "random" &&
           nstarts > 1)   {
          zm      <- lapply(zm, function(x) { x[,-G] <- replace(x[,-G], x[,-G] > 0, 1L); x })
        } else zm[,-G]   <- replace(zm[,-G], zm[,-G] > 0, 1L)
      }
      m           <- which(modtype == modtypes)
      if(init.z   == "random" &&
         g - gN0   > 1        &&
         nstarts   > 1)        {
        EMX       <- list()
        for(i in startseq)     {
         if(isTRUE(verbose))     message(paste0("\tRandom Start: #", i, "...\n"))
         switch(EXPR=algog,
               cemEM=          {
            ctrl$algo      <- "CEM"
            EMX[[i]]       <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=zm[[i]],  ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2)
            if(!EMX[[i]]$ERR)  {
              ctrl$algo    <- "EM"
              tmpEMX       <- EMX[[i]]
              j.i          <- pmax(tmpEMX$j, 2L)
              EMX[[i]]     <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=tmpEMX$z, ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2, ll=tmpEMX$ll[c(j.i - 1L, j.i)])
              if(EMX[[i]]$ERR) {
                EMX[[i]]   <- tmpEMX
                ctrl$algo  <- "CEM"
              }
            }
         }, EMX[[i]]       <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=zm[[i]],  ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2))
        }
        EMX       <- EMX[[which.max(vapply(lapply(EMX, "[[", "ll"), max, numeric(1L)))]]
      } else       {
        switch(EXPR=algog,
              cemEM=        {
          ctrl$algo        <- "CEM"
          EMX              <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=zm,       ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2)
          if(!EMX$ERR)      {
            ctrl$algo      <- "EM"
            tmpEMX         <- EMX
            j.i            <- pmax(EMX$j, 2L)
            EMX            <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=EMX$z,    ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2, ll=EMX$ll[c(j.i - 1L, j.i)])
            if(EMX$ERR)     {
              EMX          <- tmpEMX
              ctrl$algo    <- "CEM"
            }
          }
        }, EMX             <- .EM_algorithm(SEQ=SEQ, numseq=numseq, g=g, modtype=modtype, z=zm,       ctrl=ctrl, gating=gating, covars=covars, HAM.mat = HAM.mat2))
      }
      ERR         <- EMX$ERR
      j           <- EMX$j
      Mstep       <- EMX$Mstep
      ll          <- EMX$ll
      z           <- EMX$z
      cvsel.X     <- cvsel && !ERR
      j2          <- max(1L, j - switch(EXPR=algog, cemEM=1L, 2L))
      if(isTRUE(verbose))        message(paste0("\t\t# Iterations: ", ifelse(ERR, "stopped at ", ""), j2, ifelse(last.G && last.T, "\n\n", "\n")))

      if(all((Mstep$lambda -> lambda) == 0) && cvsel.X) {
        CVll      <- -N * P * log(V)
      } else if(cvsel.X)    {
        ctrl$warn <- FALSE
        lCV       <- vector("numeric", nfolds)
        zCV       <- z[cv.ind,,    drop=FALSE]
        for(i in foldseq)   {
          testX   <- which(cv.folds == i)
          CVS     <- cv.SEQ[testX]
          SCV     <- cv.SEQ[-testX]
          CVz     <- zCV[-testX,, drop=FALSE]
          CVg     <- gCV[-testX,, drop=FALSE]
          attributes(CVS)  <-
          attributes(SCV)  <- attributes(SEQ)
          attr(CVS, "N")   <- fcount[i]
          attr(SCV, "N")   <- Nfcount[i]
          if(ctrl$do.wts)   {
           attr(CVS, "Weights") <- cv.wts[testX]
           attr(SCV, "Weights") <- cv.wts[-testX]
           attr(CVS, "W")  <- sum(attr(CVS, "Weights"))
           attr(SCV, "W")  <- sum(attr(SCV, "Weights"))
          }
          if(any(modtype %in% c("CU", "UU", "CUN", "UUN"), ctrl$opti == "mode", ctrl$ordering != "none")) {
            nCV            <- cv.numseq[,-testX, drop=FALSE]
            CVn            <- cv.numseq[,testX,  drop=FALSE]
            attr(nCV, "G") <- attr(numseq, "G")
            attr(nCV, "T") <- attr(numseq, "T")
          } else   {
            nCV   <-
            CVn   <- NULL
          }
          EMX     <- .EM_algorithm(SEQ=SCV, numseq=nCV, g=g, modtype=modtype, z=CVz, ctrl=ctrl, gating=gating, covars=CVg, HAM.mat = if(ctrl$opti != "mode") hmCV[-testX,-testX, drop=FALSE])
          MCV     <- EMX$Mstep
          if(is.matrix(MCV$tau)) {
            tau.tmp        <- stats::predict(MCV$fitG, newdata=gCV[testX,, drop=FALSE], type="probs")
            MCV$tau        <- if(ctrl$noise.gate) tau.tmp else cbind(tau.tmp * (1 - MCV$tau[1L,g]), MCV$tau[1L,g])
            rm(tau.tmp)
          }
          MCV$dG  <- NULL
          ctrl$do.cv       <- TRUE
          if(is.infinite(lCV[i] <- 
             .E_step(seqs=CVS, params=MCV, modtype=modtype, ctrl=ctrl, 
             numseq=CVn)))       break
          ctrl$do.cv       <- FALSE
        }
        CVll      <- sum(lCV)
        ctrl$warn <- TRUE
      }
      
      z           <- if(do.uni) z[dis.agg,, drop=FALSE] else z
      log.lik     <- ll[j]
      nzero       <- sum(lambda == 0)
      ninfty      <- sum(is.infinite(lambda))
      Gfit        <- if(ctrl$gate.g) Mstep$fitG
      gate.pen    <- ifelse(ctrl$gate.g, length(stats::coef(Gfit)) + !ctrl$noise.gate, 
                     ifelse(ctrl$equalPro, as.integer(ctrl$nmeth && !ctrl$equalNoise), g - 1L))
      choice      <- .choice_crit(ll=log.lik, seqs=SEQ, z=z, modtype=modtype, nonzero=sum(lambda != 0), gate.pen=gate.pen)
      bicx        <- choice$bic
      iclx        <- choice$icl
      aicx        <- choice$aic
      dfx         <- choice$df
      tmp.MAP     <- if(g > 1) max.col(z)
      if(do.dbs   && g > 1) {
        DBS       <- if(ctrl$do.wts) dbs(z, weights=w2, MAP=tmp.MAP, ...) else dbs(z, MAP=tmp.MAP, ...)
        dbsx      <- DBS$wmsw
        DBSvals[[h]][[m]]  <- if(ERR)   NA else DBS$silvals
        attr(DBSvals[[h]][[m]], "G")         <- g
        attr(DBSvals[[h]][[m]], "ModelType") <- modtype
      } else dbsx <- NA
      if(do.asw   && g > 1) {
        ASWvals[[h]][[m]]  <- ASW <- if(ERR) NA else cbind(tmp.MAP, wcSilhouetteObs(dist.mat2, tmp.MAP, weights=if(ctrl$do.wts) w2, measure=ifelse(ctrl$do.wts, "ASWw", "ASW")))
        colnames(ASWvals[[h]][[m]])          <- 
        colnames(ASW)      <- c("cluster", "asw_width")
        summ      <- ifelse(any(names(list(...)) == "summ") && list(...)$summ == "median", "median", "mean")
        aswx      <- ifelse(ERR, NA, ifelse(ctrl$do.wts, 
                                            switch(EXPR=summ, median=weightedMedian(ASW[,2L], w=w2),
                                                                mean=weightedMean(ASW[,2L],   w=w2)), 
                                            switch(EXPR=summ, medians=stats::median(ASW[,2L]), mean=mean(ASW[,2L]))))
        attr(ASWvals[[h]][[m]], "G")         <- g
        attr(ASWvals[[h]][[m]], "ModelType") <- modtype
      } else aswx <- NA
      necx        <- ifelse(g > 1 && do.nec, -sum(apply(z, 1L, .entropy))/(log.lik - LL.x[1L,switch(EXPR=modtype, CC=, UC="CC", CU=, UU="CU", "CCN")]), NA)
      crit.t      <- switch(EXPR=criterion, cv=CVll, bic=bicx, icl=iclx, aic=aicx, nec=necx, dbs=dbsx, asw=aswx)
      crit.t      <- ifelse(is.na(crit.t) || ERR, -Inf, crit.t)
      if(crit.t    > crit.tx)     {
        crit.tx   <- crit.t
        theta.x   <- Mstep$theta
        lambda.x  <- lambda
        tau.x     <- Mstep$tau
        z.x       <- z
        ll.x      <- ll
        gp.x      <- gate.pen
        if(gcov.x <- ctrl$gate.g) {
          fit.x   <- Gfit  
        }
      }
      BICs[h,modtype]      <- ifelse(ERR, -Inf, bicx)
      ICLs[h,modtype]      <- ifelse(ERR, -Inf, iclx)
      AICs[h,modtype]      <- ifelse(ERR, -Inf, aicx)
      NECs[h,modtype]      <- ifelse(ERR,  Inf, -necx)
      DBSs[h,modtype]      <- ifelse(ERR, -Inf, dbsx)
      ASWs[h,modtype]      <- ifelse(ERR, -Inf, aswx)
      LL.x[h,modtype]      <- ifelse(ERR, -Inf, log.lik)
      DF.x[h,modtype]      <- ifelse(ERR, -Inf, dfx)
      IT.x[h,modtype]      <- ifelse(ERR,  Inf, j2)
      Nzero.x[h,modtype]   <- ifelse(ERR,  NA,  nzero)
      Ninfty.x[h,modtype]  <- ifelse(ERR,  NA,  ninfty)
      ZS[[h]][[m]]         <- if(ERR)      NA   else z
      if(cvsel)    {
        CV.x[h,modtype]    <- ifelse(ERR, -Inf, CVll)
      }
    } # for (modtype)

    if(crit.tx     > crit.gx)  {
      crit.gx     <- crit.tx
      x.theta     <- theta.x
      x.lambda    <- lambda.x
      x.tau       <- tau.x
      x.z         <- z.x
      x.ll        <- ll.x
      x.gp        <- gp.x
      if(x.gcov   <- gcov.x)   {
        fitG      <- fit.x
      }
    }
    ZS[[h]]       <- stats::setNames(ZS[[h]],               modtypes)
    if(do.dbs     && g > 1)    {
      DBSvals[[h]]         <- stats::setNames(DBSvals[[h]], modtypes)  
    }
    if(do.asw     && g > 1)    {
      ASWvals[[h]]         <- stats::setNames(ASWvals[[h]], modtypes)  
    }
  } # for (g)

  seqs            <- seqX
  if(any(l.warn   <- x.ll  != cummax(x.ll))) {
    if(which.max(l.warn)   != 
       length(x.ll))             warning("Log-likelihoods are not strictly increasing\n", call.=FALSE)
  }
  if(any(IT.x[!is.na(IT.x)]
         == ctrl$itmax))         warning(paste0("One or more models failed to converge in the maximum number of allowed iterations (", ctrl$itmax, ")\n"), call.=FALSE)
  class(BICs)     <-
  class(ICLs)     <-
  class(AICs)     <-
  class(DF.x)     <- 
  class(IT.x)     <-
  class(LL.x)     <- "MEDcriterion"
  attr(BICs, "Criterion")  <- "BIC"
  attr(ICLs, "Criterion")  <- "ICL"
  attr(AICs, "Criterion")  <- "AIC"
  attr(DF.x, "Criterion")  <- "DF"
  attr(IT.x, "Criterion")  <- "ITERS"
  attr(LL.x, "Criterion")  <- "loglik"
  attr(LL.x, "Weighted")   <- do.wts
  CRITs           <- switch(EXPR=criterion, cv=CV.x, bic=BICs, icl=ICLs, aic=AICs, nec=NECs, dbs=DBSs, asw=ASWs)
  best.ind        <- which(CRITs == switch(EXPR=criterion, nec=-crit.gx, crit.gx), arr.ind=TRUE)
  if(nrow(best.ind) > 1)    {    warning(paste0("Ties for the optimal model exist according to the '", toupper(criterion), "' criterion: choosing the most parsimonious model\n"), call.=FALSE, immediate.=TRUE)
    best.ind      <- which(DF.x  == min(DF.x[best.ind]), arr.ind=TRUE)
    best.ind      <- best.ind[which.min(best.ind[,1L]),]
  }
  best.G          <- best.ind[1L]
  best.mod        <- colnames(CRITs)[best.ind[2L]]
  G               <- G[best.G]
  x.bic           <- BICs[best.ind]
  x.icl           <- ICLs[best.ind]
  x.aic           <- AICs[best.ind]
  x.nec           <- NECs[best.ind]
  x.dbs           <- DBSs[best.ind]
  x.asw           <- ASWs[best.ind]
  attr(BICs, "G")          <-
  attr(ICLs, "G")          <-
  attr(AICs, "G")          <-
  attr(DF.x, "G")          <-
  attr(IT.x, "G")          <-
  attr(LL.x, "G")          <- rownames(BICs)
  attr(BICs, "modelNames") <-
  attr(ICLs, "modelNames") <-
  attr(AICs, "modelNames") <-
  attr(DF.x, "modelNames") <-
  attr(IT.x, "modelNames") <-
  attr(LL.x, "modelNames") <- colnames(BICs)
  if(cvsel)        {
    CV.x          <- CV.x * 2
    x.cv          <- CV.x[best.ind]
    class(CV.x)   <- "MEDcriterion"
    attr(CV.x, "Criterion")     <- "CV"
    attr(CV.x, "G")             <- rownames(BICs)
    attr(CV.x, "modelNames")    <- colnames(BICs)
    attr(CV.x, "Weighted")      <- 
    attr(x.cv, "Weighted")      <- do.wts
  }
  if(len.G > 1    && verbose)    {
    if(G          == min(rG))    message("Best model occurs at the min of the number of components considered\n")
    if(G          == max(rG))    message("Best model occurs at the max of the number of components considered\n")
  }
  
  noise           <- best.mod %in% c("CCN", "UCN", "CUN", "UUN")
  attr(x.lambda, "Nzero")       <- Nzero.x[best.ind]
  attr(x.lambda, "Ninfty")      <- Ninfty.x[best.ind]
  attr(DF.x,     "Nzero")       <- Nzero.x
  attr(DF.x,     "Ninfty")      <- Ninfty.x
  attr(DF.x,     "Gate.Pen")    <- x.gp
  if(any(apply(x.lambda == 0, 1L, all))) {
    x.theta                     <- if(G > 1) rbind(do.call(rbind, lapply(x.theta[-G], .char_to_num)), NA) else matrix(NaN, nrow=1L, ncol=P)
  } else x.theta                <- do.call(rbind, lapply(x.theta, .char_to_num))
  storage.mode(x.theta)         <- "integer"
  attr(x.theta, "alphabet")     <- levs
  attr(x.theta, "labels")       <- attr(seqs, "labels")
  attr(x.theta, "lambda")       <- switch(EXPR=best.mod, CCN=, CUN=rbind(matrix(x.lambda[1L,], nrow=G - 1L, ncol=P, byrow=best.mod == "CUN"), 0L), matrix(x.lambda, nrow=G, ncol=P, byrow=best.mod == "CU"))
  class(x.theta)                <- "MEDtheta"
  Gseq            <- seq_len(G)
  colnames(x.z)   <- if(G == 1  && noise) "Cluster0" else paste0("Cluster", if(noise) replace(Gseq, G, 0L) else Gseq)
  MAP             <- max.col(x.z)
  MAP             <- if(noise) replace(MAP, MAP == G, 0L) else MAP
  equalPro        <- equal.tau[1L   + noise,best.G] 
  equalNoise      <- equal.n0[1L    + noise,best.G] 
  noise.gate      <- noise.gate[1L  + noise,best.G]
  noise.gate      <- ifelse(noise, noise.gate, TRUE)
  covars          <- if(do.uni) covars[dis.agg,, drop=FALSE] else covars
  rownames(covars)                 <- seq_along(MAP)
  if(!(gate.G[1L   + noise,best.G] -> bG))  {
    if(G > 1)      {
      if(ctrl$do.wts)            {
        z         <- x.z * w2
        z[apply(z == 0, 1L, all),] <- .Machine$double.eps
      } else z    <- x.z
      fitG        <- multinom(gating, trace=FALSE, data=covars, maxit=ctrl$g.itmax, reltol=ctrl$g.tol, MaxNWts=ctrl$MaxNWts)
      if(equalPro && !equalNoise   && noise) {
        tau0      <- mean(z[,G])
        x.tau     <- c(rep((1 - tau0)/(G - 1L), G  - 1L), tau0)
      } else       {
        x.tau     <- if(equalPro   || equalNoise) rep(1/G, G) else fitG$fitted.values[1L,]
      }
      x.tau       <- stats::setNames(x.tau, paste0("Cluster", if(noise) replace(Gseq, G, 0L) else Gseq))
      if(equalPro) {
        fitG$wts[]              <- 0L
        fitG$fitted.values      <- matrix(x.tau, nrow=nrow(z), ncol=G, byrow=TRUE)
        fitG$residuals          <- z - fitG$fitted.values
      }
    }   else       {
      fitG        <- suppressWarnings(stats::glm(z ~ 1, family=stats::binomial()))
    }
  }     else       {
    x.tau         <- if(do.uni) x.tau[dis.agg,, drop=FALSE] else x.tau
  }
  fitG$lab        <- if(noise   && noise.gate && G > 1) c(paste0("Cluster", Gseq[-G]), "Noise") else if(noise && G > 1) paste0("Cluster", Gseq[-G]) else paste0("Cluster", Gseq)
  attr(fitG, "EqualNoise")      <- equalNoise
  attr(fitG, "EqualPro")        <- equalPro
  attr(fitG, "Formula")         <- Reduce(paste, deparse(gating[-2L]))
  attr(fitG, "Maxit")           <- ctrl$g.itmax
  attr(fitG, "MaxNWts")         <- ctrl$MaxNWts
  attr(fitG, "Noise")           <- noise
  attr(fitG, "NoiseGate")       <- noise.gate
  attr(fitG, "Reltol")          <- ctrl$g.tol
  class(fitG)     <- c("MEDgating", class(fitG))
  if(isTRUE(verbose))            message(paste0("\n\t\tBest Model", ifelse(length(CRITs) > 1, paste0(" (according to ", toupper(criterion), "): "), ": "), best.mod, ", with ",  paste0(G, " component", ifelse(G > 1, "s", "")),
                                         ifelse(bG | x.gcov, paste0(" (incl. ", ifelse(do.wts, "weights and ", ""), "gating network covariates)"), ifelse(do.wts, ifelse(x.ctrl$equalPro && G > 1, " (incl. weights and equal mixing proportions)", " (incl. weights)"), ifelse(x.ctrl$equalPro && G > 1, " (and equal mixing proportions)", ""))), "\n\t\t",
                                         ifelse(cvsel, paste0("CV = ",  round(x.cv,  2L), " | "), ""),
                                         ifelse(G > 1 && do.nec, paste0("NEC = ", round(x.nec, 2L), " | "), ""),
                                         ifelse(G > 1 && do.dbs, paste0("DBS = ", round(x.dbs, 2L), " | "), ""),
                                         ifelse(G > 1 && do.asw, paste0("ASW = ", round(x.asw, 2L), " | "), ""),
                                     "BIC =", round(x.bic, 2L), " | ICL =", round(x.icl, 2L), " | AIC =", round(x.aic, 2L), "\n\n"))
  params          <- list(theta   = x.theta,
                          lambda  = x.lambda,
                          tau     = x.tau)
  attr(seqs, "weights")         <- if(do.wts) w2 else rep(1L, N)
  results         <- list(call    = cX,
                          data    = seqs,
                          modtype = best.mod,
                          G       = G,
                          params  = params,
                          gating  = fitG,
                          z       = x.z,
                          MAP     = MAP)
  if(do.dbs)       {
    DBSs          <- if(any(rG  == 1)) DBSs[-1L,, drop=FALSE]            else DBSs
    class(DBSs)   <- "MEDcriterion"
    attr(DBSs, "Criterion")     <- "DBS"
    attr(DBSs, "G")             <- if(any(rG  == 1)) rownames(BICs)[-1L] else rownames(BICs)
    attr(DBSs, "modelNames")    <- colnames(BICs)
    DBSvals       <- if(any(rG  == 1)) stats::setNames(DBSvals, rG)[-1L] else stats::setNames(DBSvals, rG)
    attr(DBSs,    "Weighted")   <- 
    attr(DBSvals, "Weighted")   <- do.wts
    attr(DBSs,    "Summ")       <-
    attr(DBSvals, "Summ")       <- ifelse(any(names(list(...)) == "summ"), list(...)$summ, "mean")
    results       <- c(results, list(DBS = DBSs, DBSvals = DBSvals))
    if(G > 1)      {
      x.sils      <- DBSvals[[as.character(G)]][[best.mod]]
      attr(x.dbs,  "Weighted")  <-
      attr(x.sils, "Weighted")  <- do.wts
      attr(x.dbs,  "Summ")      <-
      attr(x.sils, "Summ")      <- ifelse(any(names(list(...)) == "summ"), list(...)$summ, "mean")
      results     <- c(results, list(dbs = x.dbs, dbsvals = x.sils))
    }
  }
  if(do.asw)       {
    ASWs          <- if(any(rG  == 1)) ASWs[-1L,, drop=FALSE]            else ASWs
    class(ASWs)   <- "MEDcriterion"
    attr(ASWs, "Criterion")     <- "ASW"
    attr(ASWs, "G")             <- if(any(rG  == 1)) rownames(BICs)[-1L] else rownames(BICs)
    attr(ASWs, "modelNames")    <- colnames(BICs)
    ASWvals       <- if(any(rG  == 1)) stats::setNames(ASWvals, rG)[-1L] else stats::setNames(ASWvals, rG)
    attr(ASWs,    "Weighted")   <- 
    attr(ASWvals, "Weighted")   <- do.wts
    attr(ASWs,    "Summ")       <-
    attr(ASWvals, "Summ")       <- ifelse(any(names(list(...)) == "summ") && list(...)$summ == "median", "median", "mean")
    results       <- c(results, list(ASW = ASWs, ASWvals = ASWvals))
    if(G > 1)      {
      x.sils      <- ASWvals[[as.character(G)]][[best.mod]]
      attr(x.asw,  "Weighted")  <-
      attr(x.sils, "Weighted")  <- do.wts
      attr(x.asw,  "Summ")      <-
      attr(x.sils, "Summ")      <- ifelse(any(names(list(...)) == "summ") && list(...)$summ == "median", "median", "mean")
      results     <- c(results, list(asw = x.asw, aswvals = x.sils))
    }
  }
  results         <- c(results, list(
                          BIC     = BICs,
                          ICL     = ICLs,
                          AIC     = AICs,
                          LOGLIK  = LL.x,
                          DF      = DF.x,
                          ITERS   = IT.x))
  if(do.nec)       {
    NECs          <- NECs[-1L,, drop=FALSE]
    class(NECs)   <- "MEDcriterion"
    attr(NECs, "Criterion")     <- "NEC"
    attr(NECs, "G")             <- rownames(BICs)[-1L]
    attr(NECs, "modelNames")    <- colnames(BICs)
    results       <- c(results, list(NEC = NECs))
  }
  results         <- if(cvsel) c(results, list(CV = CV.x)) else results
  x.ll            <- x.ll[if(G > 1) switch(EXPR=algo, cemEM=-1L, -seq_len(2L)) else 1L]
  attr(x.ll, "Weighted")        <- do.wts
  results         <- c(results, list(
                          bic     = x.bic,
                          icl     = x.icl,
                          aic     = x.aic,
                          loglik  = x.ll,
                          df      = DF.x[best.ind],
                          iters   = IT.x[best.ind]))
  results         <- if(do.nec && G > 1) c(results, list(nec = x.nec)) else results
  results         <- if(cvsel)           c(results, list(cv  = x.cv))  else results
  results         <- c(results, list(
                          ZS      = stats::setNames(ZS, rG),
                          uncert  = if(G > 1) 1 - rowMaxs(x.z) else vector("integer", N2),
                          covars  = covars))
  attr(results, "Algo")         <- algo
  attr(results, "ASW")          <- do.asw
  attr(results, "Counts")       <- if(do.uni) c2         else rep(1L, N2)
  attr(results, "Criterion")    <- criterion
  attr(results, "CV")           <- cvsel
  attr(results, "DBS")          <- do.dbs
  attr(results, "DistMat")      <- HAM.mat
  attr(results, "EqualPro")     <- x.ctrl$equalPro
  attr(results, "EqualNoise")   <- x.ctrl$equalNoise
  attr(results, "Gating")       <- x.gcov | bG
  attr(results, "N")            <- N2
  attr(results, "NEC")          <- do.nec
  attr(results, "Noise")        <- noise
  attr(results, "NoiseGate")    <- x.ctrl$noise.gate
  attr(results, "T")            <- P
  attr(results, "Unique")       <- do.uni
  attr(results, "V")            <- V
  attr(results, "Weighted")     <- do.wts
  attr(results, "Weights")      <- replace(attr(seqs, "weights"), attr(seqs, "weights") == .Machine$double.eps, 0L)
  class(results)  <- "MEDseq"
    return(results)
}

#' Plot MEDseq results
#'
#' Produces a range of plots of the results of fitted \code{MEDseq} models.
#' @param x An object of class \code{"MEDseq"} generated by \code{\link{MEDseq_fit}} or an object of class \code{"MEDseqCompare"} generated by \code{\link{MEDseq_compare}}.
#' @param type A character string giving the type of plot requested:
#' \describe{
#' \item{"\code{clusters}"}{Visualise the data set with sequences grouped into their respective clusters. See \code{seriate}.}
#' \item{"\code{mean}"}{Visualise the central sequences. See \code{seriate}. The central sequence for the noise component, if any is not shown as it doesn't contribute in any way to the likelihood.}
#' \item{"\code{precision}"}{Visualise the central sequence parameters in the form of a heatmap. Values of \code{0} and \code{Inf} are shown in \code{grey} and \code{black} respectively (see \code{log.scale}).}
#' \item{"\code{gating}"}{Visualise the gating network, i.e. the observation index (by default) against the mixing proportions for that observation, coloured by cluster. See \code{seriate}. The optional argument \code{x.axis} can be passed via the \code{...} construct to change the x-axis against which mixing proportions are plotted.}
#' \item{"\code{dbs}"}{Plots all (weighted) mean/median DBS values in a fitted \code{MEDseq} object.}
#' \item{"\code{asw}"}{Plots all (weighted) mean/median ASW values in a fitted \code{MEDseq} object.}
#' \item{"\code{bic}"}{Plots all BIC values in a fitted \code{MEDseq} object.}
#' \item{"\code{icl}"}{Plots all ICL values in a fitted \code{MEDseq} object.}
#' \item{"\code{aic}"}{Plots all AIC values in a fitted \code{MEDseq} object.}
#' \item{"\code{nec}"}{Plots all NEC values in a fitted \code{MEDseq} object.}
#' \item{"\code{cv}"}{Plots all cross-validated log-likelihood values in a fitted \code{MEDseq} object.}
#' \item{"\code{LOGLIK}"}{Plots all maximal log-likelihood values in a fitted \code{MEDseq} object.}
#' \item{"\code{dbsvals}"}{Silhouette plot using observations-specific DBS values for the optimal model (coloured by cluster).}
#' \item{"\code{aswvals}"}{Silhouette plot using observations-specific ASW values for the optimal model (coloured by cluster).}
#' \item{"\code{uncert.bar}"}{Plot the observation-specific clustering uncertainties in the form of a bar plot.}
#' \item{"\code{uncert.profile}"}{Plot the observation-specific clustering uncertainties in the form of a profile plot.}
#' \item{"\code{loglik}"}{Plot the log-likelihood at every iteration of the EM/CEM algorithm used to fit the model.}
#' }
#' Also available are the following options which act as wrappers to types of plots produced by the \code{\link[TraMineR]{seqplot}} function in the \pkg{TraMineR} package.
#' \describe{
#' \item{"\code{d}"}{State distribution plots (by cluster).}
#' \item{"\code{f}"}{Sequence frequency plots (by cluster).}
#' \item{"\code{Ht}"}{Transversal entropy plots (by cluster).}
#' \item{"\code{i}"}{Selected sequence index plots (by cluster).}
#' \item{"\code{I}"}{Whole set index plots (by cluster).}
#' }
#' @param seriate Switch indicating whether seriation should be used to improve the visualisation by re-ordering the \code{"observations"} within clusters (the default), the \code{"clusters"}, \code{"both"}, or \code{"none"}. See \code{\link[seriation]{seriate}}. The options \code{"clusters"} and \code{"both"} are only invoked when \code{type} is one of "\code{clusters}", "\code{mean}", "\code{precision}", "\code{gating}", "\code{d}", "\code{f}", "\code{Ht}", "\code{i}", or "\code{I}". Additionally, the options \code{"observations"} and \code{"both"} are only invoked when \code{type} is one of "\code{clusters}" or "\code{gating}".
#' @param preczero Logical indicating whether central sequence parameter positions corresponding to zero-valued precision parameters (if any!) should also be suppressed for the non-noise components. Defaults to \code{TRUE}; noise-component means are never shown regardless of the value of \code{preczero}.
#' @param log.scale Logical indicating whether precision parameter heatmaps should be plotted on the log-scale when \code{type="precision"}. The behaviour of \code{0} or \code{Inf} values remains unchanged; only strictly-positive finite entries are effected. Heavily imbalanced values are more likely for the "\code{UU}" and "\code{UUN}" model types, thus \code{log.scale} defaults to \code{TRUE} in those instances and \code{FALSE} otherwise.
#' @param ... Catches unused arguments, and allows arguments to \code{\link{get_MEDseq_results}} to be passed when \code{type} is one of \code{"clusters"}, \code{"dbsvals"}, \code{"aswvals"}, \code{"uncert.bar"}, \code{"uncert.profile"}, \code{"d"}, \code{"f"}, \code{"Ht"}, \code{"i"}, or \code{"I"}, as well as the \code{x.axis} argument when \code{type="gating"}.
#'
#' @return The visualisation according to \code{type} of the results of a fitted \code{MEDseq} model.
#' @details The \code{type} options related to model selection criteria plot values for \emph{all} fitted models in the "\code{MEDseq}" object \code{x}. The remaining \code{type} options plot results for the optimal model, by default. However, arguments to \code{get_MEDseq_results} can be passed via the \code{...} construct to plot corresponding results for suboptimal models in \code{x} when \code{type} is one of "\code{clusters}", "\code{d}", "\code{f}", "\code{Ht}", "\code{i}", or "\code{I}".
#' @note Every \code{type} of plot respects the sampling weights, if any. Those related to \code{\link[TraMineR]{seqdef}} plots from \pkg{TraMineR} may be too wide to display in the preview panel. The same is also true when \code{type} is "\code{dbsvals}" or "\code{aswvals}".
#' @references Keefe Murphy, T. Brendan Murphy, Raffaella Piccarreta, and I. Claire Gormley (2019). Clustering longitudinal life-course sequences using mixtures of exponential-distance models. \emph{To appear}. <\href{https://arxiv.org/abs/1908.07963}{arXiv:1908.07963}>.
#' @usage 
#' \method{plot}{MEDseq}(x,
#'        type = c("clusters", "mean", "precision", "gating", 
#'                "dbs", "asw", "bic", "icl", "aic", "nec", 
#'                "cv", "LOGLIK", "dbsvals", "aswvals", 
#'                "uncert.bar", "uncert.profile", "loglik", 
#'                "d", "f", "Ht", "i", "I"), 
#'        seriate = c("observations", "both", "clusters", "none"), 
#'        preczero = TRUE,
#'        log.scale = FALSE, 
#'        ...)
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords plotting main
#' @export
#' @importFrom matrixStats "rowMaxs" "rowSums2" "weightedMedian" "weightedMean"
#' @importFrom seriation "get_order" "seriate"
#' @importFrom TraMineR "seqdef" "seqdist" "seqdplot" "seqfplot" "seqHtplot" "seqiplot" "seqIplot" "seqplot"
#' @seealso \code{\link{MEDseq_fit}}, \code{\link[TraMineR]{seqplot}}, \code{\link{dbs}}, \code{\link{get_MEDseq_results}}, \code{\link[seriation]{seriate}}
#'
#' @examples
#' \dontshow{library(TraMineR)}
#' # Load the MVAD data
#' data(mvad)
#' mvad$Location <- factor(apply(mvad[,5:9], 1L, function(x) 
#'                  which(x == "yes")), labels = colnames(mvad[,5:9]))
#' mvad          <- list(covariates = mvad[c(3:4,10:14,87)],
#'                       sequences = mvad[,15L:86L], 
#'                       weights = mvad[,2])
#' mvad.cov      <- mvad$covariates
#' states        <- c("EM", "FE", "HE", "JL", "SC", "TR")
#' labels        <- c("Employment", "Further Education", "Higher Education", 
#'                    "Joblessness", "School", "Training")
#' mvad.seq      <- seqdef(mvad$sequences, states=states, labels=labels)
#' 
#' # Fit an exponential-distance model without clustering
#' mod0          <- MEDseq_fit(mvad.seq, G=1)
#' 
#' # Show the central sequence and precision parameters
#' plot(mod0, type="mean")
#' plot(mod0, type="precision")
#' \donttest{
#' # Fit a range of unweighted mixture models without covariates
#' # Only consider models with a noise component
#' mod1          <- MEDseq_fit(mvad.seq, G=9:10, modtype=c("CCN", "CUN", "UCN", "UUN"))
#' 
#' # Plot the DBS values for all fitted models
#' plot(mod1, "dbs")
#' 
#' # Plot the clusters of the optimal model
#' plot(mod1, "clusters")
#' 
#' # Plot the clusters of the best UUN model
#' plot(mod1, "clusters", modtype="UUN")
#' 
#' # Fit a model with weights and gating covariates
#' # Drop the 1st time point which was used to define the weights
#' mvad.seq2     <- seqdef(mvad$sequences[,-1], states=states, labels=labels)
#' mod2          <- MEDseq_fit(mvad.seq2, G=10, modtype="UCN", weights=mvad$weights, 
#'                             gating=~ fmpr + gcse5eq + livboth, covars=mvad.cov)
#' 
#' # Plot the central sequences & precision parameters of this model
#' plot(mod2, "mean")
#' plot(mod2, "precision")
#' 
#' # Plot the clustering uncertainties in the form of a barplot
#' plot(mod2, "uncert.bar")
#' 
#' # Plot the observation-specific DBS values and the sequence frequencies by cluster
#' # Note that these plots may not display properly in the preview panel
#' plot(mod2, "dbsvals")
#' plot(mod2, "Ht")}
plot.MEDseq       <- function(x, type = c("clusters", "mean", "precision", "gating", "dbs", "asw", "bic", "icl", "aic", "nec", "cv", "LOGLIK", "dbsvals", "aswvals", "uncert.bar", "uncert.profile", 
                                          "loglik", "d", "f", "Ht", "i", "I"), seriate = c("observations", "both", "clusters", "none"), preczero = TRUE, log.scale = FALSE, ...) {
  x               <- if(inherits(x, "MEDseqCompare")) x$optimal else x
  if(!missing(type)           &&
     (length(type)       > 1  ||
      !is.character(type)))      stop("'type' must be a character vector of length 1",    call.=FALSE)
  if(!missing(seriate)        &&
     (length(seriate)    > 1  ||
      !is.character(seriate)))   stop("'seriate' must be a character vector of length 1", call.=FALSE)
  type            <- match.arg(type)
  seriate         <- match.arg(seriate)
  sericlus        <- is.element(seriate, c("both", "clusters"))
  seriobs         <- is.element(seriate, c("both", "observations"))
  palette         <- grDevices::palette()
  savepar         <- graphics::par(no.readonly=TRUE)
  on.exit(graphics::par(savepar))
  on.exit(grDevices::palette(palette), add=TRUE)
  G               <- x$G
  N               <- attr(x, "N")
  P               <- attr(x, "T")
  V       <- mV   <- attr(x, "V")
  noise           <- attr(x, "Noise")
  dmat            <- attr(x, "DistMat")
  weighted        <- attr(x, "Weighted")
  dat             <- x$data
  modtype         <- x$modtype
  Nseq            <- seq_len(N)
  Pseq            <- seq_len(P)
  symbols         <- c(17, 2, 16, 10, 13, 18, 15, 7)
  use.col         <- c("gray", "dodgerblue1", "red3", "slateblue", "green3", "violet", "gold", "hotpink")
  alpha.x         <- attr(dat, "alphabet")
  cpal.x          <- attr(dat, "cpal")
  label.x <- lab  <- attr(dat, "labels")
  if(is.element(type,
     c("mean", "precision")))  {
    theta         <- x$params$theta
    lambda        <- attr(theta, "lambda")
    class(theta)  <- NULL
  }
  dots            <- list(...)
  dots            <- dots[unique(names(dots))]
  has.dot         <- length(names(dots)[names(dots) != "what"]) > 0
  if((has.MAP     <- length(dots) > 0 && any(names(dots)  == "MAP")))    {
    MAP           <- dots$MAP
    if(length(MAP)      != N  ||
       !is.numeric(MAP)       ||
       MAP        != floor(MAP)) stop(paste0("'MAP' must be an integer vector of length N=", N),  call.=FALSE)
  } else MAP      <- x$MAP
  if(is.element(type,    c("clusters", "d", "f", "Ht", "i", "I"))       ||
    (isTRUE(sericlus) && (is.element(type, c("mean", "precision"))      ||
    (type == "gating" && attr(x, "Gating"))))) {
    if(!is.element(type, c("gating", "mean", "precision"))) {
      if(has.MAP)  {
        G         <- length(unique(MAP))
        noise     <- any(MAP  == 0)
      } else if(has.dot)       {
        MAP       <- do.call(get_MEDseq_results, c(list(x=x, what="MAP"), dots[!(names(dots) %in% c("x", "what"))]))
        G         <- attr(MAP, "G")
        noise     <- attr(MAP, "Noise")
      }
    }
    if(isTRUE(sericlus)       &&
       !all((unip <- unique(MAP)) == 0))       {
      meds        <- NULL
      set.seed(100)
      for(g in sort(unip[unip > 0]))           {
        srows     <- Nseq[MAP == g]
        meds      <- c(meds, srows[which.min(rowSums2(dmat[srows,srows, drop=FALSE]))])
      }
      perm        <- c(get_order(seriate(stats::as.dist(dmat[meds,meds]), method="TSP")), G)
    }
    switch(EXPR=type,
             mean= {
       theta      <- theta[perm,,  drop=FALSE]
     }, precision= {
       lambda     <- lambda[perm,, drop=FALSE]
    })
  }
  Gseq            <- seq_len(G)
  perm            <- if(isTRUE(sericlus)) perm else Gseq
  perm            <- if(isTRUE(noise))    replace(perm, G, ifelse(is.element(type, c("clusters", "gating")), 0L, "Noise")) else perm

  if(type == "clusters" || 
    (type == "gating"   && 
    attr(x,  "Gating"))) {
    glo.order     <-
    num.cl        <- NULL
    set.seed(200)
    for(g in Gseq) {
      srows       <- Nseq[MAP == perm[g]]
      num.cl      <- c(num.cl, length(srows))
      glo.order   <- c(glo.order, if(isTRUE(seriobs)) srows[get_order(seriate(stats::as.dist(dmat[srows,srows]), method="TSP"))] else srows)
    }
    cum.cl        <- cumsum(num.cl)
    gcl           <- c(0L, cum.cl)
  }
  
  switch(EXPR=type,
         clusters= {
    graphics::layout(rbind(1, 2), heights=c(0.85, 0.15), widths=1)
    OrderedStates <- data.matrix(.fac_to_num(dat))[glo.order,]
    graphics::image(x=Pseq, z=t(OrderedStates), y=Nseq, zlim=c(1L, length(cpal.x)),
                    axes=FALSE, xlab="Time", ylab="Clusters", col=cpal.x, 
                    main=switch(EXPR=seriate, none="Clusters", observations="Observations Ordered Within Clusters", clusters="Ordered Clusters", "Ordered Clusters and Observations"))
    graphics::box(lwd=2)
    graphics::axis(side=2, at=gcl[-length(gcl)] + diff(gcl)/2, labels=if(noise) replace(perm, G, "Noise") else perm, lwd=1, line=-0.5, las=2, tick=FALSE, cex.axis=0.75)
    graphics::axis(side=1, at=Pseq, labels=attr(x$data, "names"), cex.axis=0.75)
    for(g in Gseq) {
      graphics::segments(0, cum.cl[g], P + 0.5, cum.cl[g], lwd=2)
    }
    graphics::par(mar=c(1, 1, 0.5, 1) + 0.1, xpd=FALSE)
    graphics::plot.new()
    graphics::legend("bottom", fill=cpal.x, legend=label.x, ncol=ceiling(V/ifelse(V > 6, 3, 2)), cex=0.75)
    graphics::layout(1)
      invisible()
  }, mean=         {
    miss.prec     <- missing(preczero)
    if(length(preczero)  > 1  ||
       !is.logical(preczero))    stop("'preczero' must be a single logical indicator", call.=FALSE)
    lmiss         <- lambda   == 0
    if(indmiss    <- 
       any(lmiss))       {
      lab         <- c(label.x, expression(paste(lambda, " = 0")))
      mV          <- V   + 1L
      if(any(gmiss      <- apply(lmiss, 1L, all))) {
        if(G      == 1L) {       message("The single central sequence is entirely missing\n")
        } else                   message(paste0("One or more central sequences (", paste(shQuote(ifelse(noise && which(gmiss) == G, "Noise", which(gmiss))), collapse=" & "), ") are entirely missing\n"))
      }   else                   message("Discarding sequence positions corresponding to zero-valued precision parameters:\nSupply 'preczero'=FALSE to change this behaviour\n")
    }     else if(!miss.prec)    message("No missing values to discard\n")
    if(isTRUE(preczero)) {
      missind           <- which(tabulate(theta[!lmiss], nbins=V) == 0)
      theta[lmiss]      <- NA
    } else if(indmiss)   {
      missind           <- which(tabulate(theta,         nbins=V) == 0)
      theta[gmiss,]     <- NA
    }
    l.ncol        <- ceiling(mV/ifelse(mV > 6, 3, 2))
    if(vmiss      <-
       any(missind))             message(paste0("One or more sequence categories (", paste(shQuote(label.x[missind]), collapse=" & "), ") are entirely missing\n"))
    dat           <- suppressMessages(seqdef(as.data.frame(theta), states=alpha.x, labels=label.x, cpal=if(vmiss) c(cpal.x[-missind], rep(NA, length(missind))) else cpal.x))
    attr(dat, "names")  <- attr(x$data, "names")
    graphics::layout(rbind(1, 2), heights=c(0.85, 0.15), widths=1)
    seqplot(dat, type="I", with.legend=FALSE, main="Central Sequences Plot", border=NA, missing.color=graphics::par()$bg, yaxis=FALSE, cex.axis=0.75, ylab=switch(EXPR=seriate, clusters=, both="Ordered Clusters", "Clusters"), xlab="Time")
    if(G > 1) graphics::axis(2, at=seq_len(G) - 0.5, labels=as.character(perm), tick=FALSE, las=2, line=-0.5, cex.axis=0.75)
    graphics::par(mar=c(1, 1, 0.5, 1) + 0.1, xpd=FALSE)
    graphics::plot.new()
    graphics::legend("bottom", fill=if(indmiss) c(cpal.x, graphics::par()$bg) else cpal.x, legend=lab, ncol=l.ncol, cex=0.75)
    graphics::layout(1)
      invisible()
  }, precision=    {
    log.scale     <- ifelse(missing(log.scale), is.element(modtype, c("UU", "UUN")), log.scale)
    if(length(log.scale) > 1  ||
       !is.logical(log.scale))   stop("'log.scale' must be a single logical indicator", call.=FALSE)
    graphics::layout(rbind(1, 2), heights=c(0.85, 0.15))
    graphics::par(mar=c(4.1, 4.1, 4.1, 3.1))
    i.ind         <- is.infinite(lambda)
    num.ind       <- !i.ind  &  lambda >  0
    dat           <- if(log.scale) log(lambda[num.ind]) else lambda[num.ind]
    facs          <- if(length(dat) > 1) cut(dat, 30L, include.lowest=TRUE) else 1L
    cmat          <- matrix("", nrow=G, ncol=P)
    cols          <- rev(grDevices::heat.colors(30L))
    cmat[i.ind]            <- "black"
    cmat[lambda   == 0]    <- "grey65"
    cmat[num.ind]          <- cols[as.numeric(facs)]
    levels        <- sort(unique(as.vector(cmat)))
    z             <- matrix(unclass(factor(cmat, levels=levels, labels=seq_along(levels))), nrow=P, ncol=G, byrow=TRUE)
    graphics::image(Pseq, Gseq, z, col=levels, axes=FALSE, xlab="Positions", ylab=switch(EXPR=seriate, clusters=, both="Ordered Clusters", "Clusters"), main=paste0("Precision Parameters Plot", ifelse(log.scale, " (Log Scale)",  "")))
    graphics::axis(1, at=Pseq, tick=FALSE, las=1, cex.axis=0.75, labels=Pseq)
    graphics::axis(2, at=Gseq, tick=FALSE, las=1, cex.axis=0.75, labels=as.character(perm))
    graphics::box(lwd=2)
    bx            <- graphics::par("usr")
    xpd           <- graphics::par()$xpd
    box.cx        <- c(bx[2L] + (bx[2L]  - bx[1L])/1000, bx[2L] + (bx[2L] - bx[1L])/1000 + (bx[2L] - bx[1L])/50)
    box.cy        <- c(bx[3L],   bx[3L])
    box.sy        <- (bx[4L]  -  bx[3L]) / length(cols)
    xx            <- rep(box.cx, each=2L)
    graphics::par(xpd = TRUE)
    for(i in seq_along(cols)) {
      yy          <- c(box.cy[1L] + (box.sy * (i - 1L)),
                       box.cy[1L] + (box.sy * (i)),
                       box.cy[1L] + (box.sy * (i)),
                       box.cy[1L] + (box.sy * (i - 1L)))
      graphics::polygon(xx, yy, col = cols[i], border = cols[i])
    }
    graphics::par(new=TRUE)
    graphics::plot(0, 0, type="n", ylim=if(length(dat) > 1) range(dat, na.rm=TRUE) else c(0, 1), yaxt="n", ylab="", xaxt="n", xlab="", frame.plot=FALSE)
    graphics::axis(side=4, las=2, tick=FALSE, line=0.5)
    graphics::par(mar=c(0, 0, 0, 0))
    graphics::plot.new()
    graphics::legend("center", c(expression(paste(lambda, " = 0")), expression(paste(lambda %->%~infinity))), fill=c("grey65", "black"), ncol=2, text.width=0.1, cex=1.25)
    graphics::layout(1)
      invisible()
  }, gating=       {
    suppressWarnings(graphics::par(pty="m"))
    Tau        <- .mat_byrow(x$params$tau, nrow=N, ncol=ncol(x$z))
    sericlus   <- isTRUE(sericlus)   && attr(x, "Gating")
    seriobs    <- isTRUE(seriobs)    && attr(x, "Gating")
    Tau        <- if(isTRUE(sericlus))  Tau[,perm, drop=FALSE]      else Tau
    if(miss.x  <- length(dots) > 0   && any(names(dots) == "x.axis")) {
      x.axis   <- dots$x.axis
    } else         {
      Tau      <- if(isTRUE(seriobs))   Tau[glo.order,, drop=FALSE] else Tau
      x.axis   <- seq_len(N)
    }
    xlab       <- ifelse(miss.x, "", ifelse(isTRUE(seriobs), "Seriated Observations", ifelse(isTRUE(sericlus), "Observations", "Observation")))
    type       <- ifelse(miss.x, "p", "l")
    col        <- if(noise)  replace(seq_len(G), G, "grey65")       else seq_len(G)
    if(length(x.axis) != N)      stop("'x.axis' must be of length N", call.=FALSE)
    if(x.fac   <- is.factor(x.axis)) {
      xlev     <- levels(x.axis)
      x.axis   <- as.integer(x.axis)
      xaxt     <- "n"
    } else      {
      xaxt     <- ifelse(any(seriobs, sericlus), "n", "s")
    }
    graphics::matplot(x=x.axis, y=Tau, type=type, main="Gating Network", xaxt=xaxt, xlab=xlab, ylab="", col=col, pch=1)
    graphics::mtext(expression(widehat(tau)[g]), side=2, las=2, line=3)
    if(x.fac)   {
      graphics::axis(1, at=unique(x.axis), labels=xlev)
    }
    if(isTRUE(sericlus)) {
      graphics::abline(v=cum.cl)
      graphics::mtext(perm, at=gcl[-length(gcl)] + diff(gcl)/2, side=1, las=1)
      graphics::mtext("Ordered Clusters", side=1, line=1, las=1)
    }
  }, cv=,
     bic=,
     icl=,
     aic=,
     nec=,
     dbs=,
     asw=,
     LOGLIK=       {
    if(all(type   == "cv",
       !attr(x, "CV")))          stop("Cross-validated log-likelihood values cannot be plotted as cross-validation didn't take place during model fitting\n", call.=FALSE)
    dat           <- switch(EXPR=type, cv=x$CV, bic=x$BIC, icl=x$ICL, aic=x$AIC, nec=x$NEC, dbs=x$DBS, asw=x$ASW, LOGLIK=x$LOGLIK)
    if(type ==  "nec"     &&
      (!attr(x, "NEC")     || 
       is.null(dat)))            stop(paste0("NEC values cannot be plotted as ", ifelse(attr(x, "NEC"), "only 1-component models were fitted", "'do.nec' was set to FALSE")), call.=FALSE)
    if(type ==  "dbs"      &&
      (!attr(x, "DBS")     ||
       is.null(dat)))            stop(paste0("DBS values cannot be plotted as ", ifelse(attr(x, "Algo") == "CEM", "the CEM algorithm was used to fit the models", "only 1-component models were fitted")), call.=FALSE)
    if(type ==  "asw"      &&
      (!attr(x, "ASW")     ||
       is.null(dat)))            stop("ASW values cannot be plotted as only 1-component models were fitted", call.=FALSE)
    ms            <- which(c("CC", "UC", "CU", "UU", "CCN", "UCN", "CUN", "UUN") %in% colnames(dat))
    symbols       <- symbols[ms]
    use.col       <- use.col[ms]
    graphics::matplot(dat, type="b", xlab="Number of Components (G)", ylab=switch(EXPR=type, cv=, LOGLIK=, asw=, dbs="", toupper(type)), col=use.col, pch=symbols, ylim=range(as.vector(dat[!is.na(dat) & is.finite(dat)])), xaxt="n", lty=1)
    if(type == "cv")     graphics::mtext(ifelse(weighted, expression("\u2113"["cv"]^"w"), expression("\u2113"["cv"])), side=2, line=3, las=1, cex=1.5)
    if(type == "LOGLIK") graphics::mtext(paste0(ifelse(weighted, "Weighted ", ""), "Log-Likelihood"), side=2, line=3, las=3)
    if(type == "dbs")    graphics::mtext(paste0(ifelse(weighted, "Weighted ", ""), switch(EXPR=attr(dat, "Summ"), median="Median", "Mean"), " DBS"), side=2, line=3, las=3)
    if(type == "asw")    graphics::mtext(paste0(ifelse(weighted, "Weighted ", ""), switch(EXPR=attr(dat, "Summ"), median="Median", "Mean"), " ASW"), side=2, line=3, las=3)
    graphics::axis(1, at=seq_len(nrow(dat)), labels=rownames(dat))
    graphics::legend(switch(EXPR=type, nec=, dbs="topright", "bottomright"), ncol=2, cex=1, inset=0.01, legend=colnames(dat), pch=symbols, col=use.col)
      invisible()
  }, dbsvals=,
     aswvals=      {
    switch(EXPR=type,
          dbsvals= {
      if(!attr(x, "DBS")     ||
         all(type   == "dbs",
         is.null(x$DBSvals)))    stop(paste0("DBS values cannot be plotted as ", ifelse(attr(x, "Algo") == "CEM", "the CEM algorithm was used to fit the model", "only 1-component models were fitted")), call.=FALSE)
      object      <- if(has.dot) do.call(get_MEDseq_results, c(list(x=x, what="DBS"), dots[!(names(dots) %in% c("x", "what"))])) else x$dbsvals
    },    aswvals= {
      if(!attr(x, "ASW")     ||
         all(type   == "asw",
         is.null(x$ASWvals)))    stop("ASW values cannot be plotted as only 1-component models were fitted", call.=FALSE)
      object      <- if(has.dot) do.call(get_MEDseq_results, c(list(x=x, what="ASW"), dots[!(names(dots) %in% c("x", "what"))])) else x$aswvals
    })
    rownames(object)       <- as.character(Nseq)
    cl            <- object[,"cluster"]
    X             <- object[order(cl, -object[,2L]),, drop=FALSE]
    sil           <- X[,2L]
    space         <- c(0L, rev(diff(cli <- X[,"cluster"])))
    space[space   != 0]    <- 0.5
    ng            <- table(cl)
    G             <- attr(object, "G")
    col           <- rev(X[,1L])
    if(identical(palette, grDevices::palette("default"))) {
      palette     <- c("#E41A1C", "#377EB8", "#4DAF4A", "#984EA3", "#FF7F00", "#FFFF33", "#A65628", "#F781BF", "#999999", ifelse(noise, "black", "khaki"))
      grDevices::palette(if(grDevices::dev.capabilities()$semiTransparency) grDevices::adjustcolor(palette, alpha.f=0.75) else palette)
    }
    dat           <- rev(graphics::barplot(rev(sil), space=space, xlim=c(min(0L, min(sil)), 1L), 
                                           horiz=TRUE, las=1, mgp=c(2.5, 1, 0), col=col, border=NA, 
                                           cex.names=graphics::par("cex.axis"), axisnames=FALSE))
    summ          <- attr(object, "Summ")
    if(weighted)   {
      weights     <- attr(x, "Weights")[order(cl, -object[,2L])]
      switch(EXPR=summ, median=graphics::title(main=paste0("(Weighted) ", switch(EXPR=type, dbsvals="Density-based ", ""), "Silhouette Plot"), sub=paste0("(Weighted) Median ", switch(EXPR=type, dbsvals="DBS", "Silhouette"), " Width : ", round(weightedMedian(sil, weights), digits=3)), adj=0),
                          mean=graphics::title(main=paste0("(Weighted) ", switch(EXPR=type, dbsvals="Density-based ", ""), "Silhouette Plot"), sub=paste0("(Weighted) Mean ",   switch(EXPR=type, dbsvals="DBS", "Silhouette"), " Width : ", round(weightedMean(sil,   weights), digits=3)), adj=0))
    } else         {
      switch(EXPR=summ, median=graphics::title(main=paste0(switch(EXPR=type, dbsvals="Density-based ", ""), "Silhouette Plot"), sub=paste0("Median ", switch(EXPR=type, dbsvals="DBS", "Silhouette"), " Width : ", round(stats::median(sil), digits=3)), adj=0),
                          mean=graphics::title(main=paste0(switch(EXPR=type, dbsvals="Density-based ", ""), "Silhouette Plot"), sub=paste0("Mean ",   switch(EXPR=type, dbsvals="DBS", "Silhouette"), " Width : ", round(mean(sil),          digits=3)), adj=0))
    }
    graphics::mtext(paste0("n = ", N),  adj=0)
    modtype       <- attr(object, "ModelType")
    noise         <- modtype %in% c("CCN", "UCN", "CUN", "UUN")
    graphics::mtext(substitute(G~modtype~"clusters"~C[g], list(G=G, modtype=modtype)), adj=1)
    if(weighted)   {
      switch(EXPR=summ, median=graphics::mtext(expression(paste(g, " : ", n[g], " | ", med[i %in% Cg] ~ ~w[i]~s[i])), adj=1.05, line=-1.2),
                          mean=graphics::mtext(expression(paste(g, " : ", n[g], " | ", ave[i %in% Cg] ~ ~w[i]~s[i])), adj=1.05, line=-1.2))
      silcl       <- split(sil,  cli)
      meds        <- switch(EXPR=summ, median=sapply(seq_along(silcl), function(i) weightedMedian(silcl[[i]], weights[cli == i])),
                                         mean=sapply(seq_along(silcl), function(i) weightedMean(silcl[[i]],   weights[cli == i])))
    } else         {
      switch(EXPR=summ, median=graphics::mtext(expression(paste(g, " : ", n[g], " | ", med[i %in% Cg] ~ ~s[i])), adj=1.05, line=-1.2),
                          mean=graphics::mtext(expression(paste(g, " : ", n[g], " | ", ave[i %in% Cg] ~ ~s[i])), adj=1.05, line=-1.2))
      meds        <- tapply(sil, cli, switch(EXPR=summ, median=stats::median, mean=mean))
    }
    medy          <- tapply(dat, cli, stats::median)
    for(g in seq_len(G)) {
      graphics::text(1, medy[g], paste(ifelse(g == G && noise, 0L, g), ":  ", ng[g], " | ", format(meds[g], digits=1, nsmall=2)), xpd=NA, adj=0.8)
    }
      invisible()
  }, uncert.profile=,
     uncert.bar=   {
    graphics::par(pty="m", mar=c(5.1, 4.1, 4.1, 3.1))
    if(has.dot)    {
      z           <- do.call(get_MEDseq_results, c(list(x=x, what="z"), dots[!(names(dots) %in% c("x", "what"))]))
      G           <- ncol(z)
      uncX        <- 1 - rowMaxs(z)
    } else uncX   <- x$uncert
    oneG          <- 1/G
    min1G         <- 1 - oneG
    yx            <- unique(c(0, pretty(c(0, min1G))))
    yx            <- replace(yx, length(yx), min1G)
    cm            <- c("dodgerblue2", "red3", "green3")
    switch(EXPR=type,
           uncert.bar=         {
             cu   <- cm[seq_len(2L)][(uncX >= oneG) + 1L]
             cu[uncX == 0] <- NA
             graphics::plot(uncX, type="h", ylim=range(yx), col=cu, yaxt="n", ylab="", xlab="Observations", lend=1)
             graphics::lines(x=c(0, N), y=c(oneG, oneG), lty=2, col=cm[3L])
             graphics::axis(2, at=yx, labels=replace(yx, length(yx), "1 - 1/G"), las=2, xpd=TRUE)
             graphics::axis(2, at=oneG, labels="1/G", las=2, xpd=TRUE, side=4, xpd=TRUE)
           }, uncert.profile=  {
             ord  <- order(uncX, decreasing=FALSE)
             ucO  <- uncX[ord]
             graphics::plot(ucO, type="n", ylim=c(-max(uncX)/32, max(yx)), ylab="", xaxt="n", yaxt="n", xlab=paste0("Observations in order of increasing uncertainty"))
             graphics::lines(x=c(0, N), y=c(0, 0), lty=3)
             graphics::lines(ucO)
             graphics::points(ucO, pch=15, cex=0.5, col=1)
             graphics::lines(x=c(0, N), y=c(oneG, oneG), lty=2, col=cm[3L])
             graphics::axis(2, at=yx,   las=2, xpd=TRUE, labels=replace(yx, length(yx), "1 - 1/G"))
             graphics::axis(2, at=oneG, las=2, xpd=TRUE, labels="1/G", side=4)
           })
    graphics::mtext("Uncertainty", side=2, line=3)
    graphics::title(main=list(paste0("Clustering Uncertainty ", switch(EXPR=type, uncert.bar="Barplot", "Profile Plot"))))
      invisible()
  }, loglik=       {
    x             <- x$loglik
    if(all(x      != cummax(x))) warning("Log-likelihoods are not strictly increasing\n", call.=FALSE)
    graphics::plot(x, type=ifelse(length(x) == 1, "p", "l"), xlab="Iterations", ylab=paste0(ifelse(weighted, "Weighted ", ""), "Log-Likelihood"), xaxt="n")
    seqll         <- seq_along(x)
    llseq         <- pretty(seqll)
    llseq         <- if(any(llseq != floor(llseq))) seqll else llseq
    graphics::axis(1, at=llseq, labels=llseq)
      invisible()
  },               {
    MAP           <- factor(replace(MAP, MAP == 0, "Noise"), levels=perm)
    attr(dat, "Weights")      <- if(attr(x, "Weighted")) attr(dat, "Weights") else 1L
    dots          <- c(list(seqdata=dat, with.legend=FALSE, group=MAP, type=type, with.missing=FALSE, weighted=attr(x, "Weighted")), dots[!(names(dots) %in% c("G", "modtype", "noise"))])
    dots          <- if(type  == "i") dots else c(list(border=NA), dots)
    dots          <- dots[unique(names(dots))]
    if(type       != "Ht")     {
      l.ncol      <- ceiling(V/ifelse(V > 6 | G %% 2 != 0, 3, 2))
      if(G  %% 2  == 0)        {
        graphics::par(oma=c(7.5, 0, 0, 0), xpd=TRUE)
        suppressWarnings(do.call(seqplot, dots))
        graphics::par(fig=c(0, 1, 0, 1), oma=c(0.5, 0, 0, 0), mar=c(0.5, 0, 0, 0), new=TRUE)
        graphics::plot(0, 0, type='n', bty='n', xaxt='n', yaxt='n')
        graphics::legend("bottom", fill=attr(dat, "cpal"), legend=lab, ncol=l.ncol)
      } else       {
        suppressWarnings(do.call(seqplot, dots))
        graphics::par(mar=c(1, 1, 0.5, 1) + 0.1, xpd=TRUE)
        graphics::legend("bottomright", fill=attr(dat, "cpal"), legend=lab, ncol=l.ncol)
      }
    } else         {
      suppressWarnings(do.call(seqplot, dots))
    }
      invisible()
  })
}

#' @method summary MEDseq
#' @rdname MEDseq_fit
#' @usage
#' \method{summary}{MEDseq}(object,
#'         ...)
#' @export
summary.MEDseq  <- function(object, ...) {
  object        <- if(inherits(object, "MEDseqCompare")) object$optimal else object
  G             <- object$G
  attr(G, "range")       <- eval(object$call$G)
  params        <- object$params
  equalPro      <- G <= 1 || attr(object$gating, "EqualPro")
  equalN        <- attr(object$gating, "EqualNoise") && equalPro
  summ          <- list(data = deparse(object$call$seqs), N = attr(object, "N"), P = attr(object, "T"), V = attr(object, "V"), G = G, modelName = object$modtype, algo=attr(object, "Algo"), loglik = object$loglik[length(object$loglik)], 
                        df = object$df, iters = object$iters, gating = object$gating, dbs = unname(object$dbs), asw = unname(object$asw), nec = unname(object$nec), cv = unname(object$cv), bic=unname(object$bic), icl = unname(object$icl), 
                        aic = unname(object$aic), tau = params$tau, theta = params$theta, variance = params$lambda, z = object$z, equalPro = equalPro, equalNoise = equalN, classification = object$MAP, criterion = attr(object, "Criterion"), noise.gate = attr(object, "NoiseGate"))
  class(summ)   <- "summaryMEDseq"
    summ
}

#' @method summary MEDgating
#' @export
summary.MEDgating  <- function(object, ...) {
  noise            <- attr(object, "Noise")
  noise.gate       <- attr(object, "NoiseGate")
  equalpro         <- attr(object, "EqualPro")
  equalnoise       <- attr(object, "EqualNoise")
  formula          <- attr(object, "Formula")
  class(object)    <- class(object)[2L]
  summ             <- summary(object, ...)
  class(summ)      <- "summaryMEDgate"
  attr(summ, "Class")      <- class(object)
  attr(summ, "EqualPro")   <- equalpro
  attr(summ, "EqualNoise") <- equalnoise
  attr(summ, "Formula")    <- formula
  attr(summ, "Noise")      <- noise
  attr(summ, "NoiseGate")  <- noise.gate
    summ
}

#' @method print MEDcriterion
#' @export
print.MEDcriterion       <- function(x, pick = 3L, ...) {
  if(length(pick)        != 1 ||
     !is.numeric(pick))          stop("'pick' must be a single number", call.=FALSE)
  if(floor(pick)  != pick     ||
     pick          < 1)          stop("'pick' be a strictly positive integer", call.=FALSE)
  algo            <- attr(x, "algo")
  crit            <- attr(x, "Criterion")
  choice          <- .pick_MEDCrit(x, pick)
  pick            <- choice$pick
  dim1            <- attr(x, "dim")
  dim2            <- attr(x, "dimnames")
  summ            <- attr(x, "Summ")
  weighted        <- attr(x, "Weighted")
  attributes(x)   <- NULL
  attr(x, "dim")         <- dim1
  attr(x, "dimnames")    <- dim2
  cat(switch(EXPR=crit,
             NEC="Normalised Entropy Criterion (NEC):\n",
             DBS=paste0(ifelse(weighted, "(Weighted) ", ""), switch(EXPR=summ, median="Median", mean="Mean"), " Density-based Silhouette (DBS):\n"),
             ASW=paste0(ifelse(weighted, "(Weighted) ", ""), "Average Silhouette Width (ASW):\n"),
             CV=paste0("Cross-Validated ", ifelse(weighted, "(Weighted) ", ""), "Log-Likelihood (CV):\n"),
             BIC="Bayesian Information Criterion (BIC):\n",
             ICL="Integrated Completed Likelihood (ICL):\n",
             AIC="Akaike Information Criterion (AIC):\n",
              DF="Number of Estimated Parameters (Residual DF):\n",
           ITERS=paste0("Number of ", algo, " Iterations:\n"),
          loglik=paste0("Maximal ", ifelse(weighted, "(Weighted) ", ""), "Log-Likelihood:\n")))
  print(unclass(x))
  cat(paste0("\nTop ", ifelse(pick > 1, paste0(pick, " models"), "model"), " based on the ", toupper(crit), " criterion:\n"))
    print(choice$crits)
}

#' @method print MEDgating
#' @export
print.MEDgating    <- function(x, ...) {
  noise            <- attr(x, "Noise")
  equalpro         <- attr(x, "EqualPro")
  formula          <- attr(x, "Formula")
  equalNoise       <- noise && equalpro
  gateNoise        <- noise && !equalpro && formula != "~1"
  class(x)         <- class(x)[class(x) != "MEDgating"]
  print(x, ...)
  cat(paste("Formula:", formula, "\n"))
  cat(paste("Noise:",   noise,   "\n"))
  if(gateNoise)                  cat(paste("Noise Component Gating:", attr(x, "NoiseGate"), "\n"))
  cat(paste("EqualPro:", equalpro, ifelse(equalNoise, "\n", "")))
  if(equalNoise)                 cat(paste("Noise Proportion Estimated:", attr(x, "EqualNoise")))
  if(equalpro)                   message("\n\nCoefficients set to zero as this is an equal mixing proportion model")
  message("\n\nUsers are cautioned against making inferences about statistical significance from summaries of the coefficients in the gating network\n")
    invisible(x)
}

#' @method print MEDseq
#' @rdname MEDseq_fit
#' @usage
#' \method{print}{MEDseq}(x,
#'       digits = 2L,
#'       ...)
#' @export
print.MEDseq      <- function(x, digits = 2L, ...) {
  cat("Call:\t");  print(x$call)
  if(length(digits)  > 1    || !is.numeric(digits) ||
     digits       <= 0)          stop("Invalid 'digits'", call.=FALSE)
  name            <- x$modtype
  G               <- x$G
  noise           <- attr(x, "Noise")
  gating          <- attr(x$gating, "Formula")
  gate.x          <- !attr(x, "Gating")
  equalP          <- G == 1 || attr(x$gating, "EqualPro")
  equalN          <- noise  && attr(x$gating, "EqualNoise") && equalP
  crit            <- round(c(DBS = x$dbs, ASW = x$asw, NEC=x$nex, BIC = x$bic, ICL=x$icl, AIC=x$aic), digits)
  cat(paste0("\nBest Model", ifelse(length(x$BIC)  > 1, paste0(" (according to ", toupper(attr(x, "Criterion")), "): "), ": "), name, ", with ",
             G, " component",      ifelse(G > 1, "s ", " "),
             paste0("and ", ifelse(attr(x, "Weighted"), "", "no "), "weights"),
             ifelse(gate.x,        " (no covariates)\n", " (incl. gating network covariates)\n"),
             ifelse(!equalP ||
                     G == 1, "",   paste0("Equal Mixing Proportions", ifelse(equalN | G == 1 | !noise, "\n", " (with estimated noise component mixing proportion)\n"))),
             ifelse(attr(x, "CV"), paste0("CV = ", round(unname(x$cv), digits), " | "), ""),
             paste(paste0(names(crit), " = ", crit), collapse=" | "),
             ifelse(gate.x,  "",   paste0("\nGating: ", gating, "\n"))))
    invisible()
}

#' @method print MEDseqCompare
#' @rdname MEDseq_compare
#' @usage
#' \method{print}{MEDseqCompare}(x,
#'       index = seq_len(x$pick),
#'       digits = 3L,
#'       ...)
#' @export
print.MEDseqCompare    <- function(x, index=seq_len(x$pick), digits = 3L, ...) {
  index           <- if(is.logical(index)) which(index) else index
  if(length(index) < 1 || (!is.numeric(index) &&
     (any(index    < 1  | 
          index    > x$pick))))  stop("Invalid 'index'",  call.=FALSE)
  if(length(digits)     > 1 ||
     !is.numeric(digits)    ||
     digits       <= 0)          stop("Invalid 'digits'", call.=FALSE)
  crit            <- attr(x, "Crit")
  opt             <- attr(x, "Opt")
  x$bic           <- round(x$bic,    digits)
  x$icl           <- round(x$icl,    digits)
  x$aic           <- round(x$aic,    digits)
  x$loglik        <- round(x$loglik, digits)
  na.nec          <- is.na(x$nec)
  x$nec[!na.nec]  <- if(!all(na.nec)) round(x$nec[!na.nec], digits)
  nec             <- replace(x$nec, na.nec, "")
  x$nec           <- NULL
  x$nec           <- if(all(na.nec))             NULL else nec
  na.dbs          <- is.na(x$dbs)
  x$dbs[!na.dbs]  <- if(!all(na.dbs)) round(x$dbs[!na.dbs], digits)
  dbs             <- replace(x$dbs, na.dbs, "")
  x$dbs           <- NULL
  x$dbs           <- if(all(na.dbs))             NULL else dbs
  na.asw          <- is.na(x$asw)
  x$asw[!na.asw]  <- if(!all(na.asw)) round(x$asw[!na.asw], digits)
  asw             <- replace(x$asw, na.asw, "")
  x$asw           <- NULL
  x$asw           <- if(all(na.asw))             NULL else asw
  na.cvs          <- is.na(x$cv)
  x$cv[!na.cvs]   <- if(!all(na.cvs)) round(x$cv[!na.cvs],  digits)
  cvs             <- replace(x$cv, na.cvs,  "")
  x$cv            <- NULL
  x$cv            <- if(all(na.cvs))             NULL else cvs
  x               <- c(x[seq_len(which(names(x) == "loglik") - 1L)], list(nec=x$nec, cvs=x$cv, dbs=x$dbs, asw=x$asw), x[seq(from=which(names(x) == "loglik"), to=length(x), by=1L)])
  x               <- x[unique(names(x))]
  n.all           <- all(x$noise)
  x$noise         <- if(n.all)                   NULL else x$noise
  noise.gate      <- if(n.all)                   NULL else replace(x$noise.gate, is.na(x$noise.gate), "")
  x$noise.gate    <- NULL
  x$noise.gate    <- if(all(x$gating == "None")) NULL else noise.gate
  equalPro        <- if(all(is.na(x$equalPro)))  NULL else replace(x$equalPro,   is.na(x$equalPro),   "")
  x$equalPro      <- NULL
  x$equalPro      <- equalPro
  na.equalNoise   <- is.na(x$equalNoise)
  equalNoise      <- replace(x$equalNoise, na.equalNoise,    "")
  x$equalNoise    <- NULL
  x$equalNoise    <- if(all(na.equalNoise))      NULL else equalNoise
  title           <- "Comparison of Mixtures of Exponential-Distance Models with Covariates"
  cat(paste0("---------------------------------------------------------------------\n", 
             title, "\nData: ", x$data, "\nRanking Criterion: ", toupper(crit), "\nOptimal Only: ", opt,
             "\n---------------------------------------------------------------------\n\n"))
  compX           <- data.frame(do.call(cbind, x[-seq_len(3L)]))[index,, drop=FALSE]
  compX           <- compX[,!vapply(compX, function(x) all(x == ""), logical(1L)), drop=FALSE]
  compX           <- cbind(rank = rownames(compX), compX)
  rownames(compX) <- NULL
  print(compX, row.names = FALSE)
    invisible()
}

#' @method print MEDtheta
#' @export
print.MEDtheta    <- function(x, preczero = TRUE, ...) {
  lambda          <- attr(x, "lambda")
  alpha           <- attr(x, "alphabet")
  lab.x           <- attr(x, "labels")
  G               <- nrow(x)
  V               <- length(alpha)
  class(x)        <- NULL
  miss.prec       <- missing(preczero)
  if(length(preczero)  > 1  ||
     !is.logical(preczero))      stop("'preczero' must be a single logical indicator", call.=FALSE)
  if(any(lam0     <- lambda == 0)) {
    alpha         <- c(alpha, "*")
    if(any(gm0    <- apply(lam0, 1L, all))) {
      if(G        == 1L)     {   message("The single central sequence is entirely missing\n")
      } else                     message(paste0("One or more central sequences (", paste(shQuote(which(gm0)), collapse=" & "), ") are entirely missing\n"))
    }   else                     message("Discarding sequence positions corresponding to zero-valued precision parameters:\nSupply 'preczero'=FALSE to change this behaviour\n")
  }     else if(!miss.prec)      message("No missing values to discard\n")
  if(isTRUE(preczero))       {
    missind       <- which(tabulate(x[!lam0], nbins=V) == 0)
    x[lam0]       <- V + 1L
  }     else if(any(lam0))   {
    missind       <- which(tabulate(x,        nbins=V) == 0)  
    x[gm0,]       <- V + 1L
  }
  if(any(missind))               message(paste0("One or more sequence categories (", paste(shQuote(lab.x[missind]), collapse=" & "), ") are entirely missing\n"))
  theta           <- as.data.frame(lapply(as.data.frame(x), function(theta) .replace_levels(.num_to_char(theta), alpha)))
    print(theta, ...)
}

#' @method print summaryMEDseq
#' @export
print.summaryMEDseq      <- function(x, digits = 2L, ...) {
  if(length(digits)  > 1 || !is.numeric(digits) ||
     digits     <= 0)            stop("Invalid 'digits'", call.=FALSE)
  tmp           <- data.frame(log.likelihood = round(x$loglik, digits), N = x$N, P = x$P, V = x$V, df = x$df, iters = x$iters)
  tmp           <- if(is.null(x$dbs)) tmp else cbind(tmp, DBS = round(x$dbs, digits))
  tmp           <- if(is.null(x$asw)) tmp else cbind(tmp, ASW = round(x$asw, digits))
  switch(EXPR=toupper(x$criterion),
    AIC={tmp    <- cbind(tmp, AIC=round(x$aic, digits))},
    ICL={tmp    <- cbind(tmp, ICL=round(x$icl, digits))},
    NEC={tmp    <- cbind(tmp, NEC=round(x$nec, digits))},
     CV={tmp    <- cbind(tmp,  CV=round(x$cv,  digits))})
  tmp           <- if(!is.element(x$criterion, c("aic", "icl"))) cbind(tmp, BIC = round(x$bic, digits)) else tmp
  tmp           <- cbind(tmp, Algo = x$algo)
  rownames(tmp) <- NULL
  name          <- x$modelName
  G             <- x$G
  range.G       <- attr(G, "range")
  if(!is.null(range.G)  &&
     G          == min(range.G))                message("Best model occurs at the min of the number of components considered\n")
  if(!is.null(range.G)  &&
     G          == max(range.G))                message("Best model occurs at the max of the number of components considered\n")
  noise         <- is.element(name, c("CCN", "CUN", "UCN", "UUN"))
  gating        <- attr(x$gating, "Formula")
  gate.x        <- gating == "~1"
  equalP        <- x$equalPro && gate.x
  equalN        <- noise  && x$equalNoise && equalP
  zs            <- table(x$classification)
  title         <- "Mixture of Exponential-Distance Models with Covariates"
  cat(paste0("------------------------------------------------------\n", title, "\nData: ",
             x$data,"\n", "------------------------------------------------------\n\n",
             "MEDseq ", "(", name, "), with ",    paste0(G, " component", ifelse(G > 1, "s", "")),
             paste0("\nGating Network Covariates:  ", ifelse(gate.x, "None", gating)),
             ifelse(G  > 1  && gate.x,            paste0("\nEqual Mixing Proportions:   ", equalP), ""),
             paste0("\nNoise Component:            ", noise, ""),
             ifelse(G  > 1  && !gate.x && noise,  paste0("\nNoise Component Gating:     ", x$noise.gate), ""),
             ifelse(G  > 1  && noise   && equalP, paste0("\nNoise Proportion Estimated: ", !equalN, "\n\n"), "\n\n")))
  print(tmp, row.names = FALSE)
  cat("\nClustering table:")
  print(zs,  row.names = FALSE)
    invisible()
}

#' @method print summaryMEDgate
#' @export
print.summaryMEDgate  <- function(x, ...) {
  formula          <- attr(x, "Formula")
  noise            <- attr(x, "Noise")
  equalpro         <- attr(x, "EqualPro")
  equalNoise       <- noise && equalpro
  gateNoise        <- noise && !equalpro && formula != "~1"
  class(x)         <- switch(EXPR=attr(x, "Class"), glm="summary.glm", "summary.multinom")
  print(x, ...)
  cat(paste("Formula:",  formula,  "\n"))
  cat(paste("Noise:",    noise,    "\n"))
  if(gateNoise)    cat(paste("Noise Component Gating:", attr(x, "NoiseG"), "\n"))
  cat(paste("EqualPro:", equalpro, ifelse(equalNoise, "\n", "")))
  if(equalNoise)   cat(paste("Noise Proportion Estimated:", attr(x, "EqualN")))
  if(equalpro)     message("\n\nCoefficients set to zero as this is an equal mixing proportion model")
  message("\n\n\nUsers are cautioned against making inferences about statistical significance from summaries of the coefficients in the gating network\n")
    invisible(x)
}

#' Compute the mean time spent in each sequence category
#'
#' Computes the mean time (per cluster) spent in each sequence category (i.e. state value) for a fitted \code{MEDseq} model.
#' @param x An object of class \code{"MEDseq"} generated by \code{\link{MEDseq_fit}} or an object of class \code{"MEDseqCompare"} generated by \code{\link{MEDseq_compare}}.
#' @param MAP A logical indicating whether to use the MAP classification in the computation of the averages, or the 'soft' clustering assignments given by \code{x$z}. Defaults to \code{FALSE}, but is always \code{TRUE} for models fitted by the CEM algorithm (see \code{\link{MEDseq_control}}).
#' @param norm A logical indicating whether the mean times are normalised to sum to the sequence length within each cluster (defaults to \code{TRUE}). Otherwise, when \code{FALSE}, entries give the total (weighted) number of times a given sequence category was observered in a given cluster.
#' @details Models with weights, covariates, &/or a noise component are also accounted for.
#'
#' @return A matrix with sequence category and cluster-specific mean times, giving clusters on the rows, corresponding cluster sizes in the first column, and sequence categories in the remaining columns.
#' @importFrom matrixStats "colSums2"
#' @importFrom TraMineR "seqdef"
#' @export
#' @references Keefe Murphy, T. Brendan Murphy, Raffaella Piccarreta, and I. Claire Gormley (2019). Clustering longitudinal life-course sequences using mixtures of exponential-distance models. \emph{To appear}. <\href{https://arxiv.org/abs/1908.07963}{arXiv:1908.07963}>.
#' @seealso \code{\link{MEDseq_fit}}, \code{\link{MEDseq_control}}
#' @author Keefe Murphy - <\email{keefe.murphy@@ucd.ie}>
#' @keywords utility
#' @usage
#' MEDseq_meantime(x,
#'                 MAP = FALSE,
#'                 norm = TRUE)
#' @examples
#' \donttest{data(biofam)
#' 
#' mod <- MEDseq_fit(seqdef(biofam[10:25] + 1L), G=10, modtype="UUN")
#' 
#' MEDseq_meantime(mod)
#' MEDseq_meantime(mod, MAP=TRUE, norm=FALSE)}
MEDseq_meantime        <- function(x, MAP = FALSE, norm = TRUE) {
    UseMethod("MEDseq_meantime")
}

#' @method MEDseq_meantime MEDseq
#' @export
MEDseq_meantime.MEDseq <- function(x, MAP = FALSE, norm = TRUE) {
  x               <- if(inherits(x, "MEDseqCompare")) x$optimal else x
  if(length(norm)  > 1 ||
     !is.logical(norm))          stop("'norm' must be a single logical indicator", call.=FALSE)
  if(length(MAP)   > 1 ||
     !is.logical(MAP))           stop("'MAP' must be a single logical indicator",  call.=FALSE)
  MAP             <- isTRUE(MAP) && attr(x, "Algo") != "CEM"
  alph            <- attr(x$params$theta, "alphabet")
  V               <- attr(x, "V")
  P               <- attr(x, "T")
  G               <- x$G
  noise           <- attr(x, "Noise")
  gnames          <- paste0("Cluster", seq_len(G))
  gnames          <- if(isTRUE(noise)) replace(gnames, G, "Noise")   else gnames
  class           <- if(isTRUE(noise)) replace(x$MAP, x$MAP == 0, G) else x$MAP 
  tabMAP          <- if(isTRUE(MAP))   tabulate(class)               else colSums2(x$z)
  if(isTRUE(MAP))  {
    temp          <- do.call(rbind, by(x$data, class,  function(x) tabulate(do.call(base::c, x), V)))
  } else           {
    x$data        <- .fac_to_num(x$data)
    temp          <- do.call(rbind, lapply(seq_len(G), function(g) tapply(rep(x$z[,g], P), do.call(base::c, x$data), sum)))
  }
  temp            <- if(isTRUE(norm))  temp/tabMAP                   else temp
  temp            <- cbind(tabMAP, temp)
  rownames(temp)  <- gnames
  colnames(temp)  <- c("Size", alph)
    temp
}

#' Show the NEWS file
#'
#' Show the \code{NEWS} file of the \code{MEDseq} package.
#' @return The \code{MEDseq} \code{NEWS} file, provided the session is interactive.
#' @export
#' @keywords utility
#'
#' @usage MEDseq_news()
#' @examples
#' MEDseq_news()
MEDseq_news       <- function() {
  newsfile        <- file.path(system.file(package  = "MEDseq"), "NEWS.md")
    if(interactive()) file.show(newsfile) else message("The session is not interactive\n")
}
#