% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mires.R
\name{mires}
\alias{mires}
\title{Fit mixed effects measurement model for invariance assessment.}
\usage{
mires(
  formula,
  group,
  data,
  inclusion_model = c("dependent", "independent"),
  identification = c("sum_to_zero", "hierarchical"),
  save_scores = FALSE,
  prior_only = FALSE,
  prior = c(0, 0.25),
  ...
)
}
\arguments{
\item{formula}{Formula. LHS is the factor name, and RHS contains indicators.}

\item{group}{Grouping variable (symbol). Grouping variable over which to assess invariance.}

\item{data}{data.frame. Must contain the indicators specified in formula, and the grouping variable.}

\item{inclusion_model}{String (Default: dependent). If dependent, then the regularization of RE-SDs are dependent (See Details). If independent, then regularization is per-parameter. This is useful for comparing a dependent inclusion model to a non-dependent inclusion model. Note that adaptive regularization occurs regardless (until a non-regularized version is implemented).}

\item{identification}{String (Default: sum_to_zero). If \code{hierarchical}, then latent means and (log) SDs are identified as zero-centered random effects. If \code{sum_to_zero}, then latent means are identified by a sum-to-zero constraint, and (log) latent SDs are identified by a sum-to-zero constraint.}

\item{save_scores}{Logical (Default: FALSE). If TRUE, latent scores for each observation are estimated. If FALSE (Default), latent scores are marginalized out; this can result in more efficient sampling and faster fits, due to the drastic reduction in estimated parameters. Note that the random effects for each group are always estimated, and are not marginalized out.}

\item{prior_only}{Logical (Default: FALSE). If TRUE, samples are drawn from the prior.}

\item{prior}{Numeric vector (Default: c(0, .25)). The location and scale parameters for the hierarchical inclusion model.}

\item{...}{Further arguments to \code{\link[rstan]{sampling}}.}
}
\value{
mires object.
}
\description{
Fits mixed effects measurement models for measurement invariance assessment.
}
\details{
MIRES stands for Measurement Invariance assessment with Random Effects and Shrinkage.
Unlike other measurement invariance approaches, the MIRES model assumes all measurement model parameters (loadings, residual SDs, and intercepts) can randomly vary across groups --- It is a mixed effects model on all parameters.
Unlike most mixed effects models, the random effect variances are themselves also hierarchically modeled from a half-normal distribution with location zero, and a scaling parameter.
This scaling parameter allows for rapid shrinkage of variance toward zero (invariance), while allowing variance if deemed necessary (non-invariance).

The scaling parameter (an estimated quantity) controls whether the RE variance is effectively zero (invariant) or not (non-invariant).
Therefore, the random effect variances are regularized.
When \code{inclusion_model} is \code{dependent} (Default), the scaling parameters are hierarchically modeled.
By doing so, the invariance or non-invariance of a parameter is informed by other parameters with shared characteristics.
Currently, we assume that each parameter informs the invariance of other similar parameters (presence of variance in some loadings increases the probability of variance in other loadings), and of similar items (non-invariance of item j parameters informs other parameters for item j).
This approach increases the information available about presence or absence of invariance, allowing for more certain decisions.

The "Hierarchical inclusion model" on the random effect variance manifests as a hierarchical prior.
When a dependent inclusion model is specified, then the hierarchical prior on random effect SDs is:
\deqn{p(\sigma_p | \exp(\tau)) = \mathcal{N}^+(\sigma_p | 0, \exp(\tau))}
\deqn{\tau = \tau_c + \tau_{param} + \tau_{item} + \tau_p}
\deqn{\tau_* \sim \mathcal{N}(\mu_h, \sigma_h)}
Therefore, the regularization of each RE-SD is shared between all RE-SDs (tau_c), all RE-SDs of the same parameter type (tau_param), and all RE-SDs of the same item (tau_item).

When an independent inclusion model is specified (\code{inclusion_model} is "independent"), only the independent regularization term \eqn{\tau_p} is included.
The prior is then scaled so that the marginal prior on each \eqn{p(\sigma_p)} remains the same.
In this case, RE-SDs cannot share regularization intensities between one another.

The inclusion model hyper parameters (mu_h, sigma_h) can be specified, but we recommend the default as a relatively sane, but weakly informative prior.
}
\examples{
\donttest{
data(sim_loadings) # Load simulated data set
head(sim_loadings) # 8 indicators, grouping variable is called "group"

# Fit MIRES to simulated data example.
# Assume factor name is, e.g., agreeableness.
fit <- mires(agreeableness ~ x_1 + x_2 + x_3 + x_4 + x_5 + x_6 + x_7 + x_8,
             group = group,
             data = sim_loadings, chains = 2, cores = 2)

# Summarize fit
summary(fit)

# Compare all groups' loadings:
pairwise(fit, param = "lambda")
# Compare groups "2" and "3" only:
pairwise(fit, param = "lambda", groups = c("2", "3"))

# Get random effects:
fit_ranefs <- ranef(fit)
# Look at random effects of loadings:
fit_ranefs$lambda
}
}
\references{
Stan Development Team (2020). RStan: the R interface to Stan. R package version 2.21.1. https://mc-stan.org

Martin, S. R., Williams, D. R., & Rast, P. (2019, June 18). Measurement Invariance Assessment with Bayesian Hierarchical Inclusion Modeling. <doi:10.31234/osf.io/qbdjt>
}
\author{
Stephen R. Martin
}
