#' Function to generate a unit space for the Recognition-Taguchi (RT) method
#'
#' \code{RT} generates a unit space for the Recognition-Taguchi (RT) method. In
#'   \code{\link{general_MT}}, the inversed correlation matrix is used for A and
#'   the data are transformed by the function to be generated by
#'   \code{\link{generates_dimensionality_reduction_function}} based on
#'   \code{unit_space_data}. In the transformation, the p variables in
#'   \code{unit_space_data} are reduced into 2 synthetic variables.
#'
#' @param unit_space_data Matrix with n rows (samples) and p columns (variables).
#'                          Data to generate the unit space. All data should be
#'                          continuous values and should not have missing values.
#' @param includes_transformed_data If \code{TRUE}, then the transformed data
#'                                    are included in a return object.
#' @param ... Passed to \code{\link[base]{solve}} for computing the inverse of
#'              the correlation matrix.
#'
#' @return \code{RT} returns an object of S3 \link[base]{class} "RT". An
#'           object of class "RT" is a list containing the following components:
#'
#'  \item{A}{2 x 2 matrix. Inversed correlation matrix of the transformed
#'            \code{unit_space_data}.}
#'  \item{calc_A}{\code{function(x) solve(cor(x), ...)}.}
#'  \item{transforms_data}{Function to be generated from
#'                          \code{\link{generates_dimensionality_reduction_function}}
#'                          based on \code{unit_space_data}.}
#'  \item{distance}{Vector with length n. Distances from the unit space to each
#'                   sample.}
#'  \item{n}{The number of samples.}
#'  \item{q}{The number of variables after the data transformation. q is always
#'            2.}
#'  \item{x}{If \code{includes_transformed_data} is \code{TRUE}, then the
#'            transformed data are included.}
#'
#' @references
#'   Taguchi, G. (2006). Objective Function and Generic Function (11).
#'     \emph{Journal of Quality Engineering Society, 14}(2), 5-9. (In Japanese)
#'
#'   Huda, F., Kajiwara, I., Hosoya, N., & Kawamura, S. (2013). Bolt loosening
#'    analysis and diagnosis by non-contact laser excitation vibration tests.
#'    \emph{Mechanical systems and signal processing, 40}(2), 589-604.
#'
#' @seealso \code{\link[base]{solve}}, \code{\link{general_MT}},
#'            \code{\link{generates_dimensionality_reduction_function}}, and
#'            \code{\link{diagnosis.MT}}
#'
#' @examples
#' # 40 data for versicolor in the iris dataset
#' iris_versicolor <- iris[61:100, -5]
#'
#' unit_space_RT <- RT(unit_space_data = iris_versicolor,
#'                     includes_transformed_data = TRUE)
#'
#' # The following "tol" is a parameter passed to the solve function.
#' unit_space_RT <- RT(unit_space_data = iris_versicolor,
#'                     includes_transformed_data = TRUE,
#'                     tol = 1e-9)
#'
#' (unit_space_RT$distance)
#'
#' @importFrom stats cor
#' @export
RT <- function(unit_space_data, includes_transformed_data = FALSE, ...) {

  object_RT <- general_MT(unit_space_data = unit_space_data,
                          calc_A = function(x) solve(cor(x), ...),
                          generates_transform_function =
                                    generates_dimensionality_reduction_function,
                          includes_transformed_data = includes_transformed_data)

  class(object_RT) <- "RT"

  return(object_RT)

}

#' Diagnosis method for the Recognition-Taguchi (RT) method
#'
#' \code{diagnosis.RT} (via \code{\link{diagnosis}}) calculates the
#'   distance based on the unit space generated by \code{\link{RT}} or
#'   \code{\link{generates_unit_space}}(..., method = "RT") and classifies each
#'   sample into positive (\code{TRUE}) or negative (\code{FALSE}) by comparing
#'   the values with the set threshold value.
#'
#' @param unit_space Object of class "RT" generated by \code{\link{RT}} or
#'                     \code{\link{generates_unit_space}}(..., method = "RT").
#' @param newdata Matrix with n rows (samples) and p columns (variables). The
#'                  data are used to calculate the desired distances from the
#'                  unit space. All data should be continuous values and should
#'                  not have missing values.
#' @param threshold Numeric specifying the threshold value to classify each
#'                    sample into positive (\code{TRUE}) or negative (\code{FALSE}).
#' @param includes_transformed_newdata If \code{TRUE}, then the transformed data
#'                                       for \code{newdata} are included in a
#'                                       return object.
#'
#' @return \code{diagnosis.RT} (via \code{\link{diagnosis}}) returns a list
#'           containing the following components:
#'
#'  \item{distance}{Vector with length n. Distances from the unit space to each
#'                   sample.}
#'  \item{le_threshold}{Vector with length n. Logical values indicating the
#'                       distance of each sample is less than or equal to the
#'                       threhold value (\code{TRUE}) or not (\code{FALSE}).}
#'  \item{threshold}{Numeric value to classify the sample into positive or
#'                    negative.}
#'  \item{unit_space}{Object of class "RT" passed by \code{unit_space}.}
#'  \item{n}{The number of samples for \code{newdata}.}
#'  \item{q}{The number of variables after the data transformation. q is always
#'            2.}
#'  \item{x}{If \code{includes_transformed_newdata} is \code{TRUE}, then the
#'            transformed data for \code{newdata} are included.}
#'
#' @references
#'   Taguchi, G. (2006). Objective Function and Generic Function (11).
#'     \emph{Journal of Quality Engineering Society, 14}(2), 5-9. (In Japanese)
#'
#'   Huda, F., Kajiwara, I., Hosoya, N., & Kawamura, S. (2013). Bolt loosening
#'    analysis and diagnosis by non-contact laser excitation vibration tests.
#'    \emph{Mechanical systems and signal processing, 40}(2), 589-604.
#'
#' @seealso \code{\link{general_diagnosis.MT}} and \code{\link{RT}}
#'
#' @examples
#' # 40 data for versicolor in the iris dataset
#' iris_versicolor <- iris[61:100, -5]
#'
#' unit_space_RT <- RT(unit_space_data = iris_versicolor,
#'                     includes_transformed_data = TRUE)
#'
#' # 10 data for each kind (setosa, versicolor, virginica) in the iris dataset
#' iris_test <- iris[c(1:10, 51:60, 101:111), -5]
#'
#' diagnosis_RT <- diagnosis(unit_space = unit_space_RT,
#'                           newdata = iris_test,
#'                           threshold = 0.2,
#'                           includes_transformed_newdata = TRUE)
#'
#' (diagnosis_RT$distance)
#' (diagnosis_RT$le_threshold)
#'
#' @export
diagnosis.RT <- function(unit_space,
                       newdata,
                       threshold,
                       includes_transformed_newdata = FALSE) {

  if (!inherits(unit_space, "RT")) {
    warning("calling diagnosis.RT(<fake-RT-object>) ...")
  }

  general_diagnosis.MT(unit_space = unit_space,
                     newdata = newdata,
                     threshold = threshold,
                     includes_transformed_newdata = includes_transformed_newdata)

}
