/* 
 * PDF Estimator:  A non-parametric probability density estimation tool based on maximum entropy
 * File:   Score.cpp
 * Copyright (C) 2018
 * Jenny Farmer jfarmer6@uncc.edu
 * Donald Jacobs djacobs1@uncc.edu
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published 
 * by the Free Software Foundation, either version 3 of the License, or (at your option) any later version. This program is distributed in 
 * the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with 
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sstream>

#include "Score.h"
#include "WriteResults.h"

Score::Score(string scoreFile, double confidenceTarget, double confidenceMin, double confidenceMax, bool boundaryPenalty) {     
    this->scoreFile = scoreFile;
    this->penalty = boundaryPenalty;
//    readFile();    
    getValues();
    targetScore = getTargetScore(confidenceTarget);
    minimumScore = getTargetScore(confidenceMin);
    maximumScore = getTargetScore(confidenceMax);
}

Score::Score(const Score& orig) {
}

Score::~Score() {
}

bool Score::readFile() {
    
    fstream fin;
    string line;
        
    fin.open(scoreFile.c_str());
    

    if(!fin.is_open()){
#ifndef R
        cout << "Failed to open score file " <<  scoreFile.c_str() << "\n";\

#endif
        return false;
    }
    istringstream iss;
    while (getline(fin, line)) {
        double columnOne;
        double columnTwo;
        iss.str(line);
        iss >> columnOne >> columnTwo;
        scores.push_back(columnOne);
        SURDs.push_back(columnTwo);
    }    
    return true;
}

 double Score::calculateScore(double r[], int N, int p){
    likelihood = 0;
    penaltyScore = 0;
    for (int i = 0; i < p; i++) {
        double powers = 0;
        double epsilon = .000000000000001;
        if (r[i] <= 0) 
            powers = (indices[i])*log(epsilon) + (N-indices[i]-1)*log(1.0 - r[i]);
        else if (r[i] >= 1) 
            powers = (indices[i])*log(r[i]) + (N-indices[i]-1)*log(epsilon);
        else powers = (indices[i])*log(r[i]) + (N-indices[i]-1)*log(1.0 - r[i]);
        double Ls = powers + factorials[i];
        likelihood+= Ls;
        if (penalty) {
            if (i < nEndpoints) {            
                penaltyScore += fabs(Ls - uniformL[i]);
            }
            if (i > (p - nEndpoints - 1)) {            
                penaltyScore += fabs(Ls - uniformL[p-i-1]);
            }
        }
    } 
    likelihood /= p;
    likelihood -= 0.5*log(N);
    if (penalty) {
        penaltyScore /= (nEndpoints);
        penaltyScore = log(1 + penaltyScore*0.1);
    }
      
    return likelihood - penaltyScore;
}

double Score::getTargetScore(double SURD) {
 
    vector<double>::iterator it;
    it = lower_bound (SURDs.begin(), SURDs.end(), SURD/100);
    int index = it - SURDs.begin();
    
    if (index == SURDs.size()) {
        return scores[index - 1];
    } 
    if (index == 0) {
        return scores[0];
    } 
    
    double E1 = scores[index - 1];
    double E2 = scores[index];
    double P1 = SURDs[index - 1];
    double P2 = SURDs[index];
    double E = E1 + (SURD/100 - P1)*(E2 - E1)/(P2 - P1);
    return E;  
    
}

double Score::getConfidence(double score) {
 
    vector<double>::iterator it;
    it = lower_bound (scores.begin(), scores.end(), score);
    int index = it - scores.begin();
    
    if (index == scores.size()) {
        return SURDs[index - 1];
    } 
    if (index == 0) {
        return SURDs[0];
    } 
    
    double E1 = scores[index - 1];
    double E2 = scores[index];
    double P1 = SURDs[index - 1];
    double P2 = SURDs[index];
    double P = P1 + (score - E1)*(P2 - P1)/(E2 - E1);
    return P * 100;  
    
}

void Score::setFactorials (int N, int p) {
    factorials = new double[p];
    indices = new int[p];
    vector <int>  test = Partition::getIndices(N, p);
    for (int i = 0; i < p; i++) {
        indices[i] = test[i];
        factorials[i] = (smartFactorial(N)) - (smartFactorial(indices[i])) - (smartFactorial(N-indices[i]-1));          
    }   
    
    if (penalty) {
        double endPoints = p*0.005;
        nEndpoints = (int) endPoints; 
        if (nEndpoints == 0) nEndpoints++;
        uniformL = new double[nEndpoints*2];
            
        for (int i = 0; i < nEndpoints; i++) {
            double r = (indices[i] + 1.0)/(N + 1.0);
            uniformL[i] = (indices[i])*log(r) + (N-indices[i]-1)*log(1.0 - r) + factorials[i];
            uniformL[2*nEndpoints-i-1] = uniformL[i];
        }
    } else {
        nEndpoints = 1;
    }    
}

double Score::smartFactorial (int x) {
    if (x < 10)
        return log(factorial(x));
    else
        return stirlingApproxLn(x);
}     
    
double Score::factorial (int n) {
    double product = 1;
    for (int i=2; i<=n; i++) {
        product *= i;
    }
    return product;
}
    
double Score::stirlingApproxLn(double x) {        
    if (x==0) return 1;
    return x*log(x)-x + log(sqrt(2*M_PI*x));
}
         
void Score::getValues() {
    double arrayScore[] = {-5.9923,-5.9828,-5.9734,-5.9639,-5.9544,-5.945,-5.9355,-5.926,-5.9165,-5.9071,-5.8976,-5.8881,-5.8786,-5.8692,-5.8597,-5.8502,-5.8408,-5.8313,-5.8218,-5.8123,-5.8029,-5.7934,-5.7839,-5.7745,-5.765,-5.7555,-5.746,-5.7366,-5.7271,-5.7176,-5.7082,-5.6987,-5.6892,-5.6797,-5.6703,-5.6608,-5.6513,-5.6419,-5.6324,-5.6229,-5.6134,-5.604,-5.5945,-5.585,-5.5756,-5.5661,-5.5566,-5.5471,-5.5377,-5.5282,-5.5187,-5.5093,-5.4998,-5.4903,-5.4808,-5.4714,-5.4619,-5.4524,-5.443,-5.4335,-5.424,-5.4145,-5.4051,-5.3956,-5.3861,-5.3767,-5.3672,-5.3577,-5.3482,-5.3388,-5.3293,-5.3198,-5.3103,-5.3009,-5.2914,-5.2819,-5.2725,-5.263,-5.2535,-5.244,-5.2346,-5.2251,-5.2156,-5.2062,-5.1967,-5.1872,-5.1777,-5.1683,-5.1588,-5.1493,-5.1399,-5.1304,-5.1209,-5.1114,-5.102,-5.0925,-5.083,-5.0736,-5.0641,-5.0546,-5.0451,-5.0357,-5.0262,-5.0167,-5.0073,-4.9978,-4.9883,-4.9788,-4.9694,-4.9599,-4.9504,-4.941,-4.9315,-4.922,-4.9125,-4.9031,-4.8936,-4.8841,-4.8747,-4.8652,-4.8557,-4.8462,-4.8368,-4.8273,-4.8178,-4.8084,-4.7989,-4.7894,-4.7799,-4.7705,-4.761,-4.7515,-4.742,-4.7326,-4.7231,-4.7136,-4.7042,-4.6947,-4.6852,-4.6757,-4.6663,-4.6568,-4.6473,-4.6379,-4.6284,-4.6189,-4.6094,-4.6,-4.5905,-4.581,-4.5716,-4.5621,-4.5526,-4.5431,-4.5337,-4.5242,-4.5147,-4.5053,-4.4958,-4.4863,-4.4768,-4.4674,-4.4579,-4.4484,-4.439,-4.4295,-4.42,-4.4105,-4.4011,-4.3916,-4.3821,-4.3727,-4.3632,-4.3537,-4.3442,-4.3348,-4.3253,-4.3158,-4.3064,-4.2969,-4.2874,-4.2779,-4.2685,-4.259,-4.2495,-4.2401,-4.2306,-4.2211,-4.2116,-4.2022,-4.1927,-4.1832,-4.1738,-4.1643,-4.1548,-4.1453,-4.1359,-4.1264,-4.1169,-4.1074,-4.098,-4.0885,-4.079,-4.0696,-4.0601,-4.0506,-4.0411,-4.0317,-4.0222,-4.0127,-4.0033,-3.9938,-3.9843,-3.9748,-3.9654,-3.9559,-3.9464,-3.937,-3.9275,-3.918,-3.9085,-3.8991,-3.8896,-3.8801,-3.8707,-3.8612,-3.8517,-3.8422,-3.8328,-3.8233,-3.8138,-3.8044,-3.7949,-3.7854,-3.7759,-3.7665,-3.757,-3.7475,-3.7381,-3.7286,-3.7191,-3.7096,-3.7002,-3.6907,-3.6812,-3.6718,-3.6623,-3.6528,-3.6433,-3.6339,-3.6244,-3.6149,-3.6055,-3.596,-3.5865,-3.577,-3.5676,-3.5581,-3.5486,-3.5391,-3.5297,-3.5202,-3.5107,-3.5013,-3.4918,-3.4823,-3.4728,-3.4634,-3.4539,-3.4444,-3.435,-3.4255,-3.416,-3.4065,-3.3971,-3.3876,-3.3781,-3.3687,-3.3592,-3.3497,-3.3402,-3.3308,-3.3213,-3.3118,-3.3024,-3.2929,-3.2834,-3.2739,-3.2645,-3.255,-3.2455,-3.2361,-3.2266,-3.2171,-3.2076,-3.1982,-3.1887,-3.1792,-3.1698,-3.1603,-3.1508,-3.1413,-3.1319,-3.1224,-3.1129,-3.1035,-3.094,-3.0845,-3.075,-3.0656,-3.0561,-3.0466,-3.0372,-3.0277,-3.0182,-3.0087,-2.9993,-2.9898,-2.9803,-2.9708,-2.9614,-2.9519,-2.9424,-2.933,-2.9235,-2.914,-2.9045,-2.8951,-2.8856,-2.8761,-2.8667,-2.8572,-2.8477,-2.8382,-2.8288,-2.8193,-2.8098,-2.8004,-2.7909,-2.7814,-2.7719,-2.7625,-2.753,-2.7435,-2.7341,-2.7246,-2.7151,-2.7056,-2.6962,-2.6867,-2.6772,-2.6678,-2.6583,-2.6488,-2.6393,-2.6299,-2.6204,-2.6109,-2.6015,-2.592,-2.5825,-2.573,-2.5636,-2.5541,-2.5446,-2.5352,-2.5257,-2.5162,-2.5067,-2.4973,-2.4878,-2.4783,-2.4689,-2.4594,-2.4499,-2.4404,-2.431,-2.4215,-2.412,-2.4026,-2.3931,-2.3836,-2.3741,-2.3647,-2.3552,-2.3457,-2.3362,-2.3268,-2.3173,-2.3078,-2.2984,-2.2889,-2.2794,-2.2699,-2.2605,-2.251,-2.2415,-2.2323,-2.223,-2.2137,-2.2044,-2.1951,-2.1858,-2.1766,-2.1673,-2.158,-2.1487,-2.1394,-2.1302,-2.1209,-2.1116,-2.1023,-2.093,-2.0838,-2.0745,-2.0652,-2.0559,-2.0466,-2.0373,-2.0281,-2.0188,-2.0095,-2.0002,-1.9909,-1.9817,-1.9724,-1.9631,-1.9538,-1.9445,-1.9353,-1.926,-1.9166,-1.9072,-1.8978,-1.8885,-1.8791,-1.8697,-1.8603,-1.8509,-1.8416,-1.8322,-1.8228,-1.8134,-1.8041,-1.7947,-1.7853,-1.7759,-1.7665,-1.7572,-1.7478,-1.7384,-1.729,-1.7197,-1.7103,-1.7009,-1.6916,-1.6824,-1.6731,-1.6638,-1.6545,-1.6453,-1.636,-1.6267,-1.6174,-1.6082,-1.5989,-1.5896,-1.5803,-1.5711,-1.5618,-1.5525,-1.5432,-1.5339,-1.5247,-1.5154,-1.5061,-1.4968,-1.4876,-1.4784,-1.4692,-1.4601,-1.4509,-1.4418,-1.4326,-1.4235,-1.4143,-1.4049,-1.3956,-1.3862,-1.3768,-1.3674,-1.358,-1.3498,-1.3416,-1.3334,-1.3252,-1.317,-1.3088,-1.2998,-1.2909,-1.282,-1.2731,-1.2641,-1.2552,-1.2466,-1.238,-1.2293,-1.2207,-1.2121,-1.2035,-1.1941,-1.1848,-1.1754,-1.1661,-1.1575,-1.1489,-1.1403,-1.1317,-1.1238,-1.1159,-1.108,-1.1001,-1.0922,-1.0843,-1.0768,-1.0694,-1.062,-1.0546,-1.0465,-1.0385,-1.0304,-1.0223,-1.013,-1.0038,-0.99447,-0.98518,-0.97622,-0.96726,-0.95829,-0.9508,-0.9433,-0.9358,-0.92814,-0.92048,-0.91281,-0.90585,-0.89888,-0.89191,-0.88511,-0.87832,-0.87152,-0.86309,-0.85466,-0.8474,-0.84014,-0.83354,-0.82693,-0.82035,-0.81377,-0.80637,-0.79898,-0.79251,-0.78604,-0.77862,-0.77119,-0.76265,-0.75411,-0.74797,-0.74183,-0.7349,-0.72797,-0.72243,-0.71689,-0.71064,-0.70439,-0.69828,-0.69217,-0.68698,-0.68179,-0.67515,-0.66601,-0.65989,-0.65376,-0.64873,-0.6437,-0.63761,-0.63043,-0.62345,-0.61524,-0.60774,-0.60263,-0.59751,-0.59136,-0.58346,-0.57483,-0.56738,-0.55968,-0.5538,-0.54796,-0.54195,-0.53386,-0.52662,-0.51975,-0.51333,-0.50759,-0.5023,-0.49637,-0.49016,-0.48529,-0.4797,-0.4748,-0.46895,-0.46445,-0.45913,-0.45428,-0.45033,-0.44606,-0.44124,-0.43665,-0.43259,-0.42855,-0.42267,-0.41818,-0.41434,-0.41065,-0.40587,-0.40161,-0.39714,-0.39199,-0.38832,-0.3833,-0.38022,-0.37656,-0.37269,-0.36906,-0.36482,-0.36101,-0.35624,-0.35227,-0.34943,-0.3462,-0.34137,-0.33787,-0.33332,-0.32971,-0.32688,-0.32411,-0.3199,-0.31627,-0.31323,-0.31007,-0.30684,-0.30444,-0.3011,-0.2984,-0.29498,-0.29141,-0.28831,-0.28478,-0.2818,-0.27943,-0.27719,-0.27413,-0.27125,-0.26838,-0.26598,-0.26317,-0.25965,-0.25724,-0.25446,-0.25141,-0.24851,-0.24547,-0.24258,-0.24031,-0.23843,-0.23567,-0.23318,-0.23008,-0.22764,-0.22496,-0.22233,-0.21988,-0.21746,-0.21582,-0.21318,-0.21074,-0.20839,-0.20564,-0.20328,-0.2008,-0.19865,-0.19612,-0.19406,-0.19161,-0.18964,-0.1877,-0.18541,-0.18332,-0.18124,-0.17929,-0.17731,-0.17512,-0.17285,-0.17105,-0.16823,-0.16583,-0.1634,-0.16124,-0.1593,-0.15646,-0.15483,-0.15271,-0.15029,-0.14755,-0.14554,-0.14257,-0.13937,-0.13733,-0.13484,-0.13186,-0.12986,-0.12768,-0.12514,-0.12232,-0.12003,-0.11739,-0.1148,-0.11255,-0.1104,-0.1082,-0.10562,-0.10242,-0.10017,-0.096587,-0.094255,-0.092045,-0.089238,-0.08673,-0.084672,-0.082247,-0.079485,-0.076583,-0.073872,-0.070776,-0.068161,-0.065287,-0.062474,-0.059109,-0.056353,-0.053264,-0.049248,-0.044153,-0.039359,-0.035022,-0.030188,-0.024219,-0.016631,-0.0098119,-0.002993,0.0057146,0.014422,0.02313,0.031837};
    double arraySurd[] = {4.5707e-06,9.3805e-06,1.4486e-05,1.9944e-05,2.5815e-05,3.2162e-05,3.905e-05,4.6548e-05,5.4728e-05,6.3664e-05,7.3431e-05,8.4105e-05,9.5765e-05,0.00010849,0.00012234,0.0001374,0.00015373,0.00017139,0.00019042,0.00021086,0.00023275,0.00025609,0.0002809,0.00030714,0.00033481,0.00036384,0.0003942,0.0004258,0.00045856,0.00049239,0.00052718,0.00056281,0.00059917,0.00063613,0.00067356,0.00071133,0.00074931,0.00078738,0.00082542,0.00086331,0.00090095,0.00093822,0.00097505,0.0010113,0.001047,0.0010821,0.0011164,0.0011499,0.0011826,0.0012145,0.0012456,0.0012757,0.0013051,0.0013335,0.0013611,0.0013878,0.0014136,0.0014387,0.0014629,0.0014864,0.0015091,0.001531,0.0015523,0.0015729,0.0015928,0.0016122,0.0016309,0.0016491,0.0016667,0.0016839,0.0017005,0.0017167,0.0017325,0.0017479,0.0017629,0.0017776,0.0017919,0.001806,0.0018197,0.0018332,0.0018464,0.0018594,0.0018722,0.0018847,0.0018971,0.0019094,0.0019215,0.0019334,0.0019452,0.0019569,0.0019685,0.00198,0.0019914,0.0020028,0.0020141,0.0020253,0.0020365,0.0020476,0.0020587,0.0020698,0.0020808,0.0020918,0.0021029,0.0021138,0.0021248,0.0021358,0.0021468,0.0021578,0.0021687,0.0021797,0.0021907,0.0022016,0.0022126,0.0022236,0.0022345,0.0022455,0.0022564,0.0022673,0.0022782,0.0022891,0.0022999,0.0023107,0.0023215,0.0023323,0.0023429,0.0023536,0.0023642,0.0023747,0.0023851,0.0023955,0.0024057,0.0024159,0.002426,0.002436,0.0024459,0.0024557,0.0024653,0.0024749,0.0024843,0.0024935,0.0025027,0.0025116,0.0025205,0.0025292,0.0025377,0.0025461,0.0025543,0.0025624,0.0025703,0.0025781,0.0025857,0.0025931,0.0026004,0.0026075,0.0026144,0.0026212,0.0026278,0.0026343,0.0026406,0.0026468,0.0026528,0.0026586,0.0026643,0.0026699,0.0026753,0.0026806,0.0026858,0.0026909,0.0026958,0.0027006,0.0027052,0.0027098,0.0027143,0.0027186,0.0027229,0.002727,0.0027311,0.0027351,0.002739,0.0027428,0.0027465,0.0027502,0.0027538,0.0027573,0.0027607,0.0027642,0.0027675,0.0027708,0.0027741,0.0027773,0.0027805,0.0027836,0.0027867,0.0027898,0.0027929,0.0027959,0.002799,0.002802,0.002805,0.002808,0.002811,0.002814,0.002817,0.00282,0.0028231,0.0028261,0.0028292,0.0028323,0.0028354,0.0028386,0.0028418,0.0028451,0.0028484,0.0028517,0.0028552,0.0028587,0.0028622,0.0028659,0.0028696,0.0028734,0.0028774,0.0028814,0.0028856,0.0028899,0.0028943,0.0028989,0.0029036,0.0029085,0.0029136,0.0029189,0.0029244,0.0029301,0.002936,0.0029422,0.0029487,0.0029554,0.0029625,0.0029699,0.0029776,0.0029857,0.0029942,0.0030031,0.0030125,0.0030224,0.0030327,0.0030436,0.0030551,0.0030672,0.00308,0.0030934,0.0031076,0.0031225,0.0031383,0.0031549,0.0031725,0.003191,0.0032106,0.0032312,0.003253,0.003276,0.0033003,0.003326,0.003353,0.0033815,0.0034116,0.0034433,0.0034767,0.0035119,0.003549,0.0035879,0.0036289,0.003672,0.0037172,0.0037647,0.0038145,0.0038667,0.0039213,0.0039785,0.0040382,0.0041006,0.0041658,0.0042337,0.0043044,0.004378,0.0044544,0.0045338,0.0046162,0.0047014,0.0047897,0.0048809,0.004975,0.005072,0.005172,0.0052748,0.0053804,0.0054887,0.0055996,0.0057132,0.0058293,0.0059477,0.0060684,0.0061913,0.0063163,0.0064431,0.0065717,0.006702,0.0068337,0.0069667,0.0071009,0.0072361,0.007372,0.0075087,0.0076458,0.0077833,0.0079208,0.0080584,0.0081958,0.0083328,0.0084694,0.0086053,0.0087404,0.0088745,0.0090076,0.0091394,0.00927,0.0093991,0.0095266,0.0096525,0.0097767,0.0098991,0.01002,0.010138,0.010255,0.010369,0.010481,0.010592,0.0107,0.010805,0.010909,0.011011,0.01111,0.011207,0.011302,0.011395,0.011485,0.011574,0.01166,0.011745,0.011827,0.011907,0.011986,0.012062,0.012137,0.01221,0.012281,0.012351,0.012418,0.012485,0.012549,0.012613,0.012674,0.012735,0.012794,0.012852,0.012908,0.012964,0.013018,0.013072,0.013124,0.013175,0.013226,0.013275,0.013324,0.013372,0.013419,0.013466,0.013512,0.013558,0.013602,0.013647,0.013691,0.013735,0.013778,0.013821,0.013864,0.013906,0.013948,0.013991,0.014033,0.014075,0.014117,0.014159,0.014201,0.014243,0.014286,0.014328,0.01437,0.014413,0.014455,0.014498,0.014542,0.014586,0.01463,0.014675,0.014721,0.014767,0.014814,0.014862,0.01491,0.01496,0.01501,0.015062,0.015114,0.015168,0.015223,0.015279,0.015337,0.015396,0.015456,0.015518,0.015582,0.015647,0.015714,0.015784,0.015855,0.015928,0.016004,0.016082,0.016163,0.016246,0.016332,0.016422,0.016515,0.01661,0.016709,0.016812,0.016918,0.017028,0.017142,0.017259,0.017381,0.017508,0.017639,0.017775,0.017915,0.018061,0.018212,0.018369,0.018531,0.0187,0.018874,0.019055,0.019242,0.019436,0.019635,0.019841,0.020054,0.020275,0.020503,0.020739,0.020983,0.021236,0.021497,0.021766,0.022045,0.022333,0.022631,0.022938,0.023255,0.023583,0.02392,0.024268,0.024627,0.024997,0.025378,0.02577,0.026174,0.026584,0.027005,0.027439,0.027885,0.028343,0.028813,0.029297,0.029793,0.030315,0.030852,0.031403,0.031968,0.032548,0.033144,0.033677,0.034223,0.034781,0.03535,0.035933,0.036528,0.037189,0.037866,0.038559,0.039268,0.039993,0.040736,0.041469,0.042219,0.042986,0.043769,0.04457,0.045389,0.046297,0.047227,0.048179,0.049155,0.050073,0.051011,0.05197,0.052951,0.053875,0.054818,0.055782,0.056766,0.057771,0.058797,0.05978,0.060784,0.061807,0.062852,0.064013,0.0652,0.066414,0.067656,0.06912,0.070623,0.072166,0.073752,0.075324,0.076938,0.078595,0.08002,0.081476,0.082966,0.084524,0.086118,0.087751,0.089271,0.090823,0.09241,0.093992,0.095607,0.097257,0.099349,0.1015,0.1034,0.10534,0.10715,0.109,0.11088,0.1128,0.115,0.11726,0.11928,0.12134,0.12376,0.12623,0.12915,0.13214,0.13435,0.13661,0.13921,0.14187,0.14404,0.14624,0.14878,0.15137,0.15394,0.15657,0.15885,0.16117,0.16417,0.1684,0.17131,0.17428,0.17677,0.1793,0.1824,0.18613,0.18984,0.19429,0.19846,0.20137,0.20433,0.20794,0.21266,0.21795,0.22263,0.22758,0.23145,0.23536,0.23946,0.24506,0.25021,0.2552,0.25997,0.2643,0.26837,0.273,0.27795,0.28189,0.28649,0.29059,0.29556,0.29944,0.30411,0.30844,0.312,0.31591,0.32038,0.32471,0.32859,0.3325,0.33827,0.34275,0.34664,0.35043,0.35539,0.35988,0.36465,0.37023,0.37427,0.37987,0.38337,0.38757,0.39205,0.39632,0.40136,0.40595,0.41178,0.41671,0.42028,0.42439,0.43058,0.43515,0.44116,0.44601,0.44986,0.45364,0.45947,0.46457,0.4689,0.47344,0.47814,0.48166,0.4866,0.49065,0.49582,0.50128,0.50607,0.5116,0.5163,0.5201,0.52371,0.52868,0.53339,0.53815,0.54215,0.54688,0.55286,0.55701,0.56181,0.56714,0.57226,0.57765,0.58285,0.58696,0.59038,0.59544,0.60004,0.60581,0.61038,0.61544,0.62045,0.62514,0.6298,0.63298,0.63812,0.6429,0.64754,0.65298,0.65767,0.66265,0.66697,0.67208,0.67627,0.68126,0.6853,0.68926,0.694,0.6983,0.70262,0.70666,0.71078,0.71536,0.7201,0.72388,0.7298,0.73485,0.73995,0.7445,0.7486,0.75458,0.75802,0.76249,0.76759,0.77337,0.7776,0.78382,0.79051,0.79477,0.79995,0.80612,0.81024,0.81471,0.8199,0.82562,0.83023,0.83552,0.84066,0.8451,0.84929,0.85354,0.85848,0.86454,0.86875,0.87536,0.87957,0.88352,0.88846,0.89279,0.89629,0.90034,0.90488,0.90954,0.9138,0.91853,0.92242,0.92659,0.93054,0.93513,0.93875,0.94267,0.94757,0.95347,0.95862,0.96295,0.96743,0.97251,0.97828,0.98269,0.98645,0.99049,0.99366,0.99607,0.99785,0.99912};
    for (int i = 0; i < 764; i++) {
        scores.push_back(arrayScore[i]);
        SURDs.push_back(arraySurd[i]);
    }
}
