#' D-FPSSM and SF-PSSM feature vectors
#' @description This function produces list of two feature vectors named D-FPSSM and S-FPSSM
#'which then used by FPSSM2 function to construct feature vector of length 100 for each pair of
#'proteins which then used for protein-protein interaction prediction in each dataset.
#' @param pssm_name name of PSSM Matrix file
#' @import utils
#' @return two feature vectors of different length which is used in later steps.
#' @references
#' Zahiri, J., et al. (2013) PPIevo: protein-protein interaction prediction from PSSM based
#' evolutionary information, Genomics, 102, 237-242.
#' @export
#' @examples
#' q<-FPSSM(system.file("extdata","C7GQS7.txt.pssm",package="PSSMCOOL"))
#'
FPSSM <- function(pssm_name){
  x<-read.delim(pssm_name,skip = 2,sep = "",header = FALSE)
  x<-x[-1,-c(1,23:44)]
  d<-which(x=="Lambda")
  if(length(d)!=0){
    x<-x[-c(d:dim(x)[1]),]
  }
  colnames(x)<-NULL
  rownames(x)<-NULL
  x<-as.matrix(x)
  k2<-x[,1]
  k2<-as.character(k2)
  p<-x[,-1]
  mode(p)<-"integer"
  p[is.na(p)]<-0
  s<-matrix(0,20,20)
  p[p<0]<-0
  D<-apply(p,2,sum)
  names(D)<-NULL
  D<-round(D,digits =4)
  v<-c("A","R","N","D","C","Q","E","G","H","I","L","K","M","F","P","S","T","W","Y","V")
  for(i in 1:20){
    for(j in 1:20){
      s[i,j]<-sum(p[,j][k2==v[i]])
    }
  }
  S<-c()
  for(i in 1:20){
    S<-c(S,s[i,])
  }
  S<-round(S,digits =3)
  return(list(D,S))
}
#######################################################################################################
#' Mixture of Two FPSSM Features
#' @description This function takes two PSSM files as argument and uses FPSSM function for making feature
#'vector of length 100 correspond to this pair of proteins.
#' @param pssm_name1 The name of first PSSM Matrix file
#' @param pssm_name2 The name of second PSSM Matrix file
#' @importFrom stats var
#' @importFrom infotheo mutinformation
#' @importFrom utils read.delim
#' @importFrom entropy entropy
#' @return Feature vector of length 100
#' @references
#' Zahiri, J., et al. (2013) PPIevo: protein-protein interaction prediction from PSSM based
#' evolutionary information, Genomics, 102, 237-242.
#' @seealso \code{\link[entropy]{entropy}}
#'
#' \code{\link[infotheo]{mutinformation}}
#' @export
#' @examples
#'  s1<-system.file("extdata","C7GQS7.txt.pssm",package="PSSMCOOL")
#'  s2<-system.file("extdata","C7GRQ3.txt.pssm",package="PSSMCOOL")
#'  s<-FPSSM2(s1,s2)
FPSSM2<-function(pssm_name1,pssm_name2){
  requireNamespace("infotheo",quietly = TRUE)
  requireNamespace("entropy",quietly = TRUE)
  f100<-c()
  L1<-FPSSM(pssm_name1)
  S1<-L1[[2]]
  D1<-L1[[1]]
  L2<-FPSSM(pssm_name2)
  S2<-L2[[2]]
  D2<-L2[[1]]
  S3<-abs(S1-S2)
  v1=v2=v3<-rep(0,20)
  for(i in 1:20){
    v1[i]<-min(S3[((i-1)*20+1):(i*20)])
    v2[i]<-var(S3[((i-1)*20+1):(i*20)])
    v3[i]<-infotheo::mutinformation(S1[((i-1)*20+1):(i*20)],S2[((i-1)*20+1):(i*20)])
    v3[i]<-v3[i]/sqrt(entropy::entropy(S1[((i-1)*20+1):(i*20)])*entropy::entropy(S2[((i-1)*20+1):(i*20)]))
    f100<-c(f100,v1[i],v2[i],v3[i])
  }
  f100<-c(f100,D1,D2)
  f100<-round(f100,digits =3)
  return(f100)
}
