#' Preferred Bean Varieties in Nicaragua
#'
#' This is a subset of data from trials of bean varieties in Nicaragua over five
#' growing seasons. Farmers were asked to try three varieties of bean from a
#' total of ten varieties and to rank them in order of preference. In addition,
#' for each variety the farmers were asked to compare each trial variety to the
#' local variety and state whether they considered it to be better or worse.
#'
#' There are three crop seasons in Central America:
#' \describe{
#'     \item{Primera}{May - August.}
#'     \item{Postrera}{September - October.}
#'     \item{Apante}{November - January.}
#' }
#' Beans can be planted near the beginning of each season, though are most
#' commonly planted in the Postrera or Apante seasons.
#'
#' @format A data frame with 842 records and 11 variables:
#' \describe{
#'     \item{\code{variety_a}}{The name of variety A in the comparison.}
#'     \item{\code{variety_b}}{The name of variety B in the comparison.}
#'     \item{\code{variety_c}}{The name of variety C in the comparison.}
#'     \item{\code{best}}{The variety the farmer ranked in first place ("A",
#'     "B" or "C").}
#'     \item{\code{worst}}{The variety the farmer ranked in last place ("A",
#'     "B" or "C").}
#'     \item{\code{var_a}}{How the farmer ranked variety A compared to the local
#'     variety ("Worse" or "Better").}
#'     \item{\code{var_b}}{How the farmer ranked variety B compared to the local
#'     variety ("Worse" or "Better").}
#'     \item{\code{var_c}}{How the farmer ranked variety C compared to the local
#'     variety ("Worse" or "Better").}
#'     \item{\code{season}}{A factor specifying the growing season ("Po - 15",
#'     "Ap - 15", "Pr - 16", "Po - 16", "Ap - 16".}
#'     \item{\code{year}}{The year of planting.}
#'     \item{\code{maxTN}}{The maximum temperature at night during the
#'     vegetative cycle (degrees Celsius).}
#' }
#' @source The data were provided by Bioversity International, a CGIAR research
#' centre \url{https://www.bioversityinternational.org}.
#' @examples
#'
#' # Consider the best and worst rankings. These give the variety the
#' # farmer thought was best or worst, coded as A, B or C for the
#' # first, second or third variety assigned to the farmer
#' # respectively.
#' data(beans)
#' head(beans[c("best", "worst")], 2)
#'
#' # Fill in the missing item
#' beans$middle <- complete(beans[c("best", "worst")],
#'                          items = c("A", "B", "C"))
#' head(beans[c("best", "middle", "worst")], 2)
#'
#' # This gives an ordering of the three varieties the farmer was
#' # given. The names of these varieties are stored in separate
#' # columns
#' varieties <- beans[c("variety_a", "variety_b", "variety_c")]
#' head(varieties, 2)
#'
#' # Use these names to decode the orderings of order 3
#' order3 <- decode(beans[c("best", "middle", "worst")],
#'                  items = beans[c("variety_a", "variety_b", "variety_c")],
#'                  code = c("A", "B", "C"))
#'
#' # Now consider the paired comparisons agains the local variety
#' head(beans[c("var_a", "var_b", "var_c")], 2)
#'
#' # Convert these results to a vector and get the corresponding trial variety
#' outcome <- unlist(beans[c("var_a", "var_b", "var_c")])
#' trial_variety <- unlist(beans[c("variety_a", "variety_b", "variety_c")])
#'
#' # Create a data frame of the implied orderings of order 2
#' order2 <- data.frame(Winner = ifelse(outcome == "Worse",
#'                                      "Local", trial_variety),
#'                      Loser = ifelse(outcome == "Worse",
#'                                     trial_variety, "Local"),
#'                      stringsAsFactors = FALSE, row.names = NULL)
#' head(order2, 2)
#'
#' # Finally combine the rankings of order 2 and order 3
#' R <- rbind(as.rankings(order3, input = "orderings"),
#'            as.rankings(order2, input = "orderings"))
#' head(R)
#' tail(R)
"beans"
