% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/subgroupAnalysis.R
\name{subgroupAnalysis}
\alias{subgroupAnalysis}
\title{Subgroup Analysis - Interactions and estimates}
\usage{
subgroupAnalysis(object,data,treatment,
   subgroups, confint.method="default",factor.reference="extraline")
}
\arguments{
\item{object}{- glm, coxph or cph object for which subgroups should be
analyzed.}

\item{data}{- Dataset including all relevant variables}

\item{treatment}{- Must be a factor}

\item{subgroups}{- A vector of variable names presenting the factor variables
where subgroups should be formed. These variables should
all be "factors"}

\item{confint.method}{"default" creates Wald type confidence interval, "robust",
creates creates robust standard errors - see regressionTable function.}

\item{factor.reference}{"extraline" creates an extraline for the reference,
"inline" avoids this line.}
}
\value{
A data.frame with subsgroup specifications, number in each subgroup,
parameter estimates and p-value for interaction.  A forest plot
can be obtained with "plotConfidence".
}
\description{
The function can examine Cox regression, logistic regression 
and Poisson regression (Poisson regression for survival analysis)
where the effect of one variable is of particular interest. This function
systematically checks for effect modification with a list of other variables.

In randomised studies the main regression analysis is often univariate and
includes only the exposure of interest. In
observational studies the main regression analysis can readily be adjusted for
other variables including those which may modify the effect of the variable 
of interest.
}
\details{
The function can only handle a bivariate treatment, most conviniently coded as
zero or one. The p-value for interaction is obtained with a likelihood ratio test
comparing the main regression analysis with the interaction model. 

There are plot and print functions available for the function
see helppages for plot.subgroupAnalysis and print.subgroupAnalysis
}
\examples{
#load libraries
library(data.table)
library(Publish)
library(survival)
data(traceR) #get dataframe traceR
data.table::setDT(traceR)
traceR[,':='(wmi2=factor(wallMotionIndex<0.9,levels=c(TRUE,FALSE), 
                labels=c("bad","good")),
             abd2=factor(abdominalCircumference<95, levels=c(TRUE,FALSE), 
                labels=c("slim","fat")))]
traceR[,sex:=as.factor(sex)] # all subgroup variables needs to be factor                
traceR[observationTime==0,observationTime:=1]
# remove missing covariate values
traceR=na.omit(traceR)
# univariate analysis of smoking in subgroups of age and sex
# Main regression analysis is a simple/univariate Cox regression
fit_cox <- coxph(Surv(observationTime,dead)~treatment,data=traceR)
sub_cox <- subgroupAnalysis(fit_cox,traceR,treatment="treatment", 
  subgroups=c("smoking","sex","wmi2","abd2"))
sub_cox

# to see how the results are obtained consider the variable: smoking
fit_cox_smoke <- coxph(Surv(observationTime,dead)~treatment*smoking,data=traceR)
# the last three rows of the following output:
publish(fit_cox_smoke)
# are included in the first 3 rows of the result of the sub group analysis:
sub_cox[1:3,]
# the p-value is obtained as:
fit_cox_smoke_add <- coxph(Surv(observationTime,dead)~treatment+smoking,data=traceR)
anova(fit_cox_smoke_add,fit_cox_smoke,test="Chisq")

# Note that a real subgroup analysis would be to subset the data
fit_cox1a <- coxph(Surv(observationTime,dead)~treatment,data=traceR[smoking=="never"])
fit_cox1b <- coxph(Surv(observationTime,dead)~treatment,data=traceR[smoking=="current"])
fit_cox1c <- coxph(Surv(observationTime,dead)~treatment,data=traceR[smoking=="prior"])


## when the main analysis is already adjusted 
fit_cox_adj <- coxph(Surv(observationTime,dead)~treatment+smoking+sex+wmi2+abd2,
                 data=traceR)
sub_cox_adj <- subgroupAnalysis(fit_cox_adj,traceR,treatment="treatment",
  subgroups=c("smoking","sex","wmi2","abd2")) # subgroups as character string
sub_cox_adj

# When both start and end are in the Surv statement:
traceR[,null:=0]
fit_cox2 <- coxph(Surv(null,observationTime,dead)~treatment+smoking+sex+wmi2+abd2,data=traceR)
summary(regressionTable(fit_cox))
sub_cox2 <- subgroupAnalysis(fit_cox2,traceR,treatment="treatment",
  subgroups=c("smoking","sex","wmi2","abd2")) 
# Analysis with Poisson - and the unrealistic assumption of constant hazard
# and adjusted for age in all subgroups
fit_p <- glm(dead~treatment+age+offset(log(observationTime)),family="poisson",
           data=traceR)
sub_pois <- subgroupAnalysis(fit_p,traceR,treatment="treatment",
  subgroups=~smoking+sex+wmi2+abd2) 
# Analysis with logistic regression - and very wrongly ignoring censoring
fit_log <- glm(dead~treatment+age,family="binomial",data=traceR)
sub_log <- subgroupAnalysis(fit_log,traceR,treatment="treatment",
   subgroups=~smoking+sex+wmi2+abd2, factor.reference="inline")
}
\seealso{
coxph, glm, plotConfidence
}
\author{
Christian Torp-Pedersen
}
