###########################################################################/**
# @RdocDefault setGenericS3
#
# @title "Creates a generic function in S3/UseMethod style"
#
# \description{
#  \emph{Note that this method is a internal method called by
#   @see "setMethodS3" and there is no reason for calling it directly!}\cr
#
#  Creates a generic function in S3/UseMethod style, i.e. setting a
#  function with name \code{name} that despatch the method \code{name} via
#  \code{UseMethod}. If there is already a function named \code{name} that
#  function is renamed to \code{name.default}.
# }
#
# @synopsis
#
# \arguments{
#   \item{name}{The name of the generic function.}
#   \item{envir}{The environment for where this method should be stored.}
#   \item{ellipsesOnly}{If @TRUE, the only arguments in the generic function
#      will be @....}
#   \item{dontWarn}{If a non-generic method with the same name is found it 
#      will be "renamed" to a default method. If that method is found in
#      a package with a name that is \emph{not} found in \code{dontWarn}
#      a warning will be produced, otherwise it will be renamed silently.}
#   \item{...}{Not used.}
# }
#
# \examples{
#   myCat.matrix <- function(..., sep=", ") {
#     cat("A matrix:\n");
#     cat(..., sep=sep);
#     cat("\n");
#   }
#
#   myCat.default <- function(..., sep=", ") {
#     cat(..., sep=sep);
#     cat("\n");
#   }
#
#   setGenericS3("myCat");
#
#   myCat(1:10);
#   mat <- matrix(1:10, ncol=5);
#   attr(mat, "class") <- "matrix";  # Has to be done as of [R] V1.4.0.
#   myCat(mat);
# }
#
# \seealso{
#   To define a method for a class see @see "setMethodS3".
#   For a thorough example of how to use this method see @see "Object".
#   For more information about \code{UseMethod()} see @see "base::methods".
# }
#
# @author
#
# \keyword{programming}
# \keyword{methods}
#
# \keyword{internal}
#*/###########################################################################
setGenericS3.default <- function(name, envir=parent.frame(), ellipsesOnly=TRUE, dontWarn=getOption("dontWarnPkgs"), enforceRCC=TRUE, ...) {
#  cat("setGenericS3(\"", name, "\", \"", get("class", envir=parent.frame()), "\", ...)\n", sep="");

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # 0. Define local constants and local functions
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # Known generic functions (that I know of)
  GENERIC.FUNCTIONS <- c("as.vector", "range");

  # 'get' is illegal, because if it is redefined in a package, library() will
  # maybe load and set the new get, which is then a generic function, and the
  # next thing it will try to get() (it uses get internally) will not be
  # retrieved, since get.default() might not be loaded at this time, but later.
  PROTECTED.NAMES <- c("get"); 

  is.primitive <- function(fdef)
    switch(typeof(fdef), special=, builtin=TRUE, FALSE)

  
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  # 1. Assert that RCC naming conventions are followed.
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  if (enforceRCC) {
    # Assert that the generic function name is a valid function name.
    firstLetter <- substring(gsub("^[.]*", "", name), 1,1);

    allowedFirst <- c("?", "$", "$<-", "[", "[<-", "[[", "[[<-");
    if (!is.element(firstLetter, allowedFirst)) {
      if (!is.element(tolower(firstLetter), letters))
        throw(RccViolationException("Method names must begin with a lower case letter (a-z): ", name));
    
      # Check first letter  
      if (firstLetter == toupper(firstLetter))
        throw(RccViolationException("Method names should start with a lower case letter: ", name));
    }
  }

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # 2. Check for forbidden names
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  if (is.element(name, R.KEYWORDS))
    throw(RccViolationException("Method names must not be same as a reserved keyword in R: ", name));

  if (is.element(name, PROTECTED.NAMES))
    throw(RccViolationException("Trying to use an unsafe generic method name (trust us, it is for a *good* reason): ", name));

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # 2. Find the environment where sys.source() loads the package, which is
  # the local variable (argument) of sys.source() named as "envir".
  # Unfortunately, the only way we can be sure which of the parent frames
  # are the sys.source() function frame is to compare its definition with
  # each of the definitions of the parent frames using sys.function().
  # Comment: sys.source() is used by library() and require() for loading
  # packages. Also note that packages that are currently loaded are not in
  # the search path, cf. search(), and there and standard exists() will not
  # find it. *Not* checking the currently loading environment would *not* 
  # be harmful, but it would produce too many warnings.
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  sys.source.def <- get("sys.source", mode="function", envir=NULL);
  loadenv <- NULL;
  for (framePos in sys.parents()[-1]) {
    if (identical(sys.source.def, sys.function(framePos))) {
      loadenv <- parent.frame(framePos);
      break;
    }
  }

  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  # 3. Check for preexisting functions with the same name
  #     i) in the environment that we are saving to ('envir'), 
  #    ii) in the currently loading environment ('loadenv'), or
  #   iii) in the environments in the search path (search()).
  # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
  envirs <- c(envir, loadenv, lapply(search(), FUN=as.environment));
  fcnDef <- NULL;
  for (env in envirs) {
    if (exists(name, mode="function", envir=env, inherits=FALSE)) {
      fcnDef <- get(name, mode="function", envir=env, inherits=FALSE);
      fcnPkg <- attr(env, "name");
      if (is.null(fcnPkg)) 
        fcnPkg <- "base" 
      else 
        fcnPkg <- gsub("^package:", "", fcnPkg);
      break;
    }
  }

  if (!is.null(fcnDef)) {
    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    # 4a. Is it already a generic function?
    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    if (is.element(name, GENERIC.FUNCTIONS)) {
      isGeneric <- TRUE;
    } else {
      bdy <- body(fcnDef);
      if (is.null(bdy)) {
        # Assume that all primitive functions are generic, which is 99% correct.
        isGeneric <- is.primitive(fcnDef);
      } else {
        src <- as.character(deparse(bdy));  # deparse is needed! / HB 2002-01-24
      	isGeneric <- any(regexpr("UseMethod", src) != -1) | 
      		     any(regexpr("standardGeneric", src) != -1);
      }
    }

    # If it is a generic function, we are done!
    if (isGeneric) {
      # TO DO: Update generic functions with '...', if missing.
      return();
    }

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    # 4b. ... or, is there already a default function with the same name?
    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    # Search for preexisting default function in the same environments as above.
    nameDefault <- paste(name, ".default", sep="");
    defaultExists <- FALSE;
    for (env in envirs) {
      if (exists(nameDefault, mode="function", envir=env, inherits=FALSE)) {
        defaultExists <- TRUE
        defaultPkg <- if (is.null(env)) "base" else attr(env, "name");
        break;
      }
    }

    if (defaultExists) {
      warning(paste("Could not create generic function. There already exists a",
              " non-generic function named ", name, "() in package ", fcnPkg, 
              " with the same name as an existing default function ", nameDefault, 
              "() in package", defaultPkg, ".", sep=""));
      return();
    }

    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    # 4c. "Rename" the function to a default function
    # - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
    assign(nameDefault, substitute(fcn, list=list(fcn=fcnDef)), envir=envir);
    if (!is.element(fcnPkg, dontWarn)) {
      warning(paste("Renamed the preexisting function ", name, " to ", 
        nameDefault, ", which was defined in environment ", fcnPkg, ".", sep=""));
    }
  }

  # Create a generic function
  src <- paste("\"", name, "\" <- function(...) UseMethod(\"", name, "\")", sep="");
  eval(parse(text=src), envir=envir);
}

setGenericS3.default("setGenericS3");  # Creates itself ;)




###########################################################################/**
# @RdocDefault isGenericS3
#
# @title "Checks if a function is a S3/UseMethod generic function"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{fcn}{A @function.}
#   \item{...}{Not used.}
# }
#
# \details{
#   A function is considered to be a generic S3/UseMethod function if its
#   body, that is the source code, contains the regular pattern 
#   \code{"UseMethod[(]"}.
# }
#
# \value{
#  Returns @TRUE if a generic S3/UseMethod function, otherwise @FALSE.
# }
#
# @author
#
# \keyword{programming}
# \keyword{methods}
#*/###########################################################################
isGenericS3.default <- function(fcn, ...) {
  body <- as.character(body(fcn));
  return(length(grep("UseMethod[(]", body)) > 0)
}

setGenericS3("isGenericS3");



###########################################################################/**
# @RdocDefault isGenericS4
#
# @title "Checks if a function is a S4 generic function"
#
# \description{
#  @get "title".
# }
#
# @synopsis
#
# \arguments{
#   \item{fcn}{A @function.}
#   \item{...}{Not used.}
# }
#
# \details{
#   A function is considered to be a generic S4 function if its
#   body, that is the source code, contains the regular pattern 
#   \code{"standardGeneric"}.
# }
#
# \value{
#  Returns @TRUE if a generic S4 function, otherwise @FALSE.
# }
#
# @author
#
# \keyword{programming}
# \keyword{methods}
#*/###########################################################################
isGenericS4.default <- function(fcn, ...) {
  body <- as.character(body(fcn));
  return(length(grep("standardGeneric", body)) > 0)
}

setGenericS3("isGenericS4");



############################################################################
# HISTORY:
# 2005-06-14
# o Now setGenericS3() allows a few methods that starts with a non-letter
#   as the first character. See code for details.
# 2005-02-15
# o Added arguments '...' in order to match any generic functions.
# 2004-10-18
# o Added Rdoc comments for isGenericS3() and isGenericS4().
# 2004-06-27
# o Added known generic function 'as.vector()'.
# 2003-07-07
# o Removed obsolete argument 'force' in Rdoc.
# 2002-11-29
# o Updated some error messages.
# o Now it is possible to create generic methods with one (or several) 
#   . (period) as a prefix of the name. Such a method should be considered
#   private in the same manner as fields with a period are private.
# 2002-11-28
# o SPELL CHECK: "...name name..." in one of setGenericS3()'s error messages.
# 2002-11-10
# o Updated setGenericS3() to assert that the environment variable 'envir' 
#   is actually the one in the frame of the sys.source() function call. This
#   is done by comparing function defintions.
# o Changed setGenericS3() to *always* create generic functions with no
#   arguments except "...".
# 2002-10-21
# o Made ellipsesOnly=TRUE by default.
# 2002-10-17
# o Removed obsolete "modifiers<-"().
# o Added also "Object" to the class attribute to make static methods to
#   work.
# 2002-10-16
# o There are times when
#     generic <- function(...) UseMethod() 
#   is not working, for example
#     fcn <- get("generic"); fcn(myObj, ...);
#   For this reason, always do method dispatching using the name explicitly;
#     generic <- function(...) UseMethod("generic") 
#
# 2002-10-15
# o Created from R.oo Object.R and ideas as described on
#    http://www.maths.lth.se/help/R/
############################################################################
