\name{RDML}
\alias{RDML}
\title{
qPCR data import function
}
\description{
Imports qPCR data from RDML v1.1 format files (Lefever et al. 2009) and 
transforms it to the appropriate format of the \code{qpcR} (Ritz et al. 
2008, Spiess et al. 2008) and \code{chiPCR} packages. RDML is the recommended file 
format element in the Minimum Information for Publication of Quantitative 
Real-Time PCR Experiments (MIQE) guidelines (Bustin et al. 2009). Data can be 
imported to the one table format (flat) or list of tables spited by PCR targets 
and samples types (see 'Details'). Tables consist of 'Cycles' in the first 
column and fluorescence data of the samples in the remaining columns for qPCR 
data. For melting data tables consist of 'Temperature' in the first column and 
fluorescence data of the samples in the remaining columns. Names of the samples 
can be formed by patterns (see 'Details'). 
}
\usage{
RDML(rdmlfile = NA, name.pattern = "\%NAME\%__\%TUBE\%", flat.table = FALSE, 
     omit.ntp = TRUE)
}
\arguments{
  \item{rdmlfile}{
    RDML file with qPCR data.
  }
  \item{name.pattern}{
    sample name pattern (see 'Details').
  }
  \item{flat.table}{
    logical. If \code{TRUE}, a result table is not spited by PCR targets and 
    samples types.
  }
  \item{omit.ntp}{
    logical. If \code{TRUE}, samples with 'ntp' type (mark for empty sample in 
    data files generated by \emph{LightCycler 96} software) are omitted.
  }
}
\details{
\bold{ATTENTION}: Although the format RDML claimed as data exchange format, 
  the specific implementation of the format at devices from different 
  manufacturers differ significantly. Currently this function is checked 
  against RDML data from devices: \emph{Bio-Rad CFX96}, 
  \emph{Roche LightCycler 96} and \emph{Applied Biosystems StepOne}.\cr

This function has been designed to import qPCR data from instruments that 
support RDML v1.1 format export. Output from this function can be easily 
used with \code{qpcR} package.\cr

There are two general variants of output for this function:
\itemize{
  \item{simplified (\code{flat.table = TRUE}).
  Represents three elements \code{list}: \code{vector} of named 
  dilutions -- 'Dilutions', \code{data.frame} with qPCR fluorescence data of 
  \bold{all} samples -- 'qPCR' and \code{data.frame} with melting fluorescence 
  data of \bold{all} samples -- 'Melt'.}
  \item{splitted (\code{flat.table = FALSE}).
  Represents three elements \code{list}: \code{vector} of named 
  dilutions -- 'Dilutions', \code{list} of \code{data.frames} which contain 
  qPCR fluorescence data of samples spited by targets and then by 
  types -- 'qPCR' and \code{list} of \code{data.frames} which contain 
  melting fluorescence data of samples spited by targets and then by 
  types -- 'Melt'}  
  }
Names of the samples can be generated by rules described in 
\code{name.pattern}. For example, if \code{name.pattern = "\%NAME\%__\%TUBE\%"}, 
sample name (inputted in the qPCR software) is "Sample_1", and position on 
the plate is "A4", then generated sample name will be: "Sample_1__A4".\cr

Possible keys in \code{name.pattern}
\itemize{
\item{\code{\%NAME\%} - name of the sample inputted in the qPCR software 
	    (ex.: "Sample 1")}
\item{\code{\%ID\%} - tube ID (ex.: "23")}
\item{\code{\%TUBE\%} - tube name (ex.: "B11")}
\item{\code{\%TARGET\%} - PCR target (ex.: "GAPDH")}
\item{\code{\%TYPE\%} - type of the sample (ex.: "unkn")}
}
}
\value{
A list of \code{data.frame} (or one \code{data.frame}), suitable for analysis 
with \code{qpcR} package. Also output list contains information about 
dilutions (quantity or concentration) if available.
}
\references{
RDML format http://www.rdml.org/ \cr
\code{qpcR} package http://cran.r-project.org/web/packages/qpcR/index.html

\code{chipPCR} package: http://cran.r-project.org/web/packages/chipPCR/index.html

Ritz, C., Spiess, A.-N., 2008. qpcR: an R package for sigmoidal model selection 
in quantitative real-time polymerase chain reaction analysis. 
\emph{Bioinformatics} 24, 1549--1551. doi:10.1093/bioinformatics/btn227

Spiess, A.-N., Feig, C., Ritz, C., 2008. Highly accurate sigmoidal fitting of 
real-time PCR data by introducing a parameter for asymmetry. \emph{BMC 
Bioinformatics} 9, 221. doi:10.1186/1471-2105-9-221

Bustin, S.A., Benes, V., Garson, J.A., Hellemans, J., Huggett, J., Kubista, M., 
Mueller, R., Nolan, T., Pfaffl, M.W., Shipley, G.L., Vandesompele, J., Wittwer, 
C.T., 2009. The MIQE guidelines: minimum information for publication of 
quantitative real-time PCR experiments. \emph{Clin. Chem.} 55, 611--622. 
doi:10.1373/clinchem.2008.112797

Lefever, S., Hellemans, J., Pattyn, F., Przybylski, D.R., Taylor, C., Geurts, 
R., Untergasser, A., Vandesompele, J., RDML consortium, 2009. RDML: structured 
language and reporting guidelines for real-time quantitative PCR data. 
\emph{Nucleic Acids Res.} 37, 2065--2069. doi:10.1093/nar/gkp056 

}
\author{
Konstantin A. Blagodatskikh <k.blag@yandex.ru>, 
Stefan Roediger <stefan.roediger@hs-lausitz.de>,
Michal Burdukiewicz <michalburdukiewicz@gmail.com>
}
\examples{
## EXAMPLE 1:
## internal dataset lc96_bACTXY.rdml (in 'data' directory)
## generated by Roche LightCycler 96. Contains qPCR data
## with four targets and two types.
## Import with default settings.
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "lc96_bACTXY.rdml", sep ="")
lc96 <- RDML(filename)

## Show targets names
names(lc96$qPCR)
## Show types of the samples for target 'FAM@bACT'
names(lc96$qPCR[["FAM@bACT"]])

## Show dilutions for dye - FAM
lc96$Dilutions$FAM
\dontrun{
COPIES <- unique(lc96$Dilutions$FAM["quant",])
## Define calibration curves (type of the samples - 'std').
## No replicates.
library(qpcR)
CAL <- modlist(lc96$qPCR[["FAM@bACT"]]$std,
               fluo = c(2, 4, 6, 8, 10))
## Define samples to predict (first two samples with the type - 'unkn').
PRED <- modlist(lc96$qPCR[["FAM@bACT"]]$unkn[1:5],
                fluo = grep("^S", names(lc96$qPCR[["FAM@bACT"]]$unkn)[1:2]))
## Conduct quantification.
calib(refcurve = CAL, predcurve = PRED, thresh = "cpD2",
      dil = COPIES)
}
\dontrun{
## EXAMPLE 2:
## internal dataset lc96_bACTXY.rdml (in 'data' directory)
## generated by Roche LightCycler 96. Contains qPCR data
## with four targets and two types.
## Import with default settings.
library(chipPCR)                        
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "lc96_bACTXY.rdml", sep ="")
lc96 <- RDML(filename)

## Compactly display the structure of the lc96 object
str(lc96)
## Fetch cycle dependent fluorescence for HEX chanel
tmp <- lc96[["qPCR"]][["Hex@X"]][["std"]]
## Fetch vector of dillutions for HEX chanel
dilution <- as.vector(lc96[["Dilutions"]][["Hex"]])

## Use plotCurves function from the chiPCR package to 
## get an overview of the amplification curves
plotCurves(tmp[, 1], tmp[, -1])
par(mfrow = c(1,1))
## Use inder function from the chiPCR package to 
## calculate the Cq (second derivative maximum, SDM)
SDMout <- sapply(2L:ncol(tmp), function(i) {
  SDM <- summary(inder(tmp[, 1], tmp[, i]), print = FALSE)[2]
})

## Use the effcalc function from the chipPCR package and 
## plot the results for the calculation of the amplification
## efficiency analysis.
plot(effcalc(dilution, SDMout), CI = TRUE)
}
\dontrun{
## EXAMPLE 3:
## internal dataset BioRad_qPCR_melt.rdml (in 'data' directory)
## generated by Bio-Rad CFX96. Contains qPCR and melting data.
## Import without splitting by targets/types and with
## custom name pattern.
PATH <- path.package("RDML")
filename <- paste(PATH, "/extdata/", "BioRad_qPCR_melt.rdml", sep ="")
cfx96 <- RDML(filename, flat.table=TRUE,
                        name.pattern = "\%TUBE\%_\%NAME\%_\%TYPE\%_\%TARGET\%")
## Use plotCurves function from the chiPCR package to 
## get an overview of the amplification curves
library(chipPCR)                        
plotCurves(cfx96$qPCR[, 1], cfx96$qPCR[, -1], type = "l")

## Show some generated names for samples.
names(cfx96$Melt[2L:5])
## Select index numbers of the columns that contain
## samples with dye 'EvaGreen' and have type 'pos'.
cols <- grep("pos_EvaGreen$", names(cfx96$Melt))
## Conduct melting curve analysis.
library(qpcR)
invisible(meltcurve(cfx96$Melt, fluos = cols,
          temps = rep(1, length(cols))))
}
}
\keyword{Bio--Rad}
\keyword{CFX96}
\keyword{file}
\keyword{IO}
\keyword{LightCycler}
\keyword{qPCR}
\keyword{RDML}
\keyword{StepOne}
