#' @include DBDriver.R
NULL

#' An S4 class to represent a SQL Server connection
#'
#' This class extends the \code{\link[RJDBC:JDBCConnection-class]{JDBCConnection}}
#' class to represent a SQL Server connection.
#'
#' @slot jc Java object representing the connection.
#' @slot identifier.quote quote character for a SQL Server identifier can be a
#' single quotation mark (\code{\'}), a left or right bracket (\code{[]}), or a
#' double quotation mark (\code{\"}). Usually inherited from
#' \code{\linkS4class{SQLServerDriver}}.
#' @export

setClass("SQLServerConnection", contains = 'JDBCConnection')

#' Connect to/disconnect from a SQL Server database.
#'
#' @param drv An objected generated by \code{\link{SQLServer}}, or an existing
#' \code{\linkS4class{SQLServerConnection}}. If a connection, the connection
#' will be cloned.
#' @param server The address of the server to connect to.
#' @param ... One or more \href{http://jtds.sourceforge.net/faq.html}{optional connection properties.}.
#' Note if you intend to set the \code{useNTLMv2} property to \code{'true'}
#' from the default API value of \code{'false'}, you will need to make a specific
#' authentication driver available to the SQL Server driver. See
#' \code{\link{RSQLServer}} for more details
#' @return a \code{linkS4Class{SQLServerConnection}} object
#' @examples
#' \dontrun{
#' dbConnect(SQLServer(), 'ServerName')
#' }
#' @export

setMethod(f = 'dbConnect', signature = "SQLServerDriver",
  definition = function (drv, server, ...)
  {
    url <- build_url(server, ...)
    properties <- .jnew('java/util/Properties')
    jc <- .jcall(drv@jdrv, "Ljava/sql/Connection;", "connect", url,
      properties)
    new("SQLServerConnection", jc = jc, identifier.quote = drv@identifier.quote)
  }
)

#' Send query to SQL Server
#'
#' This is basically a copy of RJDBC's \code{\link[RJDBC:JDBCConnection-methods]{dbSendQuery}}
#' method for JDBCConnection.
#'
#' @param conn connection object
#' @param statement SQL statement to execute
#' @param ... additional arguments to prepared statement substituted for "?"
#' @param list undocumented
#' @return a \code{\linkS4class{SQLServerResult}} object
#' @export

setMethod("dbSendQuery",
  signature(conn = "SQLServerConnection", statement = "character"),
  def = function (conn, statement, ..., list=NULL)
  {
    statement <- as.character(statement)[1L]
    if (isTRUE(as.logical(grepl("^\\{(call|\\?= *call)", statement))))
    {
      s <- .jcall(conn@jc, "Ljava/sql/CallableStatement;", "prepareCall",
        statement, check=FALSE)
      .verify.JDBC.result(s, "Unable to execute JDBC callable statement ",
        statement)
      if (length(list(...)))
        .fillStatementParameters(s, list(...))
      if (!is.null(list))
        .fillStatementParameters(s, list)
      r <- .jcall(s, "Ljava/sql/ResultSet;", "executeQuery", check=FALSE)
      .verify.JDBC.result(r, "Unable to retrieve JDBC result set for ",
        statement)
    } else if (length(list(...)) || length(list))
    {
      s <- .jcall(conn@jc, "Ljava/sql/PreparedStatement;", "prepareStatement",
        statement, check=FALSE)
      .verify.JDBC.result(s, "Unable to execute JDBC prepared statement ",
        statement)
      if (length(list(...)))
        .fillStatementParameters(s, list(...))
      if (!is.null(list))
        .fillStatementParameters(s, list)
      r <- .jcall(s, "Ljava/sql/ResultSet;", "executeQuery", check=FALSE)
      .verify.JDBC.result(r, "Unable to retrieve JDBC result set for ",
        statement)
    } else
    {
      s <- .jcall(conn@jc, "Ljava/sql/Statement;", "createStatement")
      .verify.JDBC.result(s, "Unable to create simple JDBC statement ",
        statement)
      r <- .jcall(s, "Ljava/sql/ResultSet;", "executeQuery",
        as.character(statement)[1], check=FALSE)
      .verify.JDBC.result(r, "Unable to retrieve JDBC result set for ",
        statement)
    }
    md <- .jcall(r, "Ljava/sql/ResultSetMetaData;", "getMetaData", check=FALSE)
    .verify.JDBC.result(md, "Unable to retrieve JDBC result set meta data for ",
      statement, " in dbSendQuery")
    new("SQLServerResult", jr=r, md=md, stat=s, pull=.jnull())
  }
)

#' Get connection info
#'
#' @param dbObj Object of type \code{\linkS4class{SQLServerConnection}} representing a
#' connection
#' @param ... other arguments to methods. Not used here.
#' @return a named list containing database product name, database version,
#' user, and whether the connection is read only.
#' @examples
#' \dontrun{
#' dbGetInfo(dbConnect(SQLServer(), 'DatabaseName'))
#' }
#' @export
setMethod(f = 'dbGetInfo', signature = 'SQLServerConnection',
  definition = function (dbObj, ...)
  {
    meta <- dbObj@jc$getMetaData()
    list(dbname = meta$getDatabaseProductName(),
      db.version = meta$getDatabaseMajorVersion(),
      user = meta$getUserName(),
      is.read.only = meta$isReadOnly())
  }
)

#' Checks whether Connection is closed
#'
#' @param dbObj An object inheriting from \code{\linkS4class{SQLServerConnection}}.
#' @param ... other parameters. Not used.
#' @return logical \code{TRUE} if the connection is closed and vice-versa
#' @export

setMethod(f = 'dbIsValid', signature = 'SQLServerConnection',
  definition = function (dbObj, ...)
  {
    dbObj@jc$isClosed()
  }
)

# dbDisconnect: Inherits from JDBCConnection
# dbGetQuery: Inherits from JDBCConnection
# dbGetException: Inherits from JDBCConnection
# dbListResults: Inherits from JDBCConnection
# dbListTables: Inherits from JDBCConnection
# dbReadTable: Inherits from JDBCConnection
# dbWriteTable: Inherits from JDBCConnection
# dbExistsTable: Inherits from JDBCConnection
# dbRemoveTable: Inherits from JDBCConnection
# dbListFields: Inherits from JDBCConnection
# dbCommit: Inherits from JDBCConnection
# dbRollback: Inherits from JDBCConnection
# dbCallProc: Not yet implemented

# Copied from RJDBC:
# https://github.com/s-u/RJDBC/blob/01c55dfe76e039a37ccda732d7332325222da8c8/R/class.R
.verify.JDBC.result <- function (result, ...) {
  if (is.jnull(result)) {
    x <- .jgetEx(TRUE)
    if (is.jnull(x))
      stop(...)
    else
      stop(...," (",.jcall(x, "S", "getMessage"),")")
  }
}
.fillStatementParameters <- function(s, l) {
  for (i in 1:length(l)) {
    v <- l[[i]]
    if (is.na(v)) { # map NAs to NULLs (courtesy of Axel Klenk)
      sqlType <- if (is.integer(v)) 4 else if (is.numeric(v)) 8 else 12
      .jcall(s, "V", "setNull", i, as.integer(sqlType))
    } else if (is.integer(v))
      .jcall(s, "V", "setInt", i, v[1])
    else if (is.numeric(v))
      .jcall(s, "V", "setDouble", i, as.double(v)[1])
    else
      .jcall(s, "V", "setString", i, as.character(v)[1])
  }
}
