#' @title Plot Raven reservoir/lake stage time series using dygraph
#'
#' @description
#' rvn_res_dygraph plots modeled vs observed stage plots when supplied with reservoir stage data
#' structure read using \code{\link{rvn_res_read}}
#'
#' @param resdata reservoir stage time series data structure generated by \code{\link{rvn_res_read}} routine
#' @param timezone data timezone; defaults to UTC
#' @param basins list of subbasin names from reservoir file. Each subbasin creates separate dygraph plots
#' @param figheight height of figure, in pixels
#'
#' @return a list of plot handles to dygraph plots
#'
#' @seealso \code{\link{rvn_hyd_dygraph}} to generate dygraphs for hydrograph series
#'
#' @examples
#' # read in Raven Reservoir Stages file
#' ff <- system.file("extdata","ReservoirStages.csv", package="RavenR")
#' resdata <- rvn_res_read(ff)
#'
#' # view contents for all subbasins as dyGraph
#' dyplots <- rvn_res_dygraph(resdata)
#' dyplots[[1]]
#' dyplots[[2]]
#'
#' @export rvn_res_dygraph
#' @importFrom purrr map
#' @importFrom dygraphs dygraph dyRangeSelector
rvn_res_dygraph <- function(resdata, timezone="UTC",basins="", figheight=400)
{
  if (basins==""){ # plots all reservoirs
    basins<-sub('_obs', "", colnames(resdata$res)) #replace _obs
    basins<-unique(basins) #gets rid of repeats with
    basins<-basins[-grep("precip",basins)] # no precip ts
  }

  plotfunc <- function(basin){
    stagets <- rvn_res_extract(subs = basin, res = resdata)
    stage <- merge(stagets$sim, stagets$obs) #not inflow
    # print (basin)
    return(dygraph(stage, group = 'stages', main = basin, height=figheight) %>%
             dyRangeSelector())
  }
  dyplots <- purrr::map(basins, plotfunc) # calls plotfunc for each basin in list basins
  return(dyplots)
}
