% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rvn_rvh_overwrite.R, R/rvn_rvh_write.R
\name{rvn_rvh_overwrite}
\alias{rvn_rvh_overwrite}
\alias{rvn_rvh_write}
\title{Write/Overwrite Raven rvh file}
\usage{
rvn_rvh_overwrite(filename, SBtable, HRUtable, basefile)

rvn_rvh_write(
  filename,
  SBtable = NULL,
  HRUtable = NULL,
  description = "Generated by RavenR rvn_rvh_write",
  author = NULL
)
}
\arguments{
\item{filename}{filepath of rvh file to write to (with .rvh extension)}

\item{SBtable}{Valid subbasin dataframe with required columns "SBID", "Name", "Downstream_ID", "Profile",
"ReachLength", and "Gauged". Can have additional columns.}

\item{HRUtable}{Valid HRU dataframe with required columns 'ID', 'Area', 'Elevation', 'Latitude', 'Longitude',
'SBID', 'LandUse', 'Vegetation', 'SoilProfile', 'Aquifer', 'Terrain', 'Slope', and 'Aspect'. Can have additional columns.}

\item{basefile}{original rvh file to overwrite (only used with rvn_rvh_overwrite)}

\item{description}{(optional) Description added after file header}

\item{author}{(optional) Name of author, to be printed in file header.}
}
\value{
\item{TRUE}{returns \code{TRUE} if function runs properly}
}
\description{
Given an HRU and SubBasin dataframe, writes to the specified .rvh file. In the case of rvn_rvh_overwrite,
just the :SubBasins-:EndSubBasins and :HRUs-:EndHRUs blocks are re-written, retaining all other content.
}
\details{
\code{rvn_rvh_write} is typically used to create a 'clean' .rvh file.

\code{\link{rvn_rvh_overwrite}} is usually used after reading an original .rvh file and
  processing the HRU and SubBasin tables, using (e.g., \code{\link{rvn_rvh_cleanhrus}}). This may also be used
  to preserve commands in the rvh file such as reservoir information, comments outside of the
  subbasin and HRU blocks, RedirectToFile commands, etc.

Note that if using the rvn_rvh_overwrite function and the filename and basefile arguments are the same, the
original file will be overwritten while preserving lines outside of the subbasin and HRU blocks.

If using \code{rvn_rvh_write}, the SBtable or HRUtable parameters may be omitted and provided as \code{NULL}.
In these cases, those sections will not be written in the rvh file. This may be useful in cases where one wishes to
separate the SubBasins and HRUs in different files.
}
\section{Functions}{
\itemize{
\item \code{rvn_rvh_overwrite()}: Overwrite contents of original .rvh file

}}
\examples{
## Example: write a blank rvh file
## create some blank tables
sbs_data <- rvn_rvh_blankSBdf(nSubBasins = 1)
hru_data <- rvn_rvh_blankHRUdf(nHRUs = 3)

# write to rvh file
rvn_rvh_write(file.path(tempdir(), "Blank.rvh"),
              SBtable = sbs_data,
              HRUtable = hru_data,
              description = "Example output - Blank Watershed Discretization File",
              author = "Raven Development Team")

# Example: Read in an rvh, clean its contents and write back to new file
nith <- system.file("extdata","Nith.rvh",package = "RavenR")
rvh <- rvn_rvh_read(nith)

# remove HRUs covering less than 5\% of the total area
rvh$HRUtable <- rvn_rvh_cleanhrus(rvh$HRUtable, rvh$SBtable, area_tol = 0.05)


# write the Subbasin and HRU tables to new file using rvn_rvh_write:
rvn_rvh_write(filename=file.path(tempdir(), "Nith_cleaned_write.rvh"),
             SBtable = rvh$SBtable,
             HRUtable = rvh$HRUtable)

# write to new file, while preserving all unedited information using rvn_rvh_overwrite:
rvn_rvh_overwrite(filename=file.path(tempdir(), "Nith_cleaned_overwrite.rvh"),
                  basefile=nith,
                  SBtable = rvh$SBtable,
                  HRUtable = rvh$HRUtable)

}
\seealso{
\code{\link{rvn_rvh_read}} for the function used to read in the HRU and SubBasin data
\code{\link{rvn_rvh_cleanhrus}} for the function used to process HRU dataframe\cr
For generating blank SubBasin and HRU tables, see: \code{\link{rvn_rvh_blankSBdf}} and \code{\link{rvn_rvh_blankHRUdf}}
}
\author{
James R. Craig, University of Waterloo

Leland Scantlebury
}
