utils::globalVariables(c(
  "atLeastOneWithVersionSpec", "Current", "GithubRepo", "GithubSHA1", "GithubUsername",
  "hasVers", "maxVersionSpec", "mtime", "newPackageFullName", "PackageTrimmed"
))

#' Determine package dependencies
#'
#' This will first look in local filesystem (in `.libPaths()`) and will use
#' a local package to find its dependencies. If the package does not exist locally,
#' including whether it is the correct version, then it will look in (currently)
#' `CRAN` and its archives (if the current `CRAN` version is not
#' the desired version to check). It will also look on `GitHub` if the
#' package description is of the form of a GitHub package with format
#' `account/repo@branch` or `account/repo@commit`. For this,
#' it will attempt to get package dependencies from
#' the GitHub \file{DESCRIPTION} file.
#' This is intended to replace `tools::package_dependencies` or
#' `pkgDep` in the \pkg{miniCRAN} package, but with modifications to allow
#' multiple sources to be searched in the same function call.
#'
#' @note `tools::package_dependencies` and `pkgDep` will differ under the following
#' circumstances:
#' \enumerate{
#'   \item GitHub packages are not detected using `tools::package_dependencies`;
#'   \item `tools::package_dependencies` does not detect the dependencies of base packages
#'     among themselves, *e.g.*, `methods` depends on `stats` and `graphics`.
#' }
#'
#' @inheritParams Require
#' @param which a character vector listing the types of dependencies, a subset of
#'   `c("Depends", "Imports", "LinkingTo", "Suggests", "Enhances")`.
#'   Character string `"all"` is shorthand for that vector,
#'   character string `"most"` for the same vector without `"Enhances"`.
#' @param depends Logical. Include packages listed in "Depends". Default `TRUE`.
#' @param imports Logical. Include packages listed in "Imports". Default `TRUE`.
#' @param suggests Logical. Include packages listed in "Suggests". Default `FALSE`.
#' @param linkingTo Logical. Include packages listed in "LinkingTo". Default `TRUE`.
#' @param recursive Logical. Should dependencies of dependencies be searched, recursively.
#'                  NOTE: Dependencies of suggests will not be recursive. Default `TRUE`.
#' @param keepVersionNumber Logical. If `TRUE`, then the package dependencies returned
#'   will include version number. Default is `FALSE`
#' @param libPath A path to search for installed packages. Defaults to `.libPaths()`
#' @param sort Logical. If `TRUE`, the default, then the packages will be sorted alphabetically.
#'        If `FALSE`, the packages will not have a discernible order as they will be a
#'        concatenation of the possibly recursive package dependencies.
#' @param includeBase Logical. Should R base packages be included, specifically, those in
#'   `tail(.libPath(), 1)`
#' @inheritParams Require
#'
#' @export
#' @rdname pkgDep
#'
#' @examples
#' \dontrun{
#'   pkgDep("Require")
#'   pkgDep("Require", keepVersionNumber = FALSE) # just names
#'   pkgDep("PredictiveEcology/reproducible") # GitHub
#'   pkgDep("PredictiveEcology/reproducible", recursive = TRUE) # GitHub
#'   pkgDep(c("PredictiveEcology/reproducible", "Require")) # GitHub package and local packages
#'   pkgDep(c("PredictiveEcology/reproducible", "Require", "plyr")) # GitHub, local, and CRAN packages
#' }
pkgDep <- function(packages, libPath = .libPaths(),
                   which = c("Depends", "Imports", "LinkingTo"), recursive = FALSE,
                   depends, imports, suggests, linkingTo,
                   repos = getOption("repos"),
                   keepVersionNumber = TRUE, includeBase = FALSE,
                   sort = TRUE, purge = getOption("Require.purge", FALSE),
                   verbose = getOption("Require.verbose")) {

  purge <- dealWithCache(purge)

  if (!includeBase) packages <- packages[!packages %in% .basePkgs]
  #if (any(!missing(depends), !missing(linkingTo), !missing(imports), !missing(suggests))) {
  if (!missing(depends)) {wh <- "Depends"; if (isTRUE(depends)) which <- unique(c(which, wh)) else setdiff(which, wh)}
  if (!missing(imports)) {wh <- "Imports"; if (isTRUE(imports)) which <- unique(c(which, wh)) else setdiff(which, wh)}
  if (!missing(suggests)) {wh <- "Suggests"; if (isTRUE(suggests)) which <- unique(c(which, wh)) else setdiff(which, wh)}
  if (!missing(linkingTo)) {wh <- "LinkingTo"; if (isTRUE(linkingTo)) which <- unique(c(which, wh)) else setdiff(which, wh)}

  which <- whichToDILES(which)

  # Only deal with first one of "which"... deal with second later
  whichCat <- paste(sort(which[[1]]), collapse = "_")
  if (length(packages)) {
    saveNames <- paste(packages, paste(whichCat, "recursive", recursive, sep = "_")[1], sep = "_")
    saveNames <- gsub("[[:punct:]]| ", "_", saveNames)
    names(saveNames) <- packages
    if (isTRUE(purge)) {
      whExist <- unlist(lapply(saveNames, exists, envir = .pkgEnv))
      if (any(whExist))
        suppressWarnings(rm(list = saveNames[whExist], envir = .pkgEnv))
    }
    neededFull1 <- lapply(saveNames, get0, envir = .pkgEnv)
    needGet <- unlist(lapply(neededFull1, is.null))
    Npackages <- NROW(packages[needGet])
    messageIfGTN <- Npackages > 5

    if (any(needGet)) {
      NpackagesGitHub <- sum(!is.na(extractPkgGitHub(packages[needGet])))
      NpackagesCRAN <- Npackages - NpackagesGitHub
      if (messageIfGTN)
        messageVerbose("Getting dependencies for ",
                       if (NpackagesCRAN > 0) paste0(NpackagesCRAN, " packages on CRAN"),
                       if (NpackagesGitHub > 0) paste0("; ", NpackagesGitHub, " packages on GitHub"),
                       verbose = verbose, verboseLevel = 0)
      neededFull <- pkgDepInner(packages[needGet], libPath, which[[1]], keepVersionNumber,
                                purge = purge, repos = repos, verbose = verbose)
      purge <- FALSE # whatever it was, it was done in line above
      theNulls <- unlist(lapply(neededFull, function(x) is.null(x) || length(x) == 0))
      neededFull2 <- neededFull[!theNulls]
      NpackagesRecursive <- NROW(neededFull2)
      if (messageIfGTN) {
        nchars <- rep(" ", 2 * nchar(NpackagesRecursive) + 5)
        messageVerbose("... ", NpackagesRecursive, " of these have recursive dependencies", nchars,
                       verbose = verbose, verboseLevel = 0)
      }
      if (NROW(neededFull2)) {
        if (recursive) {
          which <- tail(which, 1)[[1]] # take the last of the list of which
          counter <- 0
          neededFull2 <- lapply(neededFull2, function(needed) {
            counter <<- counter + 1
            i <- 1
            pkgsNew <- list()
            pkgsNew[[i]] <- needed
            while (length(unlist(pkgsNew[[i]])) > 0) {
              i <- i + 1
              pkgsToLookup <- trimVersionNumber(pkgsNew[[i - 1]])
              names(pkgsToLookup) <- pkgsToLookup
              pkgsNew[[i]] <- lapply(pkgsToLookup, function(needed) {
                unique(unlist(pkgDepInner(needed, libPath, which, keepVersionNumber,
                                          purge = purge, repos = repos, verbose = verbose)))
              })
              prevIndices <- 1:(i - 1)
              curPkgs <- unlist(pkgsNew[[i]])
              prevPkgs <- unlist(pkgsNew[prevIndices])
              dt <- data.table(Package = c(prevPkgs, curPkgs),
                               Current = c(rep(FALSE, length(prevPkgs)), rep(TRUE, length(curPkgs))))
              set(dt, NULL, "PackageTrimmed", extractPkgName(dt$Package))
              set(dt, NULL, "versionSpec", extractVersionNumber(dt$Package))
              set(dt, NULL, "hasVers", !is.na(dt$versionSpec))
              hasV <- dt$hasVers == TRUE
              dt[, `:=`(atLeastOneWithVersionSpec = any(hasVers),
                        Current = all(Current == TRUE)), by = "PackageTrimmed"]
              dt <- dt[!(dt$atLeastOneWithVersionSpec == TRUE & dt$hasVers == FALSE)] # remove cases where no version spec >1 case

              versionSpecNA <- is.na(dt$versionSpec)
              dt1 <- dt[versionSpecNA == FALSE]

              if (NROW(dt1)) {
                dt1 <- dt1[!duplicated(dt1$PackageTrimmed)]
                ord <- order(package_version(dt1$versionSpec), decreasing = TRUE)
                dt1 <- dt1[ord]
              }
              dt2 <- if (all(versionSpecNA)) dt else dt[versionSpecNA]
              dt <- if (NROW(dt1)) rbindlist(list(dt1, dt2), use.names = TRUE, fill = TRUE) else dt2

              dups <- duplicated(dt$PackageTrimmed)
              dt3 <- if (any(dups)) dt[!dups] else dt

              curTrue <- dt3$Current == TRUE
              dt4 <- if (any(curTrue)) if (all(curTrue)) dt3 else dt3[curTrue] else dt3[0]
              pkgsNew <- list()
              pkgsNew[[i - 1]] <- dt3[dt3$Current == FALSE]$Package
              pkgsNew[[i]] <- dt4$Package
            }
            needed <- unique(unlist(pkgsNew))
            if (messageIfGTN) {
              mess <- paste0(paddedFloatToChar(counter, padL = nchar(NpackagesRecursive), pad = " ")
                             , " of ", NpackagesRecursive)
              messageVerbose(rep("\b", nchar(mess) + 1),  mess, verbose = verbose, verboseLevel = 0)
            }
            needed
          })
        }
      }
      # Remove "R"
      neededFull2 <- append(neededFull2, neededFull[theNulls])
      neededFull2 <- lapply(neededFull2, function(needed) {
        grep(.grepR, needed, value = TRUE, invert = TRUE)
      })

      newOnes <- names(saveNames) %in% names(neededFull)
      Map(sn = saveNames[newOnes], n = names(saveNames)[newOnes], function(sn, n) {
        assign(sn, neededFull2[[n]], envir = .pkgEnv)
      })
      neededFull1 <- append(neededFull1[!needGet], neededFull2)
    }

    if (isTRUE(sort))
      neededFull1 <- lapply(neededFull1, function(x) sort(x))
    if (isFALSE(keepVersionNumber)) {
      neededFull1 <- lapply(neededFull1, trimVersionNumber)
    }
    if (!isTRUE(includeBase)) {
      neededFull1 <- lapply(neededFull1, setdiff, .basePkgs)
    }
    if (messageIfGTN)
      messageVerbose("\b Done!", verbose = verbose, verboseLevel = 0)
  } else {
    neededFull1 <- list()
  }
  neededFull1
}

#' @importFrom utils untar
#' @inheritParams Require
pkgDepInner <- function(packages, libPath, which, keepVersionNumber,
                        purge = getOption("Require.purge", FALSE),
                        repos = repos, includeBase = FALSE, verbose = getOption("Require.verbose")) {
  names(packages) <- packages
  pkgsNoVersion <- extractPkgName(packages)
  if (!isTRUE(includeBase)) {
    isBase <- pkgsNoVersion %in% .basePkgs
    packagesToCheck <- packages[!isBase]
    pkgsNoVersionToCheck <- pkgsNoVersion[!isBase]
    packagesBase <- packages[isBase]
    packagesBase <- lapply(packagesBase, function(x) character())
  } else {
    packagesToCheck <- packages
    pkgsNoVersionToCheck <- pkgsNoVersion
  }

  desc_paths <- getDescPath(packagesToCheck, libPath)
  needed <- Map(desc_path = desc_paths, pkg = packagesToCheck,
                pkgNoVersion = pkgsNoVersionToCheck,
                function(desc_path, pkg, pkgNoVersion) {
    if (!file.exists(desc_path)) {
      pkgDT <- parseGitHub(pkg, verbose = verbose)
      if ("GitHub" %in% pkgDT$repoLocation) {
        needed <- getGitHubDeps(pkg, pkgDT, which, purge)

      } else {
        if (internetExists(paste0("cannot check for package dependencies because ", pkg, " is not installed locally"),
                           verbose = verbose)) {
          needed <- unique(unname(unlist(pkgDepCRAN(pkg,
                                                    pkgsNoVersion = pkgNoVersion,
                                                    which = which,
                                                    keepVersionNumber = keepVersionNumber,
                                                    purge = purge,
                                                    repos = repos,
                                                    verbose = verbose))))

          if (is.null(needed)) { # essesntially, failed
            pkgName <- extractPkgName(pkg)
            td <- tempdir2(pkgName)
            packageTD <- file.path(td, pkgName)
            if (!dir.exists(packageTD)) {
              if (verbose %in% 1) # covers TRUE also
                message("available.packages() does not have correct information on package dependencies for ", pkgName,
                        "; downloading tar.gz")
              verNum <- extractVersionNumber(pkg)
              if (is.na(verNum)) {
                ava <- archiveVersionsAvailable(pkgName, repos = repos)
                dt <- if (is(ava, "list"))
                  rbindlist(lapply(ava, as.data.table, keep.rownames = "packageURL"))
                else
                  as.data.table(ava, keep.rownames = "packageURL")
                colNamesToSortOn <- intersect(colnames(dt), "mtime")
                if (length(colNamesToSortOn))
                  data.table::setorderv(dt, "mtime") # order it so last one is the most recent one
                packageURL <- if (NROW(dt)) tail(dt$packageURL, 1) else character()
              } else {
                pkgFilename <- paste0(pkgName, "_", verNum, ".tar.gz")
                packageURL <- file.path(pkgName, pkgFilename)
                dt <- numeric()
              }
              if (!is.null(packageURL)) {
                if (endsWith(packageURL, "tar.gz")) {
                  srcContrib <- "src/contrib"
                  url <- file.path(repos, srcContrib, "/Archive", packageURL)
                  url2 <- file.path(repos, srcContrib, basename(packageURL))
                  tf <- tempfile()
                  haveFile <- suppressWarnings(tryCatch(download.file(url, tf, quiet = TRUE), error = function(x)
                    tryCatch(download.file(url2, tf, quiet = TRUE), error = function(y) FALSE)))
                  if (file.exists(tf)) {
                    untar(tarfile = tf, exdir = td)
                    filesToDel <- dir(packageTD, recursive = TRUE, full.names = TRUE, include.dirs = TRUE)
                    filesToDel <- filesToDel[grep("^DESCRIPTION$", basename(filesToDel), invert = TRUE)]
                    unlink(filesToDel, recursive = TRUE)
                  }
                }
              }
            }
            needed <- if (dir.exists(packageTD))
              DESCRIPTIONFileDeps(file.path(packageTD, "DESCRIPTION"),
                                  which = which, keepVersionNumber = keepVersionNumber,
                                  purge = purge)
            else {
              character()
              if (verbose %in% 1) # covers TRUE also
                message(pkg, " dependencies not found on CRAN; perhaps incomplete description? On GitHub?")
            }
          }
          purge <<- FALSE
          needed
        } else {
          needed <- NULL
          needed
        }
      }
    } else {
      needed <- DESCRIPTIONFileDeps(desc_path, which = which, keepVersionNumber = keepVersionNumber)
    }
    needed
  })
  if (!isTRUE(includeBase)) {
    needed1 <- append(needed, packagesBase)
    needed <- needed1[match(names(packages), names(needed1))]
  }

  needed
}

getDescPath <- function(packages, libPath) {
  lapply(packages, function(pkg) {
    dp <- sprintf("%s/%s/DESCRIPTION", libPath, pkg) # nolint
    fe <- file.exists(dp)
    dp[fe][1] # take first file that exists
  })
}

#' @description
#' `pkgDep2` is a convenience wrapper of `pkgDep` that "goes one level in",
#' i.e., the first order dependencies, and runs the `pkgDep` on those.
#' @rdname pkgDep
#' @export
#' @param sorted Logical. If `TRUE`, the default, the packages will be sorted in
#'   the returned list from most number of dependencies to least.
#' @examples
#' \dontrun{
#'   pkgDep2("Require")
#'   # much bigger one
#'   pkgDep2("reproducible")
#' }
pkgDep2 <- function(packages, recursive = TRUE,
                    which = c("Depends", "Imports", "LinkingTo"),
                    depends, imports, suggests, linkingTo,
                    repos = getOption("repos"),
                    sorted = TRUE, purge = getOption("Require.purge", FALSE)) {
  a <- lapply(pkgDep(packages, recursive = FALSE, which = which, depends = depends,
                     imports = imports, suggests = suggests, linkingTo = linkingTo,
                     repos = repos)[[1]],
              pkgDep,
              depends = depends, imports = imports, suggests = suggests, linkingTo = linkingTo,
              repos = repos, recursive = recursive
  )
  a <- unlist(a, recursive = FALSE)
  if (sorted) {
    ord <- order(sapply(a, function(x) length(x)), decreasing = TRUE)
    a <- a[ord]
  }
  return(a)
}

#' @importFrom utils compareVersion
#' @inheritParams Require
pkgDepCRAN <- function(pkg, which = c("Depends", "Imports", "LinkingTo"),
                       #recursive = FALSE,
                       pkgsNoVersion,
                       keepVersionNumber = TRUE, repos = getOption("repos"),
                       purge = getOption("Require.purge", FALSE),
                       verbose = getOption("Require.verbose")) {
  capFull <- available.packagesCached(repos = repos, purge = purge, verbose = verbose)
  deps <- pkgDepCRANInner(capFull, which = which, pkgs = pkg, pkgsNoVersion = pkgsNoVersion,
                          keepVersionNumber = keepVersionNumber, verbose = verbose)
  deps
}


#' Reverse package depends
#'
#' This is a wrapper around `tools::dependsOnPkgs`,
#' but with the added option of `sorted`, which
#' will sort them such that the packages at the top will have
#' the least number of dependencies that are in `pkgs`.
#' This is essentially a topological sort, but it is done
#' heuristically. This can be used to e.g., `detach` or
#' `unloadNamespace` packages in order so that they each
#' of their dependencies are detached or unloaded first.
#' @param pkgs A vector of package names to evaluate their
#'   reverse depends (i.e., the packages that *use* each
#'   of these packages)
#' @param deps An optional named list of (reverse) dependencies.
#'   If not supplied, then `tools::dependsOnPkgs(..., recursive = TRUE)`
#'   will be used
#' @param topoSort Logical. If `TRUE`, the default, then
#'   the returned list of packages will be in order with the
#'   least number of dependencies listed in `pkgs` at
#'   the top of the list.
#' @param reverse Logical. If `TRUE`, then this will use `tools::pkgDependsOn`
#'   to determine which packages depend on the `pkgs`
#' @param useAllInSearch Logical. If `TRUE`, then all non-core
#' R packages in `search()` will be appended to `pkgs`
#' to allow those to also be identified
#' @param returnFull Logical. Primarily useful when `reverse = TRUE`.
#'   If `TRUE`, then then all installed packages will be searched.
#'   If `FALSE`, the default, only packages that are currently in
#'   the `search()` path and passed in `pkgs` will be included
#'   in the possible reverse dependencies.
#'
#' @inheritParams Require
#' @export
#' @rdname pkgDep
#' @return
#' A possibly ordered, named (with packages as names) list where list elements
#' are either full reverse depends.
#'
#' @examples
#' \dontrun{
#' pkgDepTopoSort(c("Require", "data.table"), reverse = TRUE)
#' }
pkgDepTopoSort <- function(pkgs, deps, reverse = FALSE, topoSort = TRUE,
                           useAllInSearch = FALSE,
                           returnFull = TRUE, recursive = TRUE,
                           purge = getOption("Require.purge", FALSE),
                           verbose = getOption("Require.verbose")) {

  if (isTRUE(useAllInSearch)) {
    if (missing(deps)) {
      a <- search()
      a <- setdiff(a, .defaultPackages)
      a <- gsub("package:", "", a)
      pkgs <- unique(c(pkgs, a))
    } else {
      if (verbose %in% 1) # covers TRUE also
        message("deps is provided; useAllInSearch will be set to FALSE")
    }
  }

  names(pkgs) <- pkgs
  if (missing(deps)) {
    aa <- if (isTRUE(reverse)) {
      ip <- .installed.pkgs()
      deps <- lapply(ip[,"Dependencies"], extractPkgName)
      names(deps) <- ip[, "Package"]
      names(pkgs) <- pkgs
      deps <- deps[order(names(deps))]
      revDeps <- lapply(pkgs, function(p) names(unlist(lapply(deps, function(d) if (isTRUE(any(p %in% d))) TRUE else NULL))))
      if (recursive) {
        revDeps <- lapply(revDeps, function(p) {
          if (!is.null(p)) {
            used <- p
            repeat({
              r <- unique(unlist(lapply(p, function(p1)  names(unlist(lapply(deps, function(d) if (isTRUE(any(p1 %in% d))) TRUE else NULL))))))
              used <- unique(c(r, used))
              if (length(r) == 0)
                break
              p <- r
            })
          } else {
            used <- NULL
          }
          sort(used)
        })

      }
    } else {
      pkgDep(pkgs, recursive = TRUE, purge = purge)
    }
  }
  else
    aa <- deps
  bb <- list()

  aa <- checkCircular(aa)
  cc <- lapply(aa, function(x) character())
  dd <- lapply(cc, function(x) 0)


  if (length(aa) > 1) {
    lengths <- lengths(aa)
    aa <- aa[order(lengths)]
    cc <- cc[order(lengths)]
    dd <- lapply(cc, function(x) 0)

    ddIndex <- 0
    priorsBeingInstalled <- priorsAlreadyInstalled <- character()

    if (isTRUE(topoSort)) {
      notInOrder <- TRUE
      isCorrectOrder <- logical(length(aa))
      i <- 1
      newOrd <- numeric(0)
      for (i in seq_along(aa)) {
        dif <- setdiff(seq_along(aa), newOrd)
        pkgNameNames <- extractPkgName(names(aa))
        for (j in dif) {
          pkgName <- extractPkgName(aa[[j]])
          overlapFull <- pkgName %in% pkgNameNames[-i]
          overlap <- pkgName %in% pkgNameNames[dif]
          overlapPkgs <- pkgName[overlapFull]
          isCorrectOrder <- !any(overlap)
          if (isCorrectOrder) {
            cc[j] <- list(overlapPkgs)
            priorsBeingInstalled <- vapply(dd, function(x) if (is.numeric(x)) x == ddIndex else FALSE, logical(1))
            priorsBeingInstalled <- extractPkgName(names(priorsBeingInstalled)[priorsBeingInstalled])
            overlapPkgsAdditional <- intersect(overlapPkgs, priorsBeingInstalled)
            if (length(overlapPkgsAdditional)) {
              ddIndex <- ddIndex + 1
              priorsAlreadyInstalled <- vapply(dd, function(x) if (is.numeric(x)) x < ddIndex else FALSE, logical(1))
              priorsAlreadyInstalled <- extractPkgName(names(priorsAlreadyInstalled)[priorsAlreadyInstalled])
            }
            dd[j] <- list(ddIndex)

            newOrd <- c(newOrd, j)
            # i <- i + 1
            break
          }
        }
      }
      aa <- aa[newOrd]
      cc <- cc[newOrd]
      dd <- dd[newOrd]
    }
  }

  out <- if (isTRUE(returnFull)) {
    aa
  } else {
    cc
  }
  attr(out, "installSafeGroups") <- dd

  return(out)
}

.defaultPackages <- c(".GlobalEnv", "tools:rstudio", "package:stats", "package:graphics",
                      "package:grDevices", "package:utils", "package:datasets", "package:methods",
                      "Autoloads", "package:base", "devtools_shims")


#' @inheritParams Require
pkgDepCRANInner <- function(ap, which, pkgs, pkgsNoVersion, keepVersionNumber,
                            keepSeparate = FALSE, verbose = getOption("Require.verbose")) {
  # MUCH faster to use base "ap$Package %in% pkgs" than data.table internal "Package %in% pkgs"
  if (missing(pkgsNoVersion))
    pkgsNoVersion <- trimVersionNumber(pkgs)
  if (isFALSE(keepVersionNumber)) {
    pkgs <- pkgsNoVersion
  }
  ap <- ap[ap$Package %in% pkgsNoVersion]
  keep <- match(ap$Package, pkgsNoVersion)
  keep <- keep[!is.na(keep)]
  pkgsNoVersion1 <- pkgsNoVersion[keep]
  pkgs <- pkgs[keep]
  ap <- ap[order(pkgsNoVersion1)]

  names(which) <- which
  deps <- lapply (which, function(i) {
    lapply(ap[[i]], function(x) {
      out <- strsplit(x, split = "(, {0,1})|(,\n)")[[1]]
      out <- out[!is.na(out)]
      out <- grep(.grepR, out, value = TRUE, invert = TRUE) # remove references to R
    })
  })

  ss <- seq_along(pkgsNoVersion1)
  names(ss) <- pkgsNoVersion1
  if (isFALSE(keepSeparate))
    deps <- lapply(ss, function(x) unname(unlist(lapply(deps, function(y) y[[x]]))))
  deps
}

DESCRIPTIONFileDeps <- function(desc_path, which = c("Depends", "Imports", "LinkingTo"),
                                keepVersionNumber = TRUE,
                                purge = getOption("Require.purge", FALSE),
                                keepSeparate = FALSE) {
  if (!exists("pkgDep", envir = .pkgEnv))
    .pkgEnv[["pkgDep"]] <- newPkgDepEnv()
  if (!exists("DESCRIPTIONFile", envir = .pkgEnv[["pkgDep"]]))
    .pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]] <- newPkgDepEnv()
  objName <- if (length(desc_path) == 1)
    paste0(desc_path, paste0(collapse = "_", which, "_", keepVersionNumber))
  else {
    grepPackage <- "Package *: *"
    grepVersion <- "Version *: *"
    suppressWarnings(pkg <- gsub(grepPackage, "", grep(grepPackage, desc_path, value = TRUE)))
    suppressWarnings(vers <- gsub(grepVersion, "", grep(grepVersion, desc_path, value = TRUE)))
    paste(pkg, vers, sep = "_", paste0(which, "_", keepVersionNumber, collapse = "_"))
  }
  if (isTRUE(length(objName) > 1) ||
      isTRUE(any(!exists(objName, envir = .pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]]))) ||
      isTRUE(purge)) {
    if (is.null(desc_path)) {
      needed <- NULL
    } else {

      lines <- if (length(desc_path) == 1)
        try(readLines(desc_path))
      else
        lines <- desc_path
      if (is(lines, "try-error")) stop("Cannot read the file: ", desc_path)
      Sys.setlocale(locale = "C") # required to deal with non English characters in Author names
      on.exit(Sys.setlocale(locale = ""))
      sl <- list()
      sl[["depends"]] <- grep("^Depends: *", lines) # nolint
      sl[["suggests"]] <- grep("^Suggests: *", lines) # nolint
      sl[["imports"]] <- grep("^Imports: *", lines) # nolint
      sl[["linkingto"]] <- grep("^LinkingTo: *", lines) # nolint
      sl[["remotes"]] <- grep("^Remotes: *", lines) # nolint
      sl[["colon"]] <- grep(": *", lines) # nolint

      which <- paste0(toupper(substr(which, 1, 1)), substr(which, 2, 1e4))
      which <- unique(which)
      whichLower <- tolower(which)
      needed <- Map(whLower = whichLower, wh = which, function(whLower, wh) {
        if (length(sl[[whLower]])) {
          whEnd <- which(sl[["colon"]] %in% sl[[whLower]]) + 1
          whEnd <- if (length(sl[["colon"]]) < whEnd) {
            length(lines)
          } else {
            (sl[["colon"]][whEnd] - 1)
          }
          allLines <- sl[[whLower]]:whEnd # nolint
          needs <- paste(lines[allLines], collapse = "")
          needs <- gsub(paste0(wh, ": *"), "", needs)
          needs <- strsplit(needs, split = ", *")
          needs <- gsub(" *$", "", needs[[1]]) # remove trailing spaces
          if (isFALSE(keepVersionNumber))
            needs <- trimVersionNumber(needs)
          needs
        }
      })
      if (length(objName) == 1)
        assign(objName, needed, envir = .pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]])
    }
  } else {
    needed <- get(objName, envir = .pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]])
  }
  if (!keepSeparate) {
    needed <- unname(unlist(needed))
    needed <- grep("^R[\\( ]", needed, value = TRUE, invert = TRUE)
  }
  needed
}

DESCRIPTIONFile <- function(desc_path) {
  lines <- readLines(desc_path)
}

whichToDILES <- function(which) {
  if (identical("all", which) || is.null(which)) {
    which <- list(c("Depends", "Imports", "LinkingTo", "Suggests", "Enhances"))
  } else if (identical("most", which)) {
    which <- list(c("Depends", "Imports", "LinkingTo", "Suggests"))
  } else if (isTRUE(which)) {
    which <- list(c("Depends", "Imports", "LinkingTo", "Suggests"),
                  c("Depends", "Imports", "LinkingTo"))
  } else if (any(is.na(which))) {
    which <- list(c("Depends", "Imports", "LinkingTo"))
  } else if (is.character(which)) {
    which <- list(which)
  } else if (isFALSE(which)) {
    which <- character()
  }
  which
}

.installed.pkgs <- function(lib.loc = .libPaths(), which = c("Depends", "Imports", "LinkingTo"), other = NULL,
                            purge = getOption("Require.purge", FALSE)) {
  purge <- dealWithCache(purge)
  if (!is.null(other))
    if (any(grepl("github", tolower(other)))) {
      other <- c("GithubRepo", "GithubUsername", "GithubRef", "GithubSHA1")
    }

  out <- lapply(lib.loc, function(path) {
    dirs <- dir(path, full.names = TRUE)
    areDirs <- dir.exists(dirs)
    dirs <- dirs[areDirs]
    files <- file.path(dirs, "DESCRIPTION")
    filesExist <- file.exists(files)
    files <- files[filesExist]
    names(files) <- basename(dirs[filesExist])
    desc_lines <- lapply(files, function(file) DESCRIPTIONFile(file))
    versions <- DESCRIPTIONFileVersionV(desc_lines, purge = FALSE)
    deps <- if (length(which)) lapply(desc_lines, function(lines)
      DESCRIPTIONFileDeps(lines, which = which, purge = purge)) else NULL
    if (!is.null(other)) {
      names(other) <- other
      others <- lapply(other, function(oth) DESCRIPTIONFileOtherV(desc_lines, other = oth))
      # shas <- DESCRIPTIONFileOtherV(desc_lines)
    }
    mat <- cbind("Package" = dirs[filesExist], "Version" = versions, "Depends" = deps)
    if (!is.null(other)) {
      mat <- cbind(mat, as.data.frame(others, stringsAsFactors = FALSE), stringsAsFactors = FALSE)
    }
    mat
  })
  lengths <- unlist(lapply(out, function(x) NROW(x)))
  out <- do.call(rbind, out)
  ret <- cbind("Package" = basename(unlist(out[, "Package"])), "LibPath" = rep(lib.loc, lengths),
        "Version" = out[, "Version"])
  if (length(which))
    ret <- cbind(ret, "Dependencies" = out[, "Depends"], stringsAsFactors = FALSE)
  if (!is.null(other)) {
    ncolBefore <- NCOL(ret)
    ret <- cbind(ret, out[, other], stringsAsFactors = FALSE)
  }
  ret
}

.basePkgs <- # unlist(.installed.pkgs(tail(.libPaths(), 1))[, "Package"])
  c("base", "boot", "class", "cluster", "codetools", "compiler",
    "datasets", "foreign", "graphics", "grDevices", "grid", "KernSmooth",
    "lattice", "MASS", "Matrix",
    "methods", "mgcv", "nlme", "nnet",
    "parallel", "rpart", "spatial", "splines", "stats", "stats4",
    "survival", "tcltk", "tools", "translations", "utils")

DESCRIPTIONFileDepsV <- Vectorize(DESCRIPTIONFileDeps, vectorize.args = "desc_path",
                                  SIMPLIFY = FALSE)

#' Package dependencies when one or more packages removed
#'
#' This is primarily for package developers. It allows the testing
#' of what the recursive dependencies would be if a package was removed
#' from the immediate dependencies.
#' @param pkg A package name to be testing the dependencies
#' @param depsRemoved A vector of package names who are to be "removed"
#'   from the `pkg` immediate dependencies
#' @export
#' @return
#' A list with 3 named lists `Direct`, `Recursive` and `IfRemoved`.
#' `Direct` will show the top level direct dependencies, either `Remaining` or `Removed`.
#' `Recursive` will show the full recursive dependencies, either `Remaining` or `Removed`.
#' `IfRemoved` returns all package dependencies that are removed for each top level dependency.
#' If a top level dependency is not listed in this final list, then it means that it is also a
#' recursive dependency elsewhere, so its removal has no effect.
#' @inheritParams Require
#' @examples
#' \dontrun{
#' pkgDepIfDepRemoved("Require", "remotes")
#' }
pkgDepIfDepRemoved <- function(pkg = character(), depsRemoved = character(),
                               verbose = getOption()) {
  if (length(pkg)) {
    p2 <- pkgDep2(pkg, recursive = TRUE)
    names(p2) <- extractPkgName(names(p2))
    p1 <- names(p2) # just the immediate
    p3 <- p2
    p2All <- unique(extractPkgName(c(names(p2), unname(unlist(p2)))))
    p3[depsRemoved] <- NULL
    p3All <- unique(extractPkgName(c(names(p3), unname(unlist(p3)))))
    removed <- sort(setdiff(p2All, p3All))
    left <- sort(p3All)

    removedDirect <- intersect(p1, depsRemoved)
    allInRecursive <- unique(extractPkgName(unname(unlist(p2))))
    direct <- p1
    bestToRemove <- setdiff(direct, allInRecursive)
    bestToRemove <- p2[bestToRemove]
    bestToRemove <- Map(btr = bestToRemove, top = names(bestToRemove),
                        function(btr, top) {
                          p4 <- p2
                          p4[top] <- NULL
                          setdiff(extractPkgName(unique(c(names(p4), unname(unlist(p4))))),
                                  extractPkgName(c(top, btr)))
                          })
    leftDirect <- names(p3)
    pkg <- list(Direct = list(Remaining = leftDirect, Removed = removedDirect,
                              NRemaining = length(leftDirect), NRemoved = length(removedDirect)),
                Recursive = list("Remaining" = left, "Removed" = removed,
                                 NRemaining = length(left), NRemoved = length(removed)),
                IfRemoved = bestToRemove)
  }
  return(pkg)
}

checkCircular <- function(aa) {
  circular <- lapply(names(aa), function(outer)  {
    circularWh <- extractPkgName(aa[[outer]]) %in% extractPkgName(outer)
    circ <- if (any(circularWh)) {
      aa[[outer]][circularWh]
    } else {
      NULL
    }
  })

  if (length(unlist(circular))) {
    for (i in which(!unlist(lapply(circular, is.null)))) {
      aa[[i]] <- setdiff(aa[[i]], circular[[i]])
    }
  }
  aa
}

#' @inheritParams Require
getGitHubDeps <- function(pkg, pkgDT, which, purge, verbose = getOption("Require.verbose")) {
  pkg <- masterMainToHead(pkg)
  pkgDT <- getGitHubDESCRIPTION(pkgDT, purge = purge)
  needed <- DESCRIPTIONFileDeps(pkgDT$DESCFile, which = which, purge = purge)
  neededRemotes <- DESCRIPTIONFileDeps(pkgDT$DESCFile, which = "Remotes", purge = purge)
  neededRemotesName <- extractPkgName(neededRemotes)
  neededName <- extractPkgName(needed)
  needSomeRemotes <- neededName %in% neededRemotesName
  if (any(needSomeRemotes)) {
    hasVersionNum <- grep(grepExtractPkgs, needed[needSomeRemotes])
    if (length(hasVersionNum)) {
      neededInRemotesWVersion <- needed[needSomeRemotes][hasVersionNum]
      vn <- extractVersionNumber(neededInRemotesWVersion)
      ineq <- extractInequality(neededInRemotesWVersion)
      neededPkgsInRemotes <- extractPkgName(neededInRemotesWVersion)
      inequWVN <- paste0(" (", ineq, " ", vn, ")")
      remotes <- neededRemotes[neededRemotesName %in% neededName]
      whNeedVN <- match(neededPkgsInRemotes, extractPkgName(remotes))
      remotesWVN <- remotes[whNeedVN]
      remotesWVN <- paste0(remotesWVN, inequWVN)
      remotesWoVN <- if (length(neededRemotes) != length(whNeedVN)) {
        remotes[-whNeedVN]
      } else {
        NULL
      }
      remotesAll <- c(remotesWVN, remotesWoVN)
    } else {
      remotesAll <- neededRemotes[neededRemotesName %in% neededName]
    }
    needed <- c(needed[!needSomeRemotes], remotesAll)
  }

  # Check NAMESPACE too -- because imperfect DESCRIPTION files
  namespaceFile <- getGitHubNamespace(pkgDT$packageFullName)$NAMESPACE
  if (is.null(namespaceFile)) {
    needed <- NULL
  } else {
    rr <- readLines(namespaceFile)
    depsFromNamespace <- gsub(", except.*(\\))$", "\\1", rr)
    depsFromNamespace <- unique(gsub("^import.*\\((.+)\\,.*$", "\\1",
                                     grep("^import", depsFromNamespace, value = TRUE)))
    depsFromNamespace <- unique(gsub("^import\\((.+)\\)", "\\1", depsFromNamespace))
    depsFromNamespace <- gsub(",.*", "", depsFromNamespace)
    depsFromNamespace <- gsub("\\\"", "", depsFromNamespace)
    pkgDT2 <- data.table(packageFullName = setdiff(union(needed, depsFromNamespace), .basePkgs))
    if (NROW(pkgDT2)) {
      pkgDT2[, isGitPkg := grepl("^.+/(.+)@+.*$", packageFullName)]
      setorderv(pkgDT2, "isGitPkg", order = -1)
      pkgDT2[, Package := extractPkgName(packageFullName)]

      # Here, GitHub package specification in a DESCRIPTION file Remotes section
      #   won't have version numbering --> Need to merge the two fields
      pkgDT2[, Version := extractVersionNumber(packageFullName)]
      if (any(!is.na(pkgDT2$Version))) {
        pkgDT2[!is.na(Version), inequality := extractInequality(packageFullName)]
        pkgDT2[, Version := {
          if (all(is.na(Version))) NA_character_ else as.character(max(as.package_version(Version[!is.na(Version)])))
        }, by = "Package"]
        pkgDT2[, inequality := {
          if (all(is.na(inequality))) NA_character_ else inequality[!is.na(inequality)][[1]]
        }, by = "Package"]
        pkgDT2[, github := extractPkgGitHub(packageFullName)]
        if (any(pkgDT2$isGitPkg == TRUE & !is.na(pkgDT2$Version))) {
          pkgDT2[isGitPkg == TRUE & !is.na(Version), newPackageFullName :=
                   ifelse(is.na(extractVersionNumber(packageFullName)),
                          paste0(packageFullName, " (", inequality, Version, ")"), NA) ]
          whGitNeedVersion <- !is.na(pkgDT2$newPackageFullName)
          if (any(whGitNeedVersion)) {
            pkgDT2[whGitNeedVersion == TRUE, packageFullName := newPackageFullName]
          }
        }
      }
      dup <- duplicated(pkgDT2, by = c("Package", "Version"))
      pkgDT2 <- pkgDT2[dup == FALSE]
      differences <- setdiff(pkgDT2$Package, extractPkgName(needed))
      if (length(differences)) {
        if (verbose %in% 1) # covers TRUE also
          message(" (-- The DESCRIPTION file for ", pkg, " is incomplete; there are missing imports:\n",
                  paste(differences, collapse = ", "), " --) ")
      }
    }
    needed <- pkgDT2$packageFullName
  }
  needed
}


dealWithCache <- function(purge, checkAge = TRUE) {
  if (!isTRUE(purge) && isTRUE(checkAge)) {
    purgeDiff <- as.numeric(Sys.getenv("R_AVAILABLE_PACKAGES_CACHE_CONTROL_MAX_AGE"))
    if (is.null(.pkgEnv[["startTime"]])) {
      purge = TRUE
    } else {
      purgeDiff <- if (identical(purgeDiff, "")  || is.na(purgeDiff)) 3600 else purgeDiff
      autoPurge <- purgeDiff < as.numeric(difftime(Sys.time(), .pkgEnv[["startTime"]], units = "sec"))
      purge <- purge || autoPurge
    }
  }

  if (isTRUE(purge) || is.null(.pkgEnv[["pkgDep"]])) {
    .pkgEnv[["pkgDep"]] <- newPkgDepEnv()
    .pkgEnv[["startTime"]] <- Sys.time()
  }

  if (is.null(.pkgEnv[["pkgDep"]][["deps"]]) || purge) .pkgEnv[["pkgDep"]][["deps"]] <- new.env(parent = emptyenv())
  if (is.null(.pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]]) || purge)
    .pkgEnv[["pkgDep"]][["DESCRIPTIONFile"]] <- new.env(parent = emptyenv())

  purge
}

.grepTooManySpaces <- " {2,}"
.grepTabCR <- "\n|\t"

newPkgDepEnv <- function() new.env(parent = emptyenv())

masterMainToHead <- function(gitRepo) {
  masterOrMain <- "@(master|main) *"
  areMasterOrMain <- grepl(masterOrMain, gitRepo)
  if (any(areMasterOrMain)) {
    masterOrMainNoSpace <- "@(master|main)"
    gitRepo[areMasterOrMain] <-
      gsub(paste0("(",masterOrMainNoSpace,")"), "@HEAD", gitRepo[areMasterOrMain])
  }
  return(gitRepo)
}

################################################################################
#' Convert numeric to character with padding
#'
#' This will pad floating point numbers, right or left. For integers, either class
#' integer or functionally integer (e.g., 1.0), it will not pad right of the decimal.
#' For more specific control or to get exact padding right and left of decimal,
#' try the \code{stringi} package. It will also not do any rounding. See examples.
#'
#' @param x numeric. Number to be converted to character with padding
#'
#' @param padL numeric. Desired number of digits on left side of decimal.
#'              If not enough, \code{pad} will be used to pad.
#'
#' @param padR numeric. Desired number of digits on right side of decimal.
#'              If not enough, \code{pad} will be used to pad.
#'
#' @param pad character to use as padding (\code{nchar(pad) == 1} must be \code{TRUE}).
#'   Currently, can be only `"0"` or `" "` (i.e., space).
#'
#' @return Character string representing the filename.
#'
#' @author Eliot McIntire and Alex Chubaty
#' @export
#' @rdname paddedFloatToChar
#'
#' @examples
#' paddedFloatToChar(1.25)
#' paddedFloatToChar(1.25, padL = 3, padR = 5)
#' paddedFloatToChar(1.25, padL = 3, padR = 1) # no rounding, so keeps 2 right of decimal
paddedFloatToChar <- function (x, padL = ceiling(log10(x + 1)), padR = 3, pad = "0") {
  if (!pad %in% c("0", " ")) {
    stop("pad must be either '0' or ' '")
  }
  xf <- x %% 1
  numDecimals <- nchar(gsub("(.*)(\\.)|([0]*$)","",xf))

  # this == used to be fpCompare -- but this function is more or less deprecated
  newPadR <- ifelse(abs(xf - 0) < sqrt(.Machine$double.eps), 0, pmax(numDecimals, padR))
  string <- paste0(paste0("%", pad), padL+newPadR+1*(newPadR > 0),".", newPadR, "f")
  xFCEnd <- sprintf(string, x)
  return(xFCEnd)
}


