% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AutoHierarchies.R
\name{AutoHierarchies}
\alias{AutoHierarchies}
\title{Ensure standardized coding of hierarchies}
\usage{
AutoHierarchies(
  hierarchies,
  data = NULL,
  total = "Total",
  hierarchyVarNames = c(mapsFrom = "mapsFrom", mapsTo = "mapsTo", sign = "sign", level =
    "level"),
  combineHierarchies = TRUE,
  unionComplement = FALSE,
  ...
)
}
\arguments{
\item{hierarchies}{List of hierarchies}

\item{data}{Matrix or data frame with data containing codes of relevant variables}

\item{total}{Within \code{AutoHierarchies}: Vector of total codes (possibly recycled) used when running \code{\link{Hrc2DimList}} or \code{\link{FindDimLists}}.}

\item{hierarchyVarNames}{Variable names in the hierarchy tables as in \code{\link{HierarchyFix}}}

\item{combineHierarchies}{Whether to combine several hierarchies for same variable into a single hierarchy (see examples).}

\item{unionComplement}{Logical vector as in \code{\link{Hierarchies2ModelMatrix}}. The parameter is only in use when hierarchies are combined.}

\item{...}{Extra unused parameters}
}
\value{
List of hierarchies
}
\description{
Automatic convert list of hierarchies coded in different ways to standardized to-from coding
}
\details{
Input can be to-from coded hierarchies, hierarchies/dimList as in sdcTable, TauArgus coded hierarchies or formulas.
Automatic coding from data is also supported. Output is on a from ready for input to \code{\link{HierarchyCompute}}.
A single string as hierarchy input is assumed to be a total code.
Then, the hierarchy is created as a simple hierarchy where all codes in data sum up to this total.
For consistence with \code{HierarchyCompute},
the codes \code{"rowFactor"} and \code{"colFactor"} are unchanged.
An empty string is recoded to  \code{"rowFactor"}.

A special possibility is to include character vector(s) as unnamed list element(s) of \code{hierarchies}.
Then the elements of the character vector(s) must be variable names within data.
This will cause hierarchies to be created from selected data columns by running \code{\link{FindDimLists}}.
Total coded can be specified by parameter \code{total} or by naming the character vector. See examples.
}
\examples{
# First, create different types of input
z <- SSBtoolsData("sprt_emp_withEU")
yearFormula <- c("y_14 = 2014", "y_15_16 = y_all - y_14", "y_all = 2014 + 2015 + 2016")
yearHier <- Formula2Hierarchy(yearFormula)
geoDimList <- FindDimLists(z[, c("geo", "eu")], total = "Europe")[[1]]
geoDimList2 <- FindDimLists(z[, c("geo", "eu")])[[1]]
geoHrc <- DimList2Hrc(geoDimList)
ageHier <- SSBtoolsData("sprt_emp_ageHier")

h1 <- AutoHierarchies(list(age = ageHier, geo = geoDimList, year = yearFormula))
h2 <- AutoHierarchies(list(age = "Y15-64", geo = geoHrc, year = yearHier), data = z, 
                      total = "Europe")
h3 <- AutoHierarchies(list(age = "Total", geo = geoDimList2, year = "Total"), data = z)
h4 <- FindHierarchies(z[, c(1, 2, 3, 5)])
h5 <- AutoHierarchies(list(age = "Total", geo = "", year = "colFactor"), data = z)
identical(h1, h2)
identical(h3, h4)

# Print the resulting hierarchies
h1 # = h2
h3 # = h4
h5

FindHierarchies(z[, c("geo", "eu", "age")])


# ===================================================================== 
#   Examples illustrating the combineHierarchies parameter
# =====================================================================

# First, create data
d <- SSBtoolsData("d2ws")[1:3]
d$isCounty1 <- "NO"
d$isCounty1[d$county == "county-1"] <- "YES"
d

# sdcTable coding showing two tree-shaped hierarchies
dimList <- FindDimLists(d)
dimList

# Two tree-shaped hierarchies can still be seen 
# Hierarchies with three and two levels
hA <- AutoHierarchies(dimList, combineHierarchies = FALSE)
hA

# A single hierarchy with only one level 
# Contains the information needed to create a dummy matrix
hB <- AutoHierarchies(dimList)
hB

# Dummy matrices from the hierarchies
DummyHierarchies(hA)
DummyHierarchies(hB)


# ===================================================================== 
#   Special examples with character vector(s) as unnamed list elements
# =====================================================================

# Same output as FindHierarchies above
AutoHierarchies(list(c("geo", "eu", "age")), data = z)

# Now combined with a named list element 
AutoHierarchies(list(year = yearHier, c("geo", "eu", "age")), data = z)

# Total codes by unnamed list element as named character vector 
AutoHierarchies(list(year = yearHier, c(Europe = "geo", "eu", All = "age")), data = z)

# Two types of year input. Total codes by using the parameter `total`. 
AutoHierarchies(list("year", year = yearHier, c("geo", "eu", "age")), data = z, 
                total = c("allYears", "unused", "Tot"))

# Avoid combineHierarchies to see effect of each year input separately 
# (even earlier return possible with `combineHierarchies = NA`)
AutoHierarchies(list("year", year = yearHier, c("geo", "eu", "age")), data = z, 
                total = c("allYears", "unused", "Tot"), combineHierarchies = FALSE)

}
\seealso{
\code{\link{FindHierarchies}}, \code{\link{DimList2Hierarchy}}, \code{\link{DimList2Hrc}},
\code{\link{Hierarchy2Formula}}, \code{\link{DummyHierarchies}}.
}
\author{
Øyvind Langsrud
}
