% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/updist_edges.R
\name{updist_edges}
\alias{updist_edges}
\title{Get upstream distance for edges in a Landscape Network (LSN)}
\usage{
updist_edges(
  edges,
  lsn_path = NULL,
  calc_length = FALSE,
  length_col = NULL,
  save_local = TRUE,
  overwrite = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{edges}{An \code{sf} object with LINESTRING geometry created
using \code{\link{lines_to_lsn}}.}

\item{lsn_path}{Local pathname to a directory in character format
specifying where relationships.csv resides, which is created
using \code{\link[SSNbler]{lines_to_lsn}}.}

\item{calc_length}{A logical indicating whether a column
representing line length should be calculated and added to
\code{edges}. Default = \code{FALSE}.}

\item{length_col}{Optional. If \code{calc_length = FALSE},
\code{length_col} is the name of the column in the \code{edges}
attribute table that contains the length of the edge
feature. When \code{calc_length = FALSE}, \code{length_col} is
required. If \code{calc_length = TRUE}, \code{length_col} is the
name of the new column created in \code{edges} that will store
the new length values for each feature, in character format.
When \code{calc_length = TRUE} and \code{length_col = NULL}, the default for \code{length_col}
is \code{"Length"}.}

\item{save_local}{Logical indicating whether the updated
\code{edges} should be saved to \code{lsn_path} in GeoPackage
format. Defaults to \code{TRUE}.}

\item{overwrite}{A logical indicating whether results should be
overwritten if the upDist column already exists in \code{edges}
or edges.gpkg already exists in \code{lsn_path} and
\code{save_local = TRUE}. Default = TRUE}

\item{verbose}{Logical. Indicates whether messages about the
function progress should be printed to the console. Defaults to
\code{TRUE}.}
}
\value{
An \code{sf} object representing edges in the LSN, with a new \code{upDist} column. When \code{calc_length = TRUE} an additional column named \code{length_col}
}
\description{
Calculate the distance from the stream outlet
(i.e. the most downstream location on the stream network) to the
upstream node of each edge feature (i.e. upstream distance) in
the Landscape Network (LSN)
}
\details{
\code{updist_edges()} calculates the total hydrologic distance from
the uppermost location on each edge feature (upstream node) to
the stream outlet (i.e. the most downstream location in the
stream network), when movement is restricted to the stream
network. We refer to this as the upstream distance. Upstream distances are
measured in the map projection units for the \code{sf} object edges and stored in
a new column in edges named \code{upDist}.

The upstream distances stored in \code{upDist} are used to calculate the upstream distance for
sites in \code{\link[=updist_sites]{updist_sites()}} and the pairwise hydrologic distances used to fit spatial stream network models in
the 'SSN2' package. Do not modify the name of the column in any way or the values the \code{upDist}
column contains.
}
\examples{
# Get temporary directory, where the example LSN will be stored
# locally.
temp_dir <- tempdir()
# Build the LSN. When working with your own data, lsn_path will be
# a local folder of your choice rather than a temporary directory.
edges<- lines_to_lsn(
   streams = MF_streams,
   lsn_path = temp_dir,
   snap_tolerance = 1,
   check_topology = FALSE,
   overwrite = TRUE,
   verbose = FALSE
)

# Calculate upstream distance for edges
edges<- updist_edges(
   edges = edges,
   lsn_path = temp_dir,
   calc_length = TRUE,
   length_col = "Length",
   overwrite = TRUE,
   save_local = FALSE,
   verbose = FALSE
)

# Notice that two new columns have been added to edges containing
# line feature length (Length) and the upstream distance (upDist)
names(edges)
summary(edges[,c("Length", "upDist")])

}
