% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SVEMnet.R
\name{SVEMnet}
\alias{SVEMnet}
\title{Fit an SVEMnet Model}
\usage{
SVEMnet(
  formula,
  data,
  nBoot = 300,
  glmnet_alpha = c(1),
  weight_scheme = c("SVEM", "FWR", "Identity"),
  objective = c("wAIC", "wBIC", "wGIC", "wSSE"),
  gamma = 2,
  standardize = TRUE,
  ...
)
}
\arguments{
\item{formula}{A formula specifying the model to be fitted.}

\item{data}{A data frame containing the variables in the model.}

\item{nBoot}{Number of bootstrap iterations (default is 300).}

\item{glmnet_alpha}{Elastic Net mixing parameter(s) (default is \code{c(1)}).
Can be a vector of alpha values, where \code{alpha = 1} is Lasso and \code{alpha = 0} is Ridge.}

\item{weight_scheme}{Weighting scheme for SVEM (default "SVEM").
One of \code{"SVEM"}, \code{"FWR"}, or \code{"Identity"}.}

\item{objective}{Objective used to pick \code{lambda} on each bootstrap path (default "wAIC").
One of \code{"wAIC"}, \code{"wBIC"}, \code{"wGIC"}, or \code{"wSSE"}.}

\item{gamma}{Penalty weight used only when \code{objective="wGIC"} (numeric, default \code{2}).
Setting \code{gamma = 2} reproduces wAIC.}

\item{standardize}{Logical; passed to \code{glmnet} (default \code{TRUE}).}

\item{...}{Additional args to \code{glmnet()}.}
}
\value{
An object of class \code{svem_model}.
}
\description{
Wrapper for 'glmnet' (Friedman et al. 2010) to fit an ensemble of Elastic Net
models using the Self-Validated Ensemble Model method (SVEM, Lemkus et al. 2021).
Allows searching over multiple alpha values in the Elastic Net penalty.
}
\details{
The Self-Validated Ensemble Model (SVEM, Lemkus et al., 2021) framework provides a bootstrap
approach to improve predictions from base learners, including Elastic Net regression
as implemented in \code{glmnet}. In each of the \code{nBoot} iterations, SVEMnet applies
random exponentially distributed weights to the observations; anti-correlated weights are
used for validation when \code{weight_scheme="SVEM"}.

SVEMnet allows \code{glmnet_alpha} to be a vector, enabling a search over multiple Elastic Net
mixing parameters within each bootstrap. The \code{objective} controls how the validation
criterion balances fit and complexity:

\describe{
  \item{\code{"wSSE"}}{Weighted Sum of Squared Errors: uses the weighted validation SSE directly.}
  \item{\code{"wAIC"}}{Weighted AIC (Gaussian): \code{AIC = n * log(SSE_w / n) + 2 * k},
        where \code{n = sum(w_valid)} (after normalization) and \code{k} counts parameters
        including the intercept. Candidates require \code{k < n}.}
  \item{\code{"wBIC"}}{Weighted BIC-like criterion: \code{n * log(SSE_w / n) + log(n_eff) * k},
        with \code{n_eff = (sum(w_valid)^2) / sum(w_valid^2)} (Kish). For stability, \code{n_eff}
        is clipped to \code{[5, n]}. Candidates require \code{k < n} and \code{k < n_eff - 1}.}
  \item{\code{"wGIC"}}{Weighted Generalized Information Criterion:
        \code{n * log(SSE_w / n) + gamma * k}. Here \code{gamma} is a fixed nonnegative number.
        For robustness near the boundary, candidates require \code{k < n} and \code{k < n_eff - 1}.}
}

\strong{Note on BIC:} In reweighted validation, information content varies with weight
heterogeneity; using \code{log(n_eff)} adapts the penalty to that effective size. With uniform
weights (Identity), \code{n_eff = n} and you recover standard BIC.

A debiased fit is output (along with the standard fit). This is provided to allow the user to
match the output of JMP, which returns a debiased fit whenever \code{nBoot >= 10}.
The debiasing coefficients are always calculated by \code{SVEMnet()}, and the
\code{predict()} method determines whether the raw or debiased predictions are returned via its
\code{debias} argument. The default is \code{debias = FALSE}, based on performance on unpublished simulations.

The returned object includes averaged coefficients (\code{parms}), debiased coefficients
(\code{parms_debiased}), the calibration fit (\code{debias_fit}), per-bootstrap coefficients,
chosen alphas and lambdas, the chosen \code{objective} (and \code{gamma} if applicable), and a
compact \code{diagnostics} list (median/IQR of selected model size and alpha frequencies).
}
\section{Acknowledgments}{

Development of this package was assisted by GPT o1-preview, which helped in constructing the
structure of some of the code and the roxygen documentation. The code for the significance test
is taken from the supplementary material of Karl (2024) (it was handwritten by that author).
}

\examples{
# Simulate data
set.seed(0)
n <- 21
X1 <- runif(n)
X2 <- runif(n)
X3 <- runif(n)
y <- 1 + 2*X1 + 3*X2 + X1*X2 + X1^2  + rnorm(n)
data <- data.frame(y, X1, X2, X3)

# Fit the SVEMnet model with a formula
model <- SVEMnet(
  y ~ (X1 + X2 + X3)^2 + I(X1^2) + I(X2^2) + I(X3^2),
  glmnet_alpha = c(1),
  data = data
)
coef(model)
plot(model)
predict(model, data)

# Example: BIC-like penalty
# model_bic <- SVEMnet(y ~ X1 + X2 + X3, data = data, objective = "wBIC")
# Example: GIC with custom gamma
# model_gic <- SVEMnet(y ~ X1 + X2 + X3, data = data, objective = "wGIC", gamma = 4)

}
\references{
Gotwalt, C., & Ramsey, P. (2018). Model Validation Strategies for Designed Experiments Using
Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference}.
\url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849873/redirect_from_archived_page/true}

Karl, A. T. (2024). A randomized permutation whole-model test heuristic for Self-Validated
Ensemble Models (SVEM). \emph{Chemometrics and Intelligent Laboratory Systems}, \emph{249}, 105122.
\doi{10.1016/j.chemolab.2024.105122}

Karl, A., Wisnowski, J., & Rushing, H. (2022). JMP Pro 17 Remedies for Practical Struggles with
Mixture Experiments. JMP Discovery Conference. \doi{10.13140/RG.2.2.34598.40003/1}

Lemkus, T., Gotwalt, C., Ramsey, P., & Weese, M. L. (2021). Self-Validated Ensemble Models for
Design of Experiments. \emph{Chemometrics and Intelligent Laboratory Systems}, 219, 104439.
\doi{10.1016/j.chemolab.2021.104439}

Xu, L., Gotwalt, C., Hong, Y., King, C. B., & Meeker, W. Q. (2020). Applications of the
Fractional-Random-Weight Bootstrap. \emph{The American Statistician}, 74(4), 345–358.
\doi{10.1080/00031305.2020.1731599}

Ramsey, P., Gaudard, M., & Levin, W. (2021). Accelerating Innovation with Space Filling Mixture
Designs, Neural Networks and SVEM. \emph{JMP Discovery Conference}.
\url{ https://community.jmp.com/t5/Abstracts/Accelerating-Innovation-with-Space-Filling-Mixture-Designs/ev-p/756841}

Ramsey, P., & Gotwalt, C. (2018). Model Validation Strategies for Designed Experiments Using
Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference - Europe}.
\url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849647/redirect_from_archived_page/true}

Ramsey, P., Levin, W., Lemkus, T., & Gotwalt, C. (2021). SVEM: A Paradigm Shift in Design and
Analysis of Experiments. \emph{JMP Discovery Conference - Europe}.
\url{https://community.jmp.com/t5/Abstracts/SVEM-A-Paradigm-Shift-in-Design-and-Analysis-of-Experiments-2021/ev-p/756634}

Ramsey, P., & McNeill, P. (2023). CMC, SVEM, Neural Networks, DOE, and Complexity:
It’s All About Prediction. \emph{JMP Discovery Conference}.

Friedman, J. H., Hastie, T., & Tibshirani, R. (2010). Regularization Paths for Generalized
Linear Models via Coordinate Descent. \emph{Journal of Statistical Software}, 33(1), 1–22.
}
