#' @title Plotting SynergyLMM results
#' @description
#' Generic function to generate different types of plots based on SynergyLMM outputs.
#' @param object An object generated by [`lmmModel()`] or [`lmmSynergy()`] functions.
#' @param plot_type String indicating the type of plot to generate. Possible options include:
#' - "lmmModel" for plotting of tumor growth data from a fitted model.
#' - "lmmSynergy" for plotting synergy results.
#' - "ObsvsPred" for generating plots of Observed vs Predicted Values.
#' - "ranefDiagnostics" for plots of random effects diagnostics.
#' - "residDiagnostics" for plots of residual diagnostics.
#' @param ... Additional arguments passed to the specific plot function.
#' @returns Different output plots are produced depending on the `plot_type` selected.
#' @seealso [plot_lmmModel()], [plot_lmmSynergy()], [plot_ObsvsPred()], [plot_ranefDiagnostics()], [plot_residDiagnostics()].
#' @examples
#' data(grwth_data)
#' # Fit the model
#' lmm <- lmmModel(
#'   data = grwth_data,
#'   sample_id = "subject",
#'   time = "Time",
#'   treatment = "Treatment",
#'   tumor_vol = "TumorVolume",
#'   trt_control = "Control",
#'   drug_a = "DrugA",
#'   drug_b = "DrugB",
#'   combination = "Combination",
#'   show_plot = FALSE
#'   )
#'   
#' # Plot lmmModel
#' plot_SynergyLMM(lmm, plot_type = "lmmModel",
#' trt_control = "Control",
#' drug_a = "DrugA",
#' drug_b = "DrugB",
#' combination = "Combination"
#' )
#' 
#' # Plot ObsvsPred
#' plot_SynergyLMM(lmm, plot_type = "ObsvsPred")
#' 
#' # Plot ranefDiagnostics
#' plot_SynergyLMM(lmm, plot_type = "ranefDiagnostics")
#' 
#' # Plot residDiagnostics
#' plot_SynergyLMM(lmm, plot_type = "residDiagnostics")
#' 
#' # Plot lmmSynergy
#' lmmSyn <- lmmSynergy(lmm)
#' plot_SynergyLMM(lmmSyn, plot_type = "lmmSynergy")
#' @export



plot_SynergyLMM <- function(object, plot_type = "lmmModel", ...) {
  # Check correspondance between input object and plot type
  
  if (is.null(attr(object, "SynergyLMM"))) {
    stop(
      "Incorrect input object. Input object needs to be obtained by 'lmmModel' or 'lmmSynergy' functions."
    )
  }
  
  if (plot_type %in% c("lmmModel",
                       "ObsvsPred",
                       "ranefDiagnostics",
                       "residDiagnostics")) {
    if (attr(object, "SynergyLMM") != "lmmModel") {
      stop(
        paste(
          "The input object needs to be a model generated by 'lmmModel' function if 'plot_type = ",
          plot_type,
          "' is selected.",
          sep = ""
        )
      )
    }
  }
  
  if (plot_type == "lmmSynergy") {
    if (attr(object, "SynergyLMM") != "lmmSynergy") {
      stop(
        paste(
          "The input object needs to be a model generated by 'lmmSynergy' function if 'plot_type = ",
          plot_type,
          "' is selected.",
          sep = ""
        )
      )
    }
  }
  
  # Automatically route to the appropriate plotting function
  switch(
    plot_type,
    "lmmModel" = plot_lmmModel(object, ...),
    "lmmSynergy" = plot_lmmSynergy(object, ...),
    "ObsvsPred" = plot_ObsvsPred(object, ...),
    "ranefDiagnostics" = plot_ranefDiagnostics(object, ...),
    "residDiagnostics" = plot_residDiagnostics(object, ...),
    stop("The 'plot_type' value is not a valid type of plot.")
  )
}

